<#
    Windows Maintenance Tool - GUi Edition
    CLI: Lil_Batti (author) with contributions from Chaython
    Feature Integration & Updates: Lil_Batti & Chaython
    GUI thanks to https://github.com/Chaython
    Imported and integrated from Lil_Batti (author) with contributions from Chaython
#>

# ==========================================
# 1. SETUP
# ==========================================
$AppVersion = "5.0.3"
$ErrorActionPreference = "SilentlyContinue"
[Console]::OutputEncoding = [System.Text.UTF8Encoding]::new($false)
$OutputEncoding = [System.Text.UTF8Encoding]::new($false)

# HIDE CONSOLE (Safe Check)
# This prevents crashes if you run the script twice in the same session
if (-not ([System.Management.Automation.PSTypeName]'Win32Functions.Win32ShowWindow').Type) {
    $t = '[DllImport("user32.dll")] public static extern bool ShowWindow(IntPtr handle, int state);'
    Add-Type -MemberDefinition $t -Name "Win32ShowWindow" -Namespace Win32Functions
}
try {
    $hwnd = ([System.Diagnostics.Process]::GetCurrentProcess() | Get-Process).MainWindowHandle
    if ($hwnd -ne [IntPtr]::Zero) {
        [Win32Functions.Win32ShowWindow]::ShowWindow($hwnd, 0) | Out-Null
    }
} catch {}

# ADMIN CHECK
$identity = [System.Security.Principal.WindowsIdentity]::GetCurrent()
$principal = New-Object System.Security.Principal.WindowsPrincipal($identity)
if (-not $principal.IsInRole([System.Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Start-Process powershell.exe "-File `"$PSCommandPath`"" -Verb RunAs
    exit
}

# ENABLE HIGH-DPI AWARENESS (Safe Check)
if ([Environment]::OSVersion.Version.Major -ge 6) {
    if (-not ([System.Management.Automation.PSTypeName]'Win32Dpi').Type) {
        $code = @'
        [DllImport("user32.dll")]
        public static extern bool SetProcessDPIAware();
'@
        Add-Type -MemberDefinition $code -Name "Win32Dpi"
    }
    try { [Win32Dpi]::SetProcessDPIAware() | Out-Null } catch {}
}

Add-Type -AssemblyName PresentationFramework, System.Windows.Forms, System.Drawing, Microsoft.VisualBasic
[System.Windows.Forms.Application]::EnableVisualStyles()

# OPTIMIZATION: Define Token Manipulator globally once (Prevents "Type already exists" errors)
if (-not ([System.Management.Automation.PSTypeName]'Win32.TokenManipulator').Type) {
    $tokenCode = @'
    using System;
    using System.Runtime.InteropServices;
    public class Win32 {
        public class TokenManipulator {
            [DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
            internal static extern bool AdjustTokenPrivileges(IntPtr htok, bool disall, ref TokPriv1Luid newst, int len, IntPtr prev, IntPtr relen);
            [DllImport("kernel32.dll", ExactSpelling = true)]
            internal static extern IntPtr GetCurrentProcess();
            [DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
            internal static extern bool OpenProcessToken(IntPtr h, int acc, ref IntPtr phtok);
            [DllImport("advapi32.dll", SetLastError = true)]
            internal static extern bool LookupPrivilegeValue(string host, string name, ref long pluid);
            [StructLayout(LayoutKind.Sequential, Pack = 1)]
            internal struct TokPriv1Luid { public int Count; public long Luid; public int Attr; }
            internal const int SE_PRIVILEGE_ENABLED = 0x00000002;
            internal const int TOKEN_ADJUST_PRIVILEGES = 0x00000020;
            internal const int TOKEN_QUERY = 0x00000008;
            public static bool EnablePrivilege(string privilege) {
                try {
                    IntPtr htok = IntPtr.Zero;
                    if (!OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, ref htok)) return false;
                    TokPriv1Luid tp; tp.Count = 1; tp.Attr = SE_PRIVILEGE_ENABLED; tp.Luid = 0;
                    if (!LookupPrivilegeValue(null, privilege, ref tp.Luid)) return false;
                    if (!AdjustTokenPrivileges(htok, false, ref tp, 0, IntPtr.Zero, IntPtr.Zero)) return false;
                    return true;
                } catch { return false; }
            }
        }
    }
'@
    Add-Type -TypeDefinition $tokenCode
}

# ==========================================
# 2. HELPER FUNCTIONS
# ==========================================

function Get-Ctrl { param($Name) return $window.FindName($Name) }

function Write-GuiLog {
    param($Msg)
    $lb = Get-Ctrl "LogBox"
    if ($lb) {
        $lb.AppendText("[$((Get-Date).ToString('HH:mm'))] $Msg`n")
        $lb.ScrollToEnd()
    }
}

function Invoke-UiCommand {
    param(
        [scriptblock]$Sb, 
        $Msg="Processing...", 
        [object[]]$ArgumentList = @()
    )
    [System.Windows.Forms.Cursor]::Current = [System.Windows.Forms.Cursors]::WaitCursor
    Write-GuiLog $Msg
    try { 
        # Pass arguments to the scriptblock using splatting
        $res = & $Sb @ArgumentList | Out-String
        if ($res){ Write-GuiLog $res.Trim() } 
        else { Write-GuiLog "Done." }
    } catch { 
        Write-GuiLog "ERROR: $($_.Exception.Message)" 
    }
    [System.Windows.Forms.Cursor]::Current = [System.Windows.Forms.Cursors]::Default
}

# Centralized data path for exports (in repo folder)
function Get-DataPath {
    $root = Split-Path -Parent $PSCommandPath
    $dataPath = Join-Path $root "data"
    if (-not (Test-Path $dataPath)) {
        New-Item -ItemType Directory -Path $dataPath -Force | Out-Null
    }
    return $dataPath
}
$script:DataDir = Get-DataPath

# Simple modal text viewer (read-only)
function Show-TextDialog {
    param(
        [string]$Title = "Output",
        [string]$Text = ""
    )
    $f = New-Object System.Windows.Forms.Form
    $f.Text = $Title
    $f.Size = "800,600"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.Color]::FromArgb(30,30,30)
    $f.ForeColor = [System.Drawing.Color]::White

    $tb = New-Object System.Windows.Forms.RichTextBox
    $tb.Dock = "Fill"
    $tb.ReadOnly = $true
    $tb.BackColor = [System.Drawing.Color]::FromArgb(20,20,20)
    $tb.ForeColor = [System.Drawing.Color]::White
    $tb.Font = New-Object System.Drawing.Font("Consolas", 10)
    $tb.Text = $Text
    $tb.WordWrap = $false
    $f.Controls.Add($tb)

    $btn = New-Object System.Windows.Forms.Button
    $btn.Text = "Close"
    $btn.Dock = "Bottom"
    $btn.Height = 35
    $btn.BackColor = "DimGray"
    $btn.ForeColor = "White"
    $btn.Add_Click({ $f.Close() })
    $f.Controls.Add($btn)

    $f.ShowDialog() | Out-Null
}

# --- SETTINGS MANAGER ---
# Initialize cache variable
$script:WmtSettingsCache = $null

function Save-WmtSettings {
    param($Settings)
    $path = Join-Path (Get-DataPath) "settings.json"
    try {
        # Update the memory cache immediately
        $script:WmtSettingsCache = $Settings

        # Convert Hashtable/OrderedDictionary to generic Object for cleaner JSON
        $saveObj = [PSCustomObject]@{
            TempCleanup      = $Settings.TempCleanup
            RegistryScan     = $Settings.RegistryScan
            WingetIgnore     = $Settings.WingetIgnore
            LoadWinapp2      = [bool]$Settings.LoadWinapp2
            EnabledProviders = $Settings.EnabledProviders
            Theme            = if ($Settings.Theme) { [string]$Settings.Theme } else { "dark" }
            WindowState      = if ($Settings.WindowState) { [string]$Settings.WindowState } else { "Normal" }
            WindowBounds     = if ($Settings.WindowBounds) { $Settings.WindowBounds } else { $null }
        }
        $saveObj | ConvertTo-Json -Depth 5 | Set-Content $path -Force
    } catch {
        Write-Warning "Failed to save settings: $_"
    }
}

function Get-WmtSettings {
    # OPTIMIZATION: Return cached settings if available to avoid disk I/O
    if ($script:WmtSettingsCache) { return $script:WmtSettingsCache }

    $path = Join-Path (Get-DataPath) "settings.json"
    
    # Default Structure
    $defaults = @{
        TempCleanup      = @{}
        RegistryScan     = @{}
        WingetIgnore     = @()
        LoadWinapp2      = $false 
        EnabledProviders = @("winget", "msstore", "pip", "npm", "chocolatey")
        Theme            = "dark"
        WindowState      = "Normal"
        WindowBounds     = @{
            Top    = 0
            Left   = 0
            Width  = 1280
            Height = 820
        }
    }
    
    if (Test-Path $path) {
        try {
            $json = Get-Content $path -Raw | ConvertFrom-Json
            
            if ($json.TempCleanup) { 
                foreach ($p in $json.TempCleanup.PSObject.Properties) { $defaults.TempCleanup[$p.Name] = $p.Value } 
            }
            if ($json.RegistryScan) { 
                foreach ($p in $json.RegistryScan.PSObject.Properties) { $defaults.RegistryScan[$p.Name] = $p.Value } 
            }
            if ($json.PSObject.Properties["WingetIgnore"]) {
                $raw = $json.WingetIgnore
                $clean = New-Object System.Collections.ArrayList
                if ($raw) { foreach ($item in $raw) { [void]$clean.Add("$item".Trim()) } }
                $defaults.WingetIgnore = $clean.ToArray()
            }
            if ($json.PSObject.Properties["LoadWinapp2"]) { $defaults.LoadWinapp2 = [bool]$json.LoadWinapp2 }
            if ($json.PSObject.Properties["EnabledProviders"]) { $defaults.EnabledProviders = $json.EnabledProviders }
            if ($json.PSObject.Properties["Theme"] -and $json.Theme) { $defaults.Theme = [string]$json.Theme }
            if ($json.PSObject.Properties["WindowState"] -and $json.WindowState) { $defaults.WindowState = [string]$json.WindowState }
            if ($json.PSObject.Properties["WindowBounds"] -and $json.WindowBounds) {
                $defaults.WindowBounds = @{
                    Top    = [double]$json.WindowBounds.Top
                    Left   = [double]$json.WindowBounds.Left
                    Width  = [double]$json.WindowBounds.Width
                    Height = [double]$json.WindowBounds.Height
                }
            }
        } catch { 
            Write-GuiLog "Error loading settings: $($_.Exception.Message)" 
        }
    }
    
    # Cache the result
    $script:WmtSettingsCache = $defaults
    return $defaults
}

function Show-DownloadStats {
    Invoke-UiCommand {
        try {
            $repo = "ios12checker/Windows-Maintenance-Tool"
            $rel = Invoke-RestMethod -Uri "https://api.github.com/repos/$repo/releases/latest" -UseBasicParsing
            if (-not $rel -or -not $rel.assets) { throw "No release data returned." }
            $total = ($rel.assets | Measure-Object download_count -Sum).Sum
            $lines = @()
            $lines += "Release: $($rel.name)"
            $lines += "Total downloads: $total"
            $lines += ""
            foreach ($a in $rel.assets) {
                $lines += ("{0} : {1}" -f $a.name, $a.download_count)
            }
            $msg = $lines -join "`r`n"
            Write-Output $msg
            [System.Windows.MessageBox]::Show($msg, "Latest Release Downloads", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
        } catch {
            $err = "Failed to fetch download stats: $($_.Exception.Message)"
            Write-Output $err
            [System.Windows.MessageBox]::Show($err, "Latest Release Downloads", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Error) | Out-Null
        }
    } "Fetching latest release download counts..."
}

# --- UPDATE CHECKER ---
function Start-UpdateCheckBackground {
    # 1. Access LogBox
    $lbStart = Get-Ctrl "LogBox"
    if ($lbStart) { 
        $lbStart.AppendText("`n[UPDATE] Checking for updates...`n") 
        $lbStart.ScrollToEnd()
    }

    $localVersionStr = $script:AppVersion

    # 2. Start Background Thread (Runspace)
    $script:UpdateRunspace = [PowerShell]::Create().AddScript({
        param($CurrentVer)
        [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
        
        $jobRes = @{ Status = "Failed"; RemoteVersion = "0.0"; Content = ""; Error = "" }

        try {
            $time = Get-Date -Format "yyyyMMddHHmmss"
            $url  = "https://raw.githubusercontent.com/ios12checker/Windows-Maintenance-Tool/main/WMT-GUI.ps1?t=$time"
            
            # Shorter timeout for UI responsiveness
            $req = Invoke-WebRequest -Uri $url -UseBasicParsing -TimeoutSec 10
            $content = $req.Content
            $jobRes.Content = $content

            if ($content -match '\$AppVersion\s*=\s*"(\d+(\.\d+)+)"') {
                $jobRes.RemoteVersion = $matches[1]
                $jobRes.Status = "Success"
            }
            elseif ($content -match "Windows Maintenance Tool.*v(\d+(\.\d+)+)") {
                $jobRes.RemoteVersion = $matches[1]
                $jobRes.Status = "Success"
            } else {
                $jobRes.Error = "Version string not found."
            }
        } catch {
            $jobRes.Error = $_.Exception.Message
        }
        return $jobRes
    }).AddArgument($localVersionStr)

    $script:UpdateAsyncResult = $script:UpdateRunspace.BeginInvoke()

    # 3. Setup Timer
    $script:UpdateTimer = New-Object System.Windows.Threading.DispatcherTimer
    $script:UpdateTimer.Interval = [TimeSpan]::FromMilliseconds(500)
    $script:UpdateTicks = 0
    
    $script:UpdateTimer.Add_Tick({
        $lb = Get-Ctrl "LogBox"
        $script:UpdateTicks++
        
        # A. Timeout Check (20s)
        if ($script:UpdateTicks -gt 40) {
            $script:UpdateTimer.Stop()
            if ($script:UpdateRunspace) { $script:UpdateRunspace.Dispose() }
            if ($lb) { $lb.AppendText("[UPDATE] Error: Request timed out.`n"); $lb.ScrollToEnd() }
            return
        }

        # B. Check Job Status
        if ($script:UpdateAsyncResult.IsCompleted) {
            $script:UpdateTimer.Stop()
            
            try {
                $jobResult = $script:UpdateRunspace.EndInvoke($script:UpdateAsyncResult)
                $script:UpdateRunspace.Dispose()
                
                # Retrieve the actual object (EndInvoke returns a collection)
                if ($jobResult -is [System.Collections.ObjectModel.Collection[PSObject]]) {
                     $jobResult = $jobResult[0]
                }

                if ($jobResult.Status -eq "Success") {
                    $localVer  = [Version]$script:AppVersion
                    $remoteVer = [Version]$jobResult.RemoteVersion
                    
                    if ($lb) { 
                        $lb.AppendText("[UPDATE] Local: v$localVer | Remote: v$remoteVer`n")
                        $lb.ScrollToEnd()
                    }

                    if ($remoteVer -gt $localVer) {
                        if ($lb) { $lb.AppendText(" -> Update Available!`n"); $lb.ScrollToEnd() }
                        
                        # Use Dispatcher to show dialog on UI thread
                        $window.Dispatcher.Invoke([Action]{
                            $msg = "A new version is available!`n`nLocal Version:  v$localVer`nRemote Version: v$remoteVer`n`nDo you want to update now?"
                            $mbRes = [System.Windows.MessageBox]::Show($msg, "Update Available", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Information)
                            
                            if ($mbRes -eq [System.Windows.MessageBoxResult]::Yes) {
                                $remoteContent = $jobResult.Content
                                $backupName = "$(Split-Path $PSCommandPath -Leaf).bak"
                                $backupPath = Join-Path (Get-DataPath) $backupName
                                Copy-Item -Path $PSCommandPath -Destination $backupPath -Force
                                Set-Content -Path $PSCommandPath -Value $remoteContent -Encoding UTF8
                                
                                [System.Windows.MessageBox]::Show("Update complete! Restarting...", "Updated", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
                                Start-Process powershell.exe -ArgumentList "-File `"$PSCommandPath`""
                                $window.Close()
                            }
                        })
                    } else {
                        if ($lb) { $lb.AppendText(" -> System is up to date.`n"); $lb.ScrollToEnd() }
                    }
                } else {
                    if ($lb) { $lb.AppendText("[UPDATE] Failed: $($jobResult.Error)`n"); $lb.ScrollToEnd() }
                }
            } catch {
                if ($lb) { $lb.AppendText("[UPDATE] Processing Error: $($_.Exception.Message)`n"); $lb.ScrollToEnd() }
            }
        }
    })
    
    $script:UpdateTimer.Start()
}

# --- RESTORED LOGIC ---

function Start-UpdateRepair {
    Invoke-UiCommand {
        Stop-Service -Name wuauserv, bits, cryptsvc, msiserver -Force -ErrorAction SilentlyContinue
        $rnd = Get-Random
        if(Test-Path "$env:windir\SoftwareDistribution"){ Rename-Item "$env:windir\SoftwareDistribution" "$env:windir\SoftwareDistribution.bak_$rnd" -ErrorAction SilentlyContinue }
        if(Test-Path "$env:windir\System32\catroot2"){ Rename-Item "$env:windir\System32\catroot2" "$env:windir\System32\catroot2.bak_$rnd" -ErrorAction SilentlyContinue }
        netsh winsock reset | Out-Null
        Start-Service -Name wuauserv, bits, cryptsvc, msiserver -ErrorAction SilentlyContinue
    } "Repairing Windows Update..."
}

function Start-NetRepair {
    Invoke-UiCommand {
        ipconfig /release | Out-Null
        ipconfig /renew | Out-Null
        ipconfig /flushdns | Out-Null
        netsh winsock reset | Out-Null
        netsh int ip reset | Out-Null
    } "Running Full Network Repair..."
}

function Start-RegClean {
    Invoke-UiCommand {
        $bkDir = Join-Path (Get-DataPath) "RegistryBackups"
        if(!(Test-Path $bkDir)){ New-Item -Path $bkDir -ItemType Directory | Out-Null }
        $bkFile = "$bkDir\Backup_$(Get-Date -F 'yyyyMMdd_HHmm').reg"
        reg export "HKLM\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall" $bkFile /y | Out-Null
        $keys = Get-ChildItem HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall | Where-Object { $_.PSChildName -match 'IE40|IE4Data|DirectDrawEx|DXM_Runtime|SchedulingAgent' }
        if ($keys) { foreach ($k in $keys) { Remove-Item $k.PSPath -Recurse -Force; Write-Output "Removed: $($k.PSChildName)" } } else { Write-Output "No obsolete keys found." }
        Write-Output "Backup saved to: $bkFile"
    } "Cleaning Registry..."
}

function Start-XboxClean {
    Invoke-UiCommand {
        Write-Output "Stopping Xbox Auth Manager..."
        Stop-Service -Name "XblAuthManager" -Force -ErrorAction SilentlyContinue

        $allCreds = (cmdkey /list) -split "`r?`n"
        $targets = @()
        foreach ($line in $allCreds) {
            if ($line -match "(?i)^\\s*Target:.*(Xbl.*)$") { $targets += $matches[1] }
        }

        if ($targets.Count -eq 0) {
            Write-Output "No Xbox Live credentials found."
        } else {
            foreach ($t in $targets) {
                Write-Output "Deleting credential: $t"
                cmdkey /delete:$t 2>$null
            }
            Write-Output "Deleted $($targets.Count) credential(s)."
        }

        Start-Service -Name "XblAuthManager" -ErrorAction SilentlyContinue
    } "Cleaning Xbox Credentials..."
}

function Start-GpeditInstall {
    # Check for User Confirmation
    $msg = "Install Local Group Policy Editor?`n`nThis enables the Group Policy Editor (gpedit.msc) on Windows Home editions by installing the built-in system packages.`n`nContinue?"
    $res = [System.Windows.Forms.MessageBox]::Show($msg, "Confirm Install", [System.Windows.Forms.MessageBoxButtons]::YesNo, [System.Windows.Forms.MessageBoxImage]::Question)
    if ($res -eq "No") { return }

    Invoke-UiCommand {
        $packageRoot = Join-Path $env:SystemRoot "servicing\\Packages"
        
        if (-not (Test-Path $packageRoot)) {
            throw "Package directory not found: $packageRoot"
        }

        Write-Output "Searching packages in $packageRoot..."
        
        $clientTools = Get-ChildItem -Path $packageRoot -Filter "Microsoft-Windows-GroupPolicy-ClientTools-Package~*.mum" -ErrorAction SilentlyContinue
        $clientExtensions = Get-ChildItem -Path $packageRoot -Filter "Microsoft-Windows-GroupPolicy-ClientExtensions-Package~*.mum" -ErrorAction SilentlyContinue
        
        if (-not $clientTools -or -not $clientExtensions) {
            Write-Output "WARNING: Required GroupPolicy packages were not found."
            Write-Output "Ensure you are on a compatible Windows 10/11 version."
            return
        }

        $packages = @($clientTools + $clientExtensions) | Sort-Object Name -Unique
        
        foreach ($pkg in $packages) {
            Write-Output "Installing: $($pkg.Name)..."
            # Using DISM to add package
            $proc = Start-Process dism.exe -ArgumentList "/online","/norestart","/add-package:`"$($pkg.FullName)`"" -NoNewWindow -Wait -PassThru
            if ($proc.ExitCode -ne 0) {
                 Write-Output " -> Failed (Exit Code: $($proc.ExitCode))"
            }
        }
        Write-Output "`nInstallation Complete. Try running 'gpedit.msc'. (A reboot may be required)."
    } "Installing Group Policy Editor..."
}

# --- NETWORK / DNS HELPERS (from CLI) ---
function Get-ActiveAdapters {
    Get-NetAdapter | Where-Object { $_.Status -eq 'Up' -and $_.InterfaceDescription -notlike '*Virtual*' -and $_.Name -notlike '*vEthernet*' }
}

function Set-DnsAddresses {
    param(
        [string[]]$Addresses,
        [string]$Label = "Custom DNS"
    )
    if (-not $Addresses -or $Addresses.Count -eq 0) { return }
    $addrList = $Addresses
    $labelText = $Label
    
    Invoke-UiCommand {
        param($addrList, $labelText)
        $adapters = Get-ActiveAdapters | Select-Object -ExpandProperty Name
        if (-not $adapters) { Write-Output "No active adapters found."; return }
        foreach ($adapter in $adapters) {
            try {
                Set-DnsClientServerAddress -InterfaceAlias $adapter -ServerAddresses $addrList -ErrorAction Stop
                Write-Output "[$labelText] Applied to $adapter : $($addrList -join ', ')"
            } catch {
                Write-Output "[$labelText] Failed on $adapter : $($_.Exception.Message)"
            }
        }
    } "Applying $labelText..." -ArgumentList (,$addrList), $labelText
}

# --- SHARED DNS CONFIGURATION ---
$script:DohTargets = @(
    @{ Server = "1.1.1.1";               Template = "https://cloudflare-dns.com/dns-query" }
    @{ Server = "1.0.0.1";               Template = "https://cloudflare-dns.com/dns-query" }
    @{ Server = "2606:4700:4700::1111";  Template = "https://cloudflare-dns.com/dns-query" }
    @{ Server = "2606:4700:4700::1001";  Template = "https://cloudflare-dns.com/dns-query" }
    @{ Server = "8.8.8.8";               Template = "https://dns.google/dns-query" }
    @{ Server = "8.8.4.4";               Template = "https://dns.google/dns-query" }
    @{ Server = "2001:4860:4860::8888";  Template = "https://dns.google/dns-query" }
    @{ Server = "2001:4860:4860::8844";  Template = "https://dns.google/dns-query" }
    @{ Server = "9.9.9.9";               Template = "https://dns.quad9.net/dns-query" }
    @{ Server = "149.112.112.112";       Template = "https://dns.quad9.net/dns-query" }
    @{ Server = "2620:fe::fe";           Template = "https://dns.quad9.net/dns-query" }
    @{ Server = "2620:fe::fe:9";         Template = "https://dns.quad9.net/dns-query" }
    @{ Server = "94.140.14.14";          Template = "https://dns.adguard.com/dns-query" }
    @{ Server = "94.140.15.15";          Template = "https://dns.adguard.com/dns-query" }
    @{ Server = "2a10:50c0::ad1:ff";     Template = "https://dns.adguard.com/dns-query" }
    @{ Server = "2a10:50c0::ad2:ff";     Template = "https://dns.adguard.com/dns-query" }
)

function Start-DohJob {
    param($List, $IsEnable)

    # 1. UI Feedback
    # Attempt to find buttons. If not found, variables will be null.
    $btnEnable = Get-Ctrl "btnDohAuto"
    $btnDisable = Get-Ctrl "btnDohDisable"
    
    # Safely disable if found
    if ($btnEnable) { $btnEnable.IsEnabled = $false }
    if ($btnDisable) { $btnDisable.IsEnabled = $false }
    
    $actionStr = if ($IsEnable) { "Enabling" } else { "Disabling" }
    Write-GuiLog "$actionStr DoH... (Batch Processing)"

    # 2. Start Background Job
    $script:DohJob = Start-Job -ScriptBlock {
        param($List, $IsEnable)
        
        # Create a temp file for the batch commands
        $tmpFile = [System.IO.Path]::GetTempFileName()
        $cnt = 0
        
        # Build the batch file content
        $lines = @()
        foreach ($dns in $List) {
            if ($IsEnable) {
                $lines += "dns add encryption server=$($dns.Server) dohtemplate=$($dns.Template) autoupgrade=yes udpfallback=no"
            } else {
                $lines += "dns delete encryption server=$($dns.Server)"
            }
            $cnt++
        }
        
        # Write to file
        $lines | Set-Content -Path $tmpFile -Encoding Ascii

        # Run netsh in batch mode (-f)
        $p = Start-Process -FilePath "netsh.exe" -ArgumentList "-f", $tmpFile -NoNewWindow -PassThru -Wait
        
        # Cleanup
        if (Test-Path $tmpFile) { Remove-Item $tmpFile }
        
        # Helper tasks
        ipconfig /flushdns | Out-Null
        if ($IsEnable) {
            try { Restart-Service Dnscache -Force -ErrorAction SilentlyContinue } catch {}
        }
        
        # Return the result object
        if ($p.ExitCode -eq 0) {
            Write-Output ([PSCustomObject]@{ Count = $cnt; Success = $true })
        } else {
            Write-Output ([PSCustomObject]@{ Count = 0; Success = $false })
        }

    } -ArgumentList $List, $IsEnable

    # 3. Monitor Job
    if ($script:DohTimer) { $script:DohTimer.Stop() }
    $script:DohTimer = New-Object System.Windows.Threading.DispatcherTimer
    $script:DohTimer.Interval = [TimeSpan]::FromMilliseconds(500)
    
    $script:DohTimer.Add_Tick({
        if ($script:DohJob.State -ne 'Running') {
            $script:DohTimer.Stop()
            
            # Re-fetch buttons in this scope to be safe
            $btnEnable = Get-Ctrl "btnDohAuto"
            $btnDisable = Get-Ctrl "btnDohDisable"

            try {
                $rawResults = Receive-Job -Job $script:DohJob
                $res = $rawResults | Where-Object { $_ -is [PSCustomObject] -and $_.Psobject.Properties.Match('Count') } | Select-Object -Last 1

                if ($res -and $res.Success) {
                    $c = $res.Count
                    $finMsg = if ($IsEnable) { "Successfully applied $c DoH rules." } else { "Successfully removed $c DoH rules." }
                    
                    Write-GuiLog "Done. $finMsg"
                    
                    if ($c -gt 0) {
                        # FIXED: Use [System.Windows.MessageBox] (WPF) to match the Enum types
                        [System.Windows.MessageBox]::Show($finMsg, "DoH Manager", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
                    }
                } else {
                     Write-GuiLog "Operation failed or no changes were made."
                }
            } catch {
                Write-GuiLog "Job Error: $($_.Exception.Message)"
            }

            # Cleanup
            Remove-Job -Job $script:DohJob
            $script:DohJob = $null

            # Unlock Buttons (Safe Check)
            if ($btnEnable) { $btnEnable.IsEnabled = $true }
            if ($btnDisable) { $btnDisable.IsEnabled = $true }
        }
    })
    
    $script:DohTimer.Start()
}

# Redirect existing function calls to the new Async handler
function Enable-AllDoh { Start-DohJob -List $script:DohTargets -IsEnable $true }
function Disable-AllDoh { Start-DohJob -List $script:DohTargets -IsEnable $false }

# --- Hosts Adblock ---
function Invoke-HostsUpdate {
    Invoke-UiCommand {
        # 1. Find PATHS
        $hostsPath = "$env:windir\System32\drivers\etc\hosts"
        $backupDir = Join-Path (Get-DataPath) "hosts_backups"
        if (-not (Test-Path $backupDir)) { New-Item -ItemType Directory -Path $backupDir -Force | Out-Null }

        # Capture original ACL to preserve permissions (e.g., Users read access)
        $origAcl = $null
        if (Test-Path $hostsPath) {
            try { $origAcl = Get-Acl -Path $hostsPath } catch {}
        }

        # 2. DOWNLOAD HOSTS FILE
        $mirrors = @(
            "https://o0.pages.dev/Lite/hosts.win",
            "https://raw.githubusercontent.com/badmojr/1Hosts/master/Lite/hosts.win"
        )
        $adBlockContent = $null
        foreach ($mirror in $mirrors) {
            try {
                $wc = New-Object System.Net.WebClient
                # CRITICAL SPEED FIX: Bypasses auto-proxy detection delay (saves 1-5s)
                $wc.Proxy = $null 
                $wc.Encoding = [System.Text.Encoding]::UTF8
                
                Write-GuiLog "Downloading from $mirror..."
                $tempContent = $wc.DownloadString($mirror)
                
                # SAFETY CHECK: Ensure file is valid (> 1KB)
                if ($tempContent.Length -gt 1024) { 
                    $adBlockContent = $tempContent
                    Write-Output "Download complete ($([math]::Round($adBlockContent.Length / 1KB, 2)) KB)"
                    break 
                }
            } catch { 
                Write-Output "Mirror failed: $mirror" 
            } finally { 
                if ($wc) {$wc.Dispose()} 
            }
        }

        if (-not $adBlockContent) { 
            Write-GuiLog "ERROR: Download failed or file was empty. Aborting."
            return 
        }

        # 3. BACKUP EXISTING
        if (Test-Path $hostsPath) {
            $bkName = "hosts_$(Get-Date -F yyyyMMdd_HHmmss).bak"
            Copy-Item $hostsPath (Join-Path $backupDir $bkName) -Force
            Write-Output "Backup created: $bkName"
        }

        # 4. PRESERVE CUSTOM ENTRIES
        $customStart = "# === BEGIN USER CUSTOM ENTRIES ==="
        $customEnd = "# === END USER CUSTOM ENTRIES ==="
        $userEntries = "$customStart`r`n# Add custom entries here`r`n127.0.0.1 localhost`r`n::1 localhost`r`n$customEnd"

        if (Test-Path $hostsPath) {
            try {
                $raw = Get-Content $hostsPath -Raw
                if ($raw -match "(?s)$([regex]::Escape($customStart))(.*?)$([regex]::Escape($customEnd))") {
                    $userEntries = $matches[0]
                }
            } catch {}
        }

        # 5. CONSTRUCT & WRITE
        $finalContent = "$userEntries`r`n`r`n# UPDATED: $(Get-Date)`r`n$adBlockContent"
        
        try {
            Set-Content -Path $hostsPath -Value $finalContent -Encoding UTF8 -Force

            # Re-apply original ACL or ensure Users has read access
            try {
                if ($origAcl) {
                    Set-Acl -Path $hostsPath -AclObject $origAcl
                    Write-Output "Restored original permissions on hosts file."
                } else {
                    $fs = Get-Acl -Path $hostsPath
                    $rule = New-Object System.Security.AccessControl.FileSystemAccessRule("Users","ReadAndExecute","Allow")
                    $fs.SetAccessRule($rule)
                    Set-Acl -Path $hostsPath -AclObject $fs
                    Write-Output "Applied Users read permission to hosts file."
                }
            } catch {
                Write-Output "Warning: Could not reapply permissions: $($_.Exception.Message)"
            }
            
            # Validation
            if ((Get-Item $hostsPath).Length -lt 100) { throw "Write verification failed (File empty)." }
            
            ipconfig /flushdns | Out-Null
            Write-Output "Hosts file updated successfully."
        } catch {
            Write-GuiLog "CRITICAL ERROR: $($_.Exception.Message)"
            # Restore backup if write failed
            $latestBackup = Get-ChildItem $backupDir | Sort-Object CreationTime -Descending | Select-Object -First 1
            if ($latestBackup) {
                Copy-Item $latestBackup.FullName $hostsPath -Force
                Write-GuiLog "Restored backup due to failure."
            }
        }
    } "Updating hosts file..."
}
# --- HOSTS EDITOR ---
function Show-HostsEditor {
    # 1. SETUP FORM
    $hForm = New-Object System.Windows.Forms.Form
    $hForm.Text = "Hosts File Editor"
    $hForm.Size = "900, 700"
    $hForm.StartPosition = "CenterScreen"
    $hForm.BackColor = [System.Drawing.Color]::FromArgb(30,30,30)
    $hForm.KeyPreview = $true
    
    # Initialize Dirty Flag (False)
    $hForm.Tag = $false
    
    # 2. CONTROLS
    $txtHosts = New-Object System.Windows.Forms.RichTextBox
    $txtHosts.Dock = "Fill"
    $txtHosts.BackColor = [System.Drawing.Color]::FromArgb(45,45,48)
    $txtHosts.ForeColor = "White"
    $txtHosts.Font = "Consolas, 11"
    $txtHosts.AcceptsTab = $true
    $txtHosts.DetectUrls = $false
    $hForm.Controls.Add($txtHosts)
    
    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Dock = "Bottom"
    $pnl.Height = 50
    $hForm.Controls.Add($pnl)
    
    $btn = New-Object System.Windows.Forms.Button
    $btn.Text = "Save"
    $btn.BackColor = "SeaGreen"
    $btn.ForeColor = "White"
    $btn.FlatStyle = "Flat"
    $btn.Top = 10
    $btn.Left = 20
    $btn.Width = 100
    $pnl.Controls.Add($btn)
    
    $lblInfo = New-Object System.Windows.Forms.Label
    $lblInfo.Text = "Ctrl+S to Save"
    $lblInfo.ForeColor = "Gray"
    $lblInfo.AutoSize = $true
    $lblInfo.Top = 15
    $lblInfo.Left = 140
    $pnl.Controls.Add($lblInfo)
    
    $hostsPath = "$env:windir\System32\drivers\etc\hosts"
    
    # 3. LOAD FILE
    if (Test-Path $hostsPath) {
        $diskSize = (Get-Item $hostsPath).Length
        $content = Get-Content $hostsPath -Raw -ErrorAction SilentlyContinue
        
        # Safety Check
        if ($diskSize -gt 0 -and [string]::IsNullOrWhiteSpace($content)) {
            [System.Windows.Forms.MessageBox]::Show("Could not read Hosts file. Aborting.", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
            return
        }
        $txtHosts.Text = $content
    }
    
    # 4. HIGHLIGHTING HELPER
    $Highlight = {
        $sel = $txtHosts.SelectionStart
        $len = $txtHosts.SelectionLength
        $txtHosts.SelectAll()
        $txtHosts.SelectionColor = "White"
        $s = $txtHosts.Text.IndexOf("# === BEGIN USER CUSTOM ENTRIES ===")
        $e = $txtHosts.Text.IndexOf("# === END USER CUSTOM ENTRIES ===")
        if ($s -ge 0 -and $e -gt $s) {
            $txtHosts.Select($s, ($e + 33) - $s)
            $txtHosts.SelectionColor = "Cyan"
        }
        $txtHosts.Select($sel, $len)
    }
    & $Highlight
    
    # 5. CHANGE TRACKING
    $txtHosts.Add_TextChanged({
        $hForm.Tag = $true
        if ($hForm.Text -notmatch "\*$") {
            $hForm.Text = "Hosts File Editor *"
        }
    })
    
    # 6. SAVE LOGIC (Modified to use local variables)
    $SaveAction = {
        param($FormObj, $TextBox, $FilePath, $HighlightScript)
        
        try {
            if ([string]::IsNullOrWhiteSpace($TextBox.Text)) {
                $check = [System.Windows.Forms.MessageBox]::Show(
                    "Save EMPTY file?", 
                    "Warning", 
                    [System.Windows.Forms.MessageBoxButtons]::YesNo, 
                    [System.Windows.Forms.MessageBoxIcon]::Warning
                )
                if ($check -eq "No") { return $false }
            }
            
            Set-Content -Path $FilePath -Value $TextBox.Text -Encoding UTF8 -Force
            Start-Process icacls.exe -ArgumentList "`"$FilePath`" /reset" -NoNewWindow -Wait -ErrorAction SilentlyContinue
            
            if ((Get-Item $FilePath).Length -eq 0 -and $TextBox.Text.Length -gt 0) {
                throw "Write failed (0 bytes)."
            }
            
            # Reset State
            if ($FormObj) {
                $FormObj.Tag = $false
                $FormObj.Text = "Hosts File Editor"
            }
            
            [System.Windows.Forms.MessageBox]::Show("Saved successfully!", "Success", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
            
            # Re-apply highlighting
            & $HighlightScript
            
            return $true
        } catch {
            [System.Windows.Forms.MessageBox]::Show("Error saving: $_", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
            return $false
        }
    }
    
    # 7. EVENTS
    $btn.Add_Click({
        $null = & $SaveAction -FormObj $hForm -TextBox $txtHosts -FilePath $hostsPath -HighlightScript $Highlight
    })
    
    $hForm.Add_KeyDown({
        param($src, $e)
        if ($e.Control -and $e.KeyCode -eq 'S') {
            $e.SuppressKeyPress = $true
            $null = & $SaveAction -FormObj $src -TextBox $txtHosts -FilePath $hostsPath -HighlightScript $Highlight
        }
    })
    
    # 8. CLOSE PROMPT (FIXED - Pass all required parameters)
    $hForm.Add_FormClosing({
        param($src, $e)
        
        if ($src.Tag -eq $true) {
            $res = [System.Windows.Forms.MessageBox]::Show(
                "You have unsaved changes. Save now?", 
                "Confirm", 
                [System.Windows.Forms.MessageBoxButtons]::YesNoCancel, 
                [System.Windows.Forms.MessageBoxIcon]::Warning
            )
            
            if ($res -eq "Yes") {
                # Pass all required parameters to SaveAction
                $success = & $SaveAction -FormObj $src -TextBox $txtHosts -FilePath $hostsPath -HighlightScript $Highlight
                if (-not $success) {
                    $e.Cancel = $true
                }
            } elseif ($res -eq "Cancel") {
                $e.Cancel = $true
            }
            # If "No", just close without saving
        }
    })
    
    $hForm.ShowDialog()
}
# --- STORAGE / SYSTEM ---
function Invoke-ChkdskAll {
    $confirm = [System.Windows.MessageBox]::Show("Run CHKDSK /f /r on all drives? This may require a reboot and can take a while.", "Confirm CHKDSK", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    if ($confirm -ne "Yes") { return }
    Invoke-UiCommand {
        $drives = Get-PSDrive -PSProvider FileSystem | Where-Object { $null -ne $_.Free } | Select-Object -ExpandProperty Name
        foreach ($drive in $drives) {
            Write-Output "Scanning drive $drive`:"
            chkdsk "${drive}:" /f /r /x
        }
    } "Running CHKDSK on all drives..."
}
# ==========================================
# WINAPP2.INI INTEGRATION
# ==========================================

function Expand-EnvPath {
    param($Path)
    if ([string]::IsNullOrWhiteSpace($Path)) { return $Path }
    # Expand standard vars (%AppData%, etc)
    $expanded = [Environment]::ExpandEnvironmentVariables($Path)
    # Handle common Winapp2 specific variables if needed (e.g. %ProgramFiles%)
    return $expanded
}

function Get-Winapp2Rules {
    param([switch]$Download)

    $dataPath  = Get-DataPath
    $iniPath   = Join-Path $dataPath "winapp2.ini"
    $cachePath = Join-Path $dataPath "winapp2_cache.json" 

    # --- 1. SMART CACHE CHECK ---
    $forceRebuild = $false
    
    if (Test-Path $cachePath) {
        if ($PSCommandPath -and (Test-Path $PSCommandPath)) {
            $scriptTime = (Get-Item $PSCommandPath).LastWriteTime
            $cacheTime  = (Get-Item $cachePath).LastWriteTime
            if ($scriptTime -gt $cacheTime) { 
                $forceRebuild = $true 
            }
        }
    }

    # --- 2. DOWNLOAD ---
    if ($Download) {
        try {
            [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
            Add-Type -AssemblyName System.Net.Http
            $client = New-Object System.Net.Http.HttpClient
            $client.Timeout = [TimeSpan]::FromSeconds(15)
            
            $url = "https://cdn.jsdelivr.net/gh/MoscaDotTo/Winapp2@master/Winapp2.ini"
            $response = $client.GetAsync($url).Result
            if ($response.IsSuccessStatusCode) {
                $contentBytes = $response.Content.ReadAsByteArrayAsync().Result
                $iniContent = [System.Text.Encoding]::UTF8.GetString($contentBytes)
                [System.IO.File]::WriteAllText($iniPath, $iniContent)
                $forceRebuild = $true
            }
            $client.Dispose()
        } catch { Write-GuiLog "Download Warning: $($_.Exception.Message)" }
    }

    # --- 3. CACHE LOAD ---
    if (-not $forceRebuild -and (Test-Path $cachePath)) {
        try { 
            $cachedRules = Get-Content $cachePath -Raw | ConvertFrom-Json
            if ($cachedRules.Count -gt 5) { return $cachedRules }
        } catch {}
    }

    # --- 4. PARSE INI ---
    $iniContent = $null
    if (Test-Path $iniPath) { $iniContent = Get-Content $iniPath -Raw }
    if ([string]::IsNullOrWhiteSpace($iniContent)) { return @() }

    $rules = New-Object System.Collections.Generic.List[Object]
    
    $envVars = @{ 
        "%Documents%" = [Environment]::GetFolderPath("MyDocuments")
        "%ProgramFiles%" = $env:ProgramFiles
        "%ProgramFiles(x86)%" = ${env:ProgramFiles(x86)}
        "%SystemDrive%" = $env:SystemDrive
        "%AppData%" = $env:APPDATA
        "%LocalAppData%" = $env:LOCALAPPDATA
        "%CommonAppData%" = $env:ProgramData
        "%UserProfile%" = $env:USERPROFILE
    }
    $dirCache = @{} 

    $lines = $iniContent -split "\r?\n"
    $currentApp = $null; $skipApp = $false; $hasDetect = $false

    foreach ($line in $lines) {
        if ([string]::IsNullOrWhiteSpace($line) -or $line[0] -eq ';') { continue }

        if ($line[0] -eq '[') {
            if ($currentApp -and -not $skipApp) { $rules.Add([PSCustomObject]$currentApp) }
            $appName = $line.Trim(" []")
            $currentApp = [ordered]@{ 
                Name = $appName
                ID = "Winapp2_" + ($appName -replace '[^a-zA-Z0-9]','')
                Section = "Applications"
                AppGroup = "General"
                Paths = New-Object System.Collections.Generic.List[Object]
                Desc = ""
                IsInternal = $false
            }
            $skipApp = $false; $hasDetect = $false
            continue
        }

        $eqIndex = $line.IndexOf('=')
        if ($eqIndex -le 0) { continue }
        if ($skipApp) { continue }

        $key = $line.Substring(0, $eqIndex).Trim()
        $val = $line.Substring($eqIndex + 1).Trim()

        if ($key -eq "Section") { $currentApp.Section = $val }
        elseif ($key.StartsWith("Detect")) {
            if (-not $hasDetect) { $hasDetect = $true; $skipApp = $true } 
            
            if ($val.IndexOf('%') -ge 0) { 
                foreach ($k in $envVars.Keys) { 
                    if ($val.Contains($k)) { $val = $val.Replace($k, $envVars[$k]) } 
                } 
            }

            # Registry Detection
            if ($val -match "^HK") {
                $regPath = $val -replace "^(?i)HKCU", "Registry::HKEY_CURRENT_USER" `
                                -replace "^(?i)HKLM", "Registry::HKEY_LOCAL_MACHINE" `
                                -replace "^(?i)HKCR", "Registry::HKEY_CLASSES_ROOT" `
                                -replace "^(?i)HKU",  "Registry::HKEY_USERS"
                if (Test-Path $regPath) { $skipApp = $false }
            } 
            # File Detection
            else {
                try {
                    $parent = [System.IO.Path]::GetDirectoryName($val)
                    if (-not [string]::IsNullOrWhiteSpace($parent)) {
                        if (-not $dirCache.ContainsKey($parent)) { $dirCache[$parent] = (Test-Path $parent) }
                        if ($dirCache[$parent]) { 
                            if (Test-Path $val) { $skipApp = $false } 
                        }
                    }
                } catch { 
                    if (Test-Path $val) { $skipApp = $false } 
                }
            }
        }
        elseif ($key.StartsWith("FileKey")) {
            $parts = $val -split "\|"
            if ($parts.Count -ge 2) {
                $rawPath = $parts[0]
                if ($rawPath.IndexOf('%') -ge 0) { foreach ($k in $envVars.Keys) { if ($rawPath.Contains($k)) { $rawPath = $rawPath.Replace($k, $envVars[$k]) } } }
                $rawPath = [Environment]::ExpandEnvironmentVariables($rawPath)
                if (-not $skipApp) { $currentApp.Paths.Add(@{ Path = $rawPath; Pattern = $parts[1]; Options = if ($parts.Count -gt 2) { $parts[2] } else { "" } }) }
            }
        }
        elseif ($key -eq "Description") { $currentApp.Desc = $val }
    }
    if ($currentApp -and -not $skipApp) { $rules.Add([PSCustomObject]$currentApp) }

    # --- 5. CATEGORIZATION ---
    $finalList = $rules | Where-Object { $_.Paths.Count -gt 0 }
    
    foreach ($app in $finalList) {
        $name = $app.Name

        # 1. BROWSERS
        if ($name -match "^Google Chrome") { $app.AppGroup = "Google Chrome"; $app.Section = "Browsers / Internet"; $app.Name = $name -replace "Google Chrome\s*", "" }
        elseif ($name -match "^Microsoft Edge") { $app.AppGroup = "Microsoft Edge"; $app.Section = "Browsers / Internet"; $app.Name = $name -replace "Microsoft Edge\s*", "" }
        elseif ($name -match "^Mozilla Firefox") { $app.AppGroup = "Mozilla Firefox"; $app.Section = "Browsers / Internet"; $app.Name = $name -replace "Mozilla Firefox\s*", "" }
        elseif ($name -match "^Opera") { $app.AppGroup = "Opera"; $app.Section = "Browsers / Internet" }
        elseif ($name -match "^Brave") { $app.AppGroup = "Brave"; $app.Section = "Browsers / Internet" }

        # 2. SPECIFIC PRODUCTIVITY
        elseif ($name -match "PowerToys") { 
            $app.Section = "Productivity"
            $app.AppGroup = "Microsoft PowerToys"
            $app.Name = $name -replace "^Microsoft\s*PowerToys\s*", "" 
        }
        elseif ($name -match "^Microsoft\sOffice|^Office\s") { $app.Section = "Productivity"; $app.AppGroup = "Microsoft Office" }
        elseif ($name -match "^Adobe\s") { $app.Section = "Productivity"; $app.AppGroup = "Adobe"; $app.Name = $name -replace "^Adobe\s+", "" }
        
        # 3. GAMES (New Category)
        elseif ($name -match "(?i)\b(Steam|Epic Games|Origin|Uplay|Ubisoft Connect|Battle.net|GOG Galaxy)\b") {
            $app.Section = "Games"
            $app.AppGroup = $name -split " " | Select-Object -First 1
        }

        # 4. CHAT APPS
        elseif ($name -match "(?i)\b(Discord|Spotify|Skype|TeamViewer|Zoom|Slack|Telegram|WhatsApp)\b") { 
            $app.Section = "Internet & Chat"
            $app.AppGroup = $name -split " " | Select-Object -First 1 
        }

        # 5. SYSTEM CATCH-ALL
        elseif ($name -match "^Windows\s" -or $name -eq "Windows" -or $name -match "Defender|Explorer|Store|Management Console") {
            $app.Section = "System"
            $app.AppGroup = "Windows"
            $app.Name = $name -replace "^Windows\s+", "" 
        }
    }

    try { $finalList | ConvertTo-Json -Depth 5 | Set-Content $cachePath -Force } catch {}
    
    return $finalList
}

function Show-AdvancedCleanupSelection {

    $currentSettings = Get-WmtSettings
    $savedStates = $currentSettings.TempCleanup
    $isWinapp2Enabled = $currentSettings.LoadWinapp2

    # --- FORM SETUP ---
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Advanced Cleanup Selection"
    $form.Size = New-Object System.Drawing.Size(650, 850)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = 'FixedDialog'
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $form.ForeColor = "White"

    # --- 1. TOP PANEL (Search) ---
    $topPanel = New-Object System.Windows.Forms.Panel
    $topPanel.Dock = "Top"; $topPanel.Height = 50
    $topPanel.BackColor = [System.Drawing.Color]::FromArgb(40, 40, 40)

    $chkToggleWinapp2 = New-Object System.Windows.Forms.CheckBox
    $chkToggleWinapp2.Text = "Load Community Rules *"
    $chkToggleWinapp2.Size = "220, 30"; $chkToggleWinapp2.Location = "15, 10"
    $chkToggleWinapp2.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
    $chkToggleWinapp2.ForeColor = "White"
    $chkToggleWinapp2.Checked = $isWinapp2Enabled
    $tt = New-Object System.Windows.Forms.ToolTip
    $tt.SetToolTip($chkToggleWinapp2, "Enables 1000+ extra rules from Winapp2.ini")
    $topPanel.Controls.Add($chkToggleWinapp2)

    $txtSearch = New-Object System.Windows.Forms.TextBox
    $txtSearch.Size = "200, 25"; $txtSearch.Location = "420, 12"
    $txtSearch.BackColor = [System.Drawing.Color]::FromArgb(20, 20, 20)
    $txtSearch.ForeColor = "White"
    $txtSearch.BorderStyle = "FixedSingle"
    $topPanel.Controls.Add($txtSearch)
    
    $lblSearch = New-Object System.Windows.Forms.Label
    $lblSearch.Text = "Search:"
    $lblSearch.AutoSize = $true; $lblSearch.Location = "370, 15"
    $topPanel.Controls.Add($lblSearch)

    # --- 2. BOTTOM PANEL (Buttons) ---
    $btnPanel = New-Object System.Windows.Forms.Panel
    $btnPanel.Dock = "Bottom"; $btnPanel.Height = 60
    $btnPanel.BackColor = [System.Drawing.Color]::FromArgb(25, 25, 25)

    $btnClean = New-Object System.Windows.Forms.Button
    $btnClean.Text = "Clean Selected"
    $btnClean.Size = "140, 35"; $btnClean.Location = "470, 12"
    $btnClean.BackColor = "SeaGreen"; $btnClean.ForeColor = "White"; $btnClean.FlatStyle = "Flat"
    $btnClean.DialogResult = "OK"
    $btnPanel.Controls.Add($btnClean)

    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Size = "100, 35"; $btnCancel.Location = "360, 12"
    $btnCancel.BackColor = "DimGray"; $btnCancel.ForeColor = "White"; $btnCancel.FlatStyle = "Flat"
    $btnCancel.DialogResult = "Cancel"
    $btnPanel.Controls.Add($btnCancel)

    # --- 3. MAIN CONTENT PANEL ---
    $mainPanel = New-Object System.Windows.Forms.FlowLayoutPanel
    $mainPanel.FlowDirection = "TopDown"; $mainPanel.WrapContents = $false
    $mainPanel.AutoScroll = $true; $mainPanel.Dock = "Fill"
    $mainPanel.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $mainPanel.Padding = New-Object System.Windows.Forms.Padding(5, 10, 0, 0)

    $form.Controls.Add($btnPanel)
    $form.Controls.Add($topPanel)
    $form.Controls.Add($mainPanel)
    $mainPanel.BringToFront()

    # --- INTERNAL RULES ---
    $internalRules = @(
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Windows"; Name="Temporary Files"; Key="TempFiles"; Desc="User and System Temp"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Windows"; Name="Recycle Bin"; Key="RecycleBin"; Desc="Empties Recycle Bin"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Windows"; Name="Error Logs (WER)"; Key="WER"; Desc="Crash dumps"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Windows"; Name="DNS Cache"; Key="DNS"; Desc="Network cache"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Explorer"; Name="Thumbnail Cache"; Key="Thumbnails"; Desc="Explorer thumbnails"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Explorer"; Name="Recent Items"; Key="Recent"; Desc="Recent files list"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="System"; AppGroup="Explorer"; Name="Run History"; Key="RunMRU"; Desc="Run dialog history"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="Browsers / Internet"; AppGroup="Google Chrome"; Name="Cache (Internal)"; Key="Chrome"; Desc="Standard Cache"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="Browsers / Internet"; AppGroup="Microsoft Edge"; Name="Cache (Internal)"; Key="Edge"; Desc="Standard Cache"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="Browsers / Internet"; AppGroup="Mozilla Firefox"; Name="Cache (Internal)"; Key="Firefox"; Desc="Standard Cache"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="Browsers / Internet"; AppGroup="Brave"; Name="Cache (Internal)"; Key="Brave"; Desc="Standard Cache"; IsInternal=$true }
        [PSCustomObject][ordered]@{ Section="Browsers / Internet"; AppGroup="Opera"; Name="Cache (Internal)"; Key="Opera"; Desc="Standard Cache"; IsInternal=$true }
    )

    $RenderList = {
        param($IncludeWinapp2, $InteractiveMode)
        
        $mainPanel.SuspendLayout()
        $mainPanel.Controls.Clear()
        
        $allRules = @($internalRules)

        if ($IncludeWinapp2) {
            $lbl = New-Object System.Windows.Forms.Label; $lbl.Text = "Loading..."; $lbl.ForeColor = "Yellow"; $lbl.AutoSize = $true; $lbl.Margin = "10,0,0,0"
            $mainPanel.Controls.Add($lbl); $form.Update()

            $iniPath = Join-Path (Get-DataPath) "winapp2.ini"
            $shouldDownload = $false
            if (Test-Path $iniPath) {
                if ((Get-Item $iniPath).LastWriteTime -lt (Get-Date).AddDays(-7) -and $InteractiveMode) {
                   if ([System.Windows.Forms.MessageBox]::Show("Update Rules?", "Update", "YesNo") -eq "Yes") { $shouldDownload = $true }
                }
            } elseif ($InteractiveMode) { $shouldDownload = $true }

            try { $winRules = Get-Winapp2Rules -Download:$shouldDownload; $allRules += $winRules } catch {}
            if ($mainPanel.Controls.Count -gt 0) { $mainPanel.Controls.RemoveAt(0) }
        }

        $global:checkboxes = @{}
        $global:sections = @()

        $sections = $allRules | Select-Object -ExpandProperty Section -Unique | Sort-Object

        foreach ($sec in $sections) {
            $secPanel = New-Object System.Windows.Forms.Panel
            $secPanel.Size = "600, 35"; $secPanel.Margin = "5, 10, 0, 0"
            $secPanel.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 48)
            $secPanel.Tag = "HEADER"
            
            $secChk = New-Object System.Windows.Forms.CheckBox
            $secChk.Text = $sec
            $secChk.Font = New-Object System.Drawing.Font("Segoe UI", 11, [System.Drawing.FontStyle]::Bold)
            $secChk.ForeColor = [System.Drawing.Color]::DeepSkyBlue
            $secChk.AutoSize = $true; $secChk.Location = "5, 5"
            $secPanel.Controls.Add($secChk)
            $mainPanel.Controls.Add($secPanel)
            
            $global:sections += $secPanel

            $itemFlow = New-Object System.Windows.Forms.FlowLayoutPanel
            $itemFlow.FlowDirection = "TopDown"; $itemFlow.AutoSize = $true
            $itemFlow.Margin = "25, 0, 0, 0"
            $itemFlow.Tag = "FLOW"

            $secItems = $allRules | Where-Object { $_.Section -eq $sec } | Sort-Object AppGroup, Name
            $childChecks = @()
            $currentGroup = $null
            $isSecChecked = $true

            foreach ($item in $secItems) {
                if ($item.AppGroup -ne $currentGroup) {
                    $currentGroup = $item.AppGroup
                    $grpLbl = New-Object System.Windows.Forms.Label
                    $grpLbl.Text = $currentGroup
                    $grpLbl.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
                    $grpLbl.ForeColor = [System.Drawing.Color]::LightGray
                    $grpLbl.AutoSize = $true; $grpLbl.Margin = "0, 10, 0, 2"
                    $grpLbl.Tag = "GROUPHEADER"
                    $itemFlow.Controls.Add($grpLbl)
                }

                $itemKey = if ($item.Key) { $item.Key } else { $item.ID }
                $chk = New-Object System.Windows.Forms.CheckBox
                
                if ($item.IsInternal) {
                    $chk.Text = $item.Name
                } else {
                    $cleanName = $item.Name.Trim(" *")
                    $chk.Text = "$cleanName (*)"
                    $chk.ForeColor = [System.Drawing.Color]::FromArgb(200, 200, 255)
                }
                
                $chk.AutoSize = $true; $chk.Margin = "10, 0, 0, 2"
                $chk.Tag = if ($item.IsInternal) { $itemKey } else { $item }
                
                if ($savedStates.ContainsKey($itemKey)) { $chk.Checked = $savedStates[$itemKey] }
                else { $chk.Checked = ($item.IsInternal -eq $true) }
                
                if (-not $chk.Checked) { $isSecChecked = $false }
                if ($item.Desc) { $tt.SetToolTip($chk, $item.Desc) }

                $itemFlow.Controls.Add($chk)
                $global:checkboxes[$itemKey] = $chk
                $childChecks += $chk
            }

            $secChk.Checked = $isSecChecked
            $mainPanel.Controls.Add($itemFlow)
            $secChk.Add_Click({ param($s,$e) foreach ($c in $childChecks) { $c.Checked = $s.Checked } }.GetNewClosure())
        }
        $mainPanel.ResumeLayout()
    }

    & $RenderList -IncludeWinapp2 $isWinapp2Enabled -InteractiveMode $false

    # --- SEARCH LOGIC ---
    $txtSearch.Add_TextChanged({
        $q = $txtSearch.Text.ToLower()
        $mainPanel.SuspendLayout()
        
        for ($i = 0; $i -lt $mainPanel.Controls.Count; $i++) {
            $ctrl = $mainPanel.Controls[$i]
            
            if ($ctrl.Tag -eq "FLOW") {
                $hasVisibleChildren = $false
                foreach ($child in $ctrl.Controls) {
                    if ($child -is [System.Windows.Forms.CheckBox]) {
                        if ($child.Text.ToLower().Contains($q)) {
                            $child.Visible = $true
                            $hasVisibleChildren = $true
                        } else {
                            $child.Visible = $false
                        }
                    } 
                    elseif ($child.Tag -eq "GROUPHEADER") {
                        # Hide group headers during search to save space
                        $child.Visible = ($q.Length -eq 0) 
                    }
                }
                $ctrl.Visible = $hasVisibleChildren
                # Toggle SECTION Header
                if ($i -gt 0) { $mainPanel.Controls[$i-1].Visible = $hasVisibleChildren }
            }
        }
        $mainPanel.ResumeLayout()
    })

    $chkToggleWinapp2.Add_Click({
        $currentSettings.LoadWinapp2 = $chkToggleWinapp2.Checked
        Save-WmtSettings -Settings $currentSettings
        & $RenderList -IncludeWinapp2 $chkToggleWinapp2.Checked -InteractiveMode $true
    })

    $form.AcceptButton = $btnClean
    $form.CancelButton = $btnCancel

    if ($form.ShowDialog() -eq "OK") {
        $selectedItems = @()
        foreach ($key in $global:checkboxes.Keys) {
            $cb = $global:checkboxes[$key]
            # --- FIX APPLIED HERE ---
            # Removed "-and $cb.Visible". If it is Checked, we clean it, 
            # regardless of whether the user has currently filtered it out via Search.
            if ($cb.Checked) { $selectedItems += $cb.Tag }
            $currentSettings.TempCleanup[$key] = $cb.Checked
        }
        $currentSettings.LoadWinapp2 = $chkToggleWinapp2.Checked
        Save-WmtSettings -Settings $currentSettings
        return $selectedItems
    }
    return $null
}

function Invoke-TempCleanup {
    # 1. GET SELECTION
    $selections = Show-AdvancedCleanupSelection
    if (-not $selections -or $selections.Count -eq 0) { 
        Write-GuiLog "Cleanup canceled: No items selected."
        return 
    }

    # 2. SETUP PROGRESS UI
    $pForm = New-Object System.Windows.Forms.Form
    $pForm.Text = "Deep Cleaning System"
    $pForm.Size = "500,160"
    $pForm.StartPosition = "CenterScreen"
    $pForm.ControlBox = $false
    $pForm.BackColor = [System.Drawing.Color]::FromArgb(30, 30, 30)
    $pForm.ForeColor = "White"

    $pLabel = New-Object System.Windows.Forms.Label
    $pLabel.Location = "20,15"; $pLabel.Size = "460,20"
    $pLabel.Text = "Initializing..."
    $pForm.Controls.Add($pLabel)

    $pStatus = New-Object System.Windows.Forms.Label
    $pStatus.Location = "20,40"; $pStatus.Size = "460,20"
    $pStatus.ForeColor = "Gray"
    $pStatus.Text = "Preparing..."
    $pForm.Controls.Add($pStatus)

    $pBar = New-Object System.Windows.Forms.ProgressBar
    $pBar.Location = "20,70"; $pBar.Size = "440,20"
    $pForm.Controls.Add($pBar)

    $pForm.Show()
    [System.Windows.Forms.Application]::DoEvents()

    # 3. STATS TRACKING
    $stats = @{
        Deleted = 0
        Bytes   = 0
        Progress = 0.0
    }
    
    $ruleWeight = 100.0 / ($selections.Count)

    # --- HELPER: ROBUST CLEANER ---
    function Invoke-RobustClean {
        param($Path, $Pattern="*", $Recurse=$true)
        
        $Path = [Environment]::ExpandEnvironmentVariables($Path)
        if (-not (Test-Path -LiteralPath $Path)) { return }

        $pStatus.Text = "Scanning: $(Split-Path $Path -Leaf)"
        [System.Windows.Forms.Application]::DoEvents()

        $opt = if ($Recurse) { [System.IO.SearchOption]::AllDirectories } else { [System.IO.SearchOption]::TopDirectoryOnly }
        
        try {
            # Use EnumerateFiles for lower memory usage on large folders
            $files = [System.IO.Directory]::EnumerateFiles($Path, $Pattern, $opt)
            
            $batchCount = 0

            foreach ($file in $files) {
                try {
                    $fInfo = [System.IO.FileInfo]::new($file)
                    $size = $fInfo.Length
                    $fInfo.Delete() 

                    $stats.Deleted++
                    $stats.Bytes += $size
                    $batchCount++

                    # OPTIMIZATION: Update UI only every 50 files
                    if ($batchCount -gt 50) {
                        $mb = [math]::Round($stats.Bytes / 1MB, 2)
                        $pLabel.Text = "Removed: $($stats.Deleted) | Recovered: $mb MB"
                        [System.Windows.Forms.Application]::DoEvents()
                        $batchCount = 0
                    }
                } catch {}
            }

            if ($Recurse) {
                $dirs = [System.IO.Directory]::GetDirectories($Path, "*", [System.IO.SearchOption]::AllDirectories)
                [Array]::Sort($dirs, [System.Collections.Comparer]::Default)
                [Array]::Reverse($dirs)
                
                foreach ($dir in $dirs) {
                    try {
                        [System.IO.Directory]::Delete($dir, $false) 
                    } catch {}
                }
            }
        } catch {}
    }

    # 4. MAIN EXECUTION LOOP
    Write-GuiLog "--- Starting Cleanup ---"
    
    try {
        foreach ($item in $selections) {
            if ($pForm.IsDisposed) { break }

            # Snapshot bytes BEFORE this item
            $startBytes = $stats.Bytes

            # Update UI
            $stats.Progress += $ruleWeight
            $pBar.Value = [int]$stats.Progress
            
            # Determine Name for Logging
            $itemName = if ($item -is [string]) { $item } else { $item.Name }
            $pLabel.Text = "Cleaning: $itemName"

            # --- A. WINAPP2 RULES (Object) ---
            if ($item -is [System.Collections.IDictionary] -or $item -is [PSCustomObject]) {
                foreach ($rule in $item.Paths) {
                    $isRecurse = ($rule.Options -notmatch "REMOVESELF")
                    Invoke-RobustClean -Path $rule.Path -Pattern $rule.Pattern -Recurse $isRecurse
                }
            }
            # --- B. INTERNAL RULES (String) ---
            else {
                switch ($item) {
                    "TempFiles" { 
                        Invoke-RobustClean $env:TEMP
                        Invoke-RobustClean "$env:SystemRoot\Temp"
                    }
                    "RecycleBin" { 
                        try {
                            # FORCE CLEAN METHOD (COM Object)
                            $shell = New-Object -ComObject Shell.Application
                            $bin = $shell.Namespace(0xA) # 0xA = Recycle Bin
                            $items = $bin.Items()
                            $count = $items.Count

                            if ($count -gt 0) {
                                Write-GuiLog "Recycle Bin: Force cleaning $count items..."
                                
                                foreach ($f in $items) {
                                    try {
                                        $path = $f.Path
                                        
                                        # Measure Size (Accurately using Get-ChildItem on the real path)
                                        if ($path -and (Test-Path -LiteralPath $path)) {
                                            $size = (Get-ChildItem -LiteralPath $path -Recurse -Force -ErrorAction SilentlyContinue | Measure-Object -Property Length -Sum).Sum
                                            $stats.Bytes += $size
                                            
                                            # Force Delete
                                            Remove-Item -LiteralPath $path -Recurse -Force -ErrorAction SilentlyContinue
                                            $stats.Deleted++
                                        }
                                    } catch {}
                                }
                            }
                        } catch {
                            Write-GuiLog "Recycle Bin Error: $($_.Exception.Message)"
                        }
                    }
                    "WER" { Invoke-RobustClean "$env:ProgramData\Microsoft\Windows\WER" }
                    "DNS" { Clear-DnsClientCache -ErrorAction SilentlyContinue }
                    "Thumbnails" { Invoke-RobustClean "$env:LOCALAPPDATA\Microsoft\Windows\Explorer" -Pattern "thumbcache_*.db" -Recurse:$false }
                    "Recent" { Invoke-RobustClean "$env:APPDATA\Microsoft\Windows\Recent" }
                    "RunMRU" { Remove-ItemProperty -Path "HKCU:\Software\Microsoft\Windows\CurrentVersion\Explorer\RunMRU" -Name * -ErrorAction SilentlyContinue }
                    "Edge"    { Invoke-RobustClean "$env:LOCALAPPDATA\Microsoft\Edge\User Data\Default\Cache" }
                    "Chrome"  { Invoke-RobustClean "$env:LOCALAPPDATA\Google\Chrome\User Data\Default\Cache" }
                    "Brave"   { Invoke-RobustClean "$env:LOCALAPPDATA\BraveSoftware\Brave-Browser\User Data\Default\Cache" }
                    "Firefox" { 
                        if (Test-Path "$env:LOCALAPPDATA\Mozilla\Firefox\Profiles") {
                            Get-ChildItem "$env:LOCALAPPDATA\Mozilla\Firefox\Profiles" -Directory | ForEach-Object { Invoke-RobustClean "$($_.FullName)\cache2\entries" }
                        }
                    }
                    "Opera"   { Invoke-RobustClean "$env:APPDATA\Opera Software\Opera Stable\Cache" }
                    "OperaGX" { Invoke-RobustClean "$env:APPDATA\Opera Software\Opera GX Stable\Cache" }
                }
            }

            # Calculate difference for this specific item
            $diffBytes = $stats.Bytes - $startBytes
            
            # Log it if we actually removed something
            if ($diffBytes -gt 0) {
                $itemMB = [math]::Round($diffBytes / 1MB, 2)
                Write-GuiLog "Cleaned $itemName : $itemMB MB"
            }
        }
    } catch {
        Write-GuiLog "Error: $($_.Exception.Message)"
    } finally {
        $pForm.Close()
    }

    # 5. FINAL REPORT
    $finalMB = [math]::Round($stats.Bytes / 1MB, 2)
    Write-GuiLog "Total Removed: $finalMB MB"
    
    $msg = "Cleanup Complete.`n`nFiles Removed: $($stats.Deleted)`nSpace Recovered: $finalMB MB"
    [System.Windows.Forms.MessageBox]::Show($msg, "Cleanup Results", [System.Windows.Forms.MessageBoxButton]::OK, [System.Windows.Forms.MessageBoxImage]::Information) | Out-Null
}

# --- Registry Scan Selection UI ---
function Show-RegScanSelection {
    # 1. LOAD SETTINGS
    $currentSettings = Get-WmtSettings
    $savedStates = $currentSettings.RegistryScan

    # --- Form Setup ---
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Select Registry Scan Targets"
    $f.Size = "600, 550"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $f.ForeColor = "White"
    $f.FormBorderStyle = "FixedDialog"
    $f.MaximizeBox = $false

    # --- Header Label ---
    $lbl = New-Object System.Windows.Forms.Label
    $lbl.Text = "Select areas to scan:"
    $lbl.AutoSize = $true; $lbl.Location = "20, 15"
    $lbl.Font = New-Object System.Drawing.Font("Segoe UI", 10, [System.Drawing.FontStyle]::Bold)
    $f.Controls.Add($lbl)

    # --- Scrollable Panel for Checkboxes ---
    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Location = "20, 50"; $pnl.Size = "550, 380"; $pnl.AutoScroll = $true
    $f.Controls.Add($pnl)

    # --- Define Categories ---
    $categories = [ordered]@{
        "Missing Shared DLLs"            = "SharedDLLs"
        "Unused File Extensions (System)"= "Ext"
        "Unused File Extensions (User)"  = "FileExts"
        "ActiveX & COM Issues"           = "ActiveX"
        "Type Libraries (TLB)"           = "TypeLib"
        "Application Paths"              = "AppPaths"
        "Applications (Registered)"      = "Apps"
        "Installer Folders"              = "Installer"
        "Obsolete Software (Uninstall)"  = "Uninstall"
        "Run At Startup"                 = "Startup"
        "Invalid Default Icons"          = "Icons"
        "File Associations"              = "ProgIDs"
        "Windows Services"               = "Services"
        "MUI Cache (MRU Lists)"          = "MuiCache"
        "Compatibility Store (Flags)"    = "AppCompat"
        "Firewall Rules"                 = "Firewall"
    }

    # --- Generate Checkboxes Dynamically ---
    $chkBoxes = @(); $y = 0; $count = 0
    foreach ($key in $categories.Keys) {
        $tag = $categories[$key]
        $chk = New-Object System.Windows.Forms.CheckBox
        $chk.Text = $key
        $chk.Tag = $tag
        $chk.AutoSize = $true
        
        # APPLY SAVED STATE (Default to True)
        if ($savedStates.ContainsKey($tag)) {
            $chk.Checked = $savedStates[$tag]
        } else {
            $chk.Checked = $true
        }
        
        # Grid Layout: 2 Columns
        if ($count % 2 -eq 0) { $x = 0 } else { $x = 280 }
        $chk.Location = "$x, $y"
        
        if ($count % 2 -ne 0) { $y += 30 }
        
        $pnl.Controls.Add($chk); $chkBoxes += $chk; $count++
    }

    # --- Buttons ---
    $btnScan = New-Object System.Windows.Forms.Button
    $btnScan.Text = "Start Deep Scan"
    $btnScan.Location = "340, 450"; $btnScan.Width = 200; $btnScan.Height = 40
    $btnScan.BackColor = "SeaGreen"; $btnScan.ForeColor = "White"; $btnScan.FlatStyle = "Flat"
    $btnScan.DialogResult = "OK"
    $f.Controls.Add($btnScan)

    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = "20, 450"; $btnCancel.Width = 100; $btnCancel.Height = 40
    $btnCancel.BackColor = "DimGray"; $btnCancel.ForeColor = "White"; $btnCancel.FlatStyle = "Flat"
    $f.Controls.Add($btnCancel)
    
    $f.AcceptButton = $btnScan; $f.CancelButton = $btnCancel

    if ($f.ShowDialog() -eq "OK") {
        $selected = @()
        
        # SAVE SETTINGS
        foreach ($c in $chkBoxes) { 
            if ($c.Checked) { $selected += $c.Tag }
            $currentSettings.RegistryScan[$c.Tag] = $c.Checked
        }
        Save-WmtSettings -Settings $currentSettings

        return $selected
    }
    return $null
}
# --- Registry Results UI ---
function Show-RegistryCleaner {
    param($ScanResults)

    # --- 1. Form Setup ---
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Deep Registry Cleaner"
    $f.Size = "1100, 600"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.ForeColor = [System.Drawing.Color]::White
    
    # --- 2. Header Panel ---
    $pnlHead = New-Object System.Windows.Forms.Panel
    $pnlHead.Dock = "Top"; $pnlHead.Height = 60
    $pnlHead.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
    $f.Controls.Add($pnlHead)

    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Text = "Scan Complete. Issues found: $($ScanResults.Count)"
    $lblStatus.AutoSize = $true; $lblStatus.Top = 18; $lblStatus.Left = 15
    $lblStatus.Font = New-Object System.Drawing.Font("Segoe UI", 11, [System.Drawing.FontStyle]::Bold)
    $pnlHead.Controls.Add($lblStatus)

    # --- 3. Data Grid Configuration ---
    $dg = New-Object System.Windows.Forms.DataGridView
    $dg.Dock = "Fill"
    $dg.BackgroundColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.ForeColor = [System.Drawing.Color]::White
    $dg.GridColor = [System.Drawing.ColorTranslator]::FromHtml("#333333")
    $dg.BorderStyle = "None"
    $dg.RowHeadersVisible = $false
    $dg.AllowUserToAddRows = $false
    $dg.SelectionMode = "FullRowSelect"
    $dg.MultiSelect = $true
    
    # Header Styling
    $dg.EnableHeadersVisualStyles = $false
    $dg.ColumnHeadersDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
    $dg.ColumnHeadersDefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.ColumnHeadersDefaultCellStyle.Padding = (New-Object System.Windows.Forms.Padding 4)
    $dg.ColumnHeadersHeight = 35
    $dg.ColumnHeadersBorderStyle = "Single"

    # Row Styling
    $dg.DefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.DefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.DefaultCellStyle.SelectionBackColor = [System.Drawing.ColorTranslator]::FromHtml("#007ACC")
    $dg.DefaultCellStyle.SelectionForeColor = "White"
    
    $f.Controls.Add($dg)
    $dg.BringToFront()

    # --- 4. Define Columns ---
    # Checkbox Column
    $colChk = New-Object System.Windows.Forms.DataGridViewCheckBoxColumn
    $colChk.HeaderText = " "
    $colChk.Width = 30; $colChk.Name = "Check"; $colChk.TrueValue = $true; $colChk.FalseValue = $false
    [void]$dg.Columns.Add($colChk)

    # Visible Columns
    [void]$dg.Columns.Add("Problem", "Problem"); $dg.Columns["Problem"].Width = 200
    [void]$dg.Columns.Add("Data", "Data (Path/Value)"); $dg.Columns["Data"].AutoSizeMode = "Fill"; $dg.Columns["Data"].FillWeight = 50
    [void]$dg.Columns.Add("Key", "Registry Key"); $dg.Columns["Key"].AutoSizeMode = "Fill"; $dg.Columns["Key"].FillWeight = 50
    
    # Hidden Columns (Data needed for fixing)
    [void]$dg.Columns.Add("FullPath", "FullPath"); $dg.Columns["FullPath"].Visible = $false
    [void]$dg.Columns.Add("ValueName", "ValueName"); $dg.Columns["ValueName"].Visible = $false
    [void]$dg.Columns.Add("Type", "Type"); $dg.Columns["Type"].Visible = $false

    # --- 5. Populate Data ---
    foreach ($item in $ScanResults) {
        $row = $dg.Rows.Add()
        $dg.Rows[$row].Cells["Check"].Value = $true
        $dg.Rows[$row].Cells["Problem"].Value = $item.Problem
        $dg.Rows[$row].Cells["Data"].Value = $item.Data
        $dg.Rows[$row].Cells["Key"].Value = $item.DisplayKey
        $dg.Rows[$row].Cells["FullPath"].Value = $item.RegPath
        $dg.Rows[$row].Cells["ValueName"].Value = $item.ValueName
        $dg.Rows[$row].Cells["Type"].Value = $item.Type
    }

    # --- 6. Footer Panel & Buttons ---
    $pnlBot = New-Object System.Windows.Forms.Panel
    $pnlBot.Dock = "Bottom"; $pnlBot.Height = 60
    $pnlBot.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.Controls.Add($pnlBot)

    $btnFix = New-Object System.Windows.Forms.Button
    $btnFix.Text = "Fix Selected Issues..."
    $btnFix.Width = 200; $btnFix.Height = 35; $btnFix.Top = 12; $btnFix.Left = 860
    $btnFix.Anchor = "Right, Bottom"
    $btnFix.FlatStyle = "Flat"; $btnFix.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#007ACC"); $btnFix.ForeColor = "White"
    $pnlBot.Controls.Add($btnFix)

    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Close"
    $btnCancel.Width = 100; $btnCancel.Height = 35; $btnCancel.Top = 12; $btnCancel.Left = 20
    $btnCancel.FlatStyle = "Flat"; $btnCancel.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#333333"); $btnCancel.ForeColor = "White"
    $btnCancel.Add_Click({ $f.Close() })
    $pnlBot.Controls.Add($btnCancel)

    # --- 7. Fix Button Logic ---
    $btnFix.Add_Click({
        $toFix = @()
        foreach ($row in $dg.Rows) {
            if ($row.Cells["Check"].Value -eq $true) {
                $toFix += [PSCustomObject]@{
                    RegPath    = $row.Cells["FullPath"].Value
                    ValueName  = $row.Cells["ValueName"].Value
                    Type       = $row.Cells["Type"].Value
                    DisplayKey = $row.Cells["Key"].Value
                }
            }
        }

        if ($toFix.Count -eq 0) {
            [System.Windows.Forms.MessageBox]::Show("No issues selected.", "Registry Cleaner", "OK", "Information") | Out-Null
            return
        }

        # Return results to the main controller
        $f.Tag = $toFix
        $f.DialogResult = "OK"
        $f.Close()
    })

    [void]$f.ShowDialog()
    return $f.Tag
}
# =========================================================
# 1. GLOBAL REGISTRY HELPERS
# (Place these ABOVE Invoke-RegistryTask)
# =========================================================

function Test-PathExists {
    param($Path)
    if ([string]::IsNullOrWhiteSpace($Path)) { return $false }
    if ($Path -match "%.*%" -or $Path -match "\$\(.*\)") { return $true }
    if (Test-Path -Path $Path) { return $true }
    if ($Path -match "(?i)System32") {
        $nativePath = $Path -replace "(?i)System32", "Sysnative"
        if (Test-Path -Path $nativePath) { return $true }
    }
    return $false
}

function Remove-RegKeyForced {
    param($Path, $IsKey, $ValName)
    if ($Path -match "^HKEY_CLASSES_ROOT") { $Path = $Path -replace "^HKEY_CLASSES_ROOT", "HKCR:" }
    if ($Path -match "^HKEY_LOCAL_MACHINE") { $Path = $Path -replace "^HKEY_LOCAL_MACHINE", "HKLM:" }
    if ($Path -match "^HKEY_CURRENT_USER") { $Path = $Path -replace "^HKEY_CURRENT_USER", "HKCU:" }

    $realPaths = @()
    if ($Path -match "^HKLM" -or $Path -match "^HKCU") { $realPaths += $Path }
    elseif ($Path -match "^HKCR:\\(?<SubPath>.*)") {
        $sub = $Matches.SubPath
        if (Test-Path "HKLM:\SOFTWARE\Classes\$sub") { $realPaths += "HKLM:\SOFTWARE\Classes\$sub" }
        if (Test-Path "HKLM:\SOFTWARE\WOW6432Node\Classes\$sub") { $realPaths += "HKLM:\SOFTWARE\WOW6432Node\Classes\$sub" }
        if (Test-Path "HKCU:\Software\Classes\$sub") { $realPaths += "HKCU:\Software\Classes\$sub" }
    }
    if ($realPaths.Count -eq 0) { $realPaths += $Path }

    $globalSuccess = $true
    foreach ($targetPath in $realPaths) {
        try {
            if ($IsKey) { Remove-Item -Path $targetPath -Recurse -Force -ErrorAction Stop }
            else { Remove-ItemProperty -Path $targetPath -Name $ValName -ErrorAction Stop }
            continue
        } catch {}
        try {
            $sid = New-Object System.Security.Principal.SecurityIdentifier([System.Security.Principal.WellKnownSidType]::BuiltinAdministratorsSid, $null)
            $adminUser = $sid.Translate([System.Security.Principal.NTAccount])
            $rule = New-Object System.Security.AccessControl.RegistryAccessRule($adminUser, "FullControl", "ContainerInherit,ObjectInherit", "None", "Allow")
            $UnlockItem = { 
                param($p) 
                try { 
                    $acl=Get-Acl $p; $acl.SetOwner($adminUser); Set-Acl $p $acl -ErrorAction SilentlyContinue; 
                    $acl=Get-Acl $p; $acl.SetAccessRule($rule); Set-Acl $p $acl -ErrorAction SilentlyContinue 
                } catch {} 
            }
            if ($IsKey) { $children = Get-ChildItem -Path $targetPath -Recurse -ErrorAction SilentlyContinue; foreach ($c in $children) { & $UnlockItem -p $c.PSPath } }
            & $UnlockItem -p $targetPath
            if ($IsKey) { Remove-Item -Path $targetPath -Recurse -Force -ErrorAction Stop }
            else { Remove-ItemProperty -Path $targetPath -Name $ValName -ErrorAction Stop }
        } catch { $globalSuccess = $false }
    }
    return $globalSuccess
}

function Backup-RegKey {
    param($ItemObj, $FilePath)
    $path = $ItemObj.RegPath; $targetValue = $ItemObj.ValueName; $type = $ItemObj.Type
    if ([string]::IsNullOrWhiteSpace($path)) { return }
    $regKeyPath = $path -replace "^HKLM:?\\", "HKEY_LOCAL_MACHINE\" -replace "^HKCU:?\\", "HKEY_CURRENT_USER\" -replace "^HKCR:?\\", "HKEY_CLASSES_ROOT\"
    $sb = [System.Text.StringBuilder]::new(); [void]$sb.AppendLine("[$regKeyPath]")
    try {
        if ($type -eq "Value") {
            $val = Get-ItemProperty -Path $path -Name $targetValue -ErrorAction SilentlyContinue
            if ($val) { 
                $vData = $val.$targetValue
                if ($vData -is [string]) { $vData = '"' + ($vData -replace '\\', '\\' -replace '"', '\"') + '"' } 
                elseif ($vData -is [int]) { $vData = "dword:{0:x8}" -f $vData }
                [void]$sb.AppendLine("`"$targetValue`"=$vData") 
            }
        }
        [void]$sb.AppendLine(""); Add-Content -Path $FilePath -Value $sb.ToString() -Encoding Unicode
    } catch {}
}

function Show-SafetyDialog {
    param($Count)
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Safety Pre-Check"; $f.Size = "450, 320"; $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32); $f.ForeColor = "White"
    $f.FormBorderStyle = "FixedDialog"; $f.ControlBox = $false
    
    $lbl = New-Object System.Windows.Forms.Label
    $lbl.Location = "20, 20"; $lbl.Size = "400, 80"; $lbl.Font = New-Object System.Drawing.Font("Segoe UI", 10)
    $lbl.Text = "You are about to force-delete $Count invalid registry keys.`n`nThese keys are locked by the system. Deleting them is generally safe for cleanup, but carries a small risk."
    $f.Controls.Add($lbl)
    
    $b1 = New-Object System.Windows.Forms.Button; $b1.Text = "Create Restore Point && Force Clean"; $b1.DialogResult = "Yes"; $b1.Location = "50, 110"; $b1.Size = "340, 45"; $b1.BackColor = "SeaGreen"; $b1.ForeColor = "White"; $b1.FlatStyle = "Flat"; $f.Controls.Add($b1)
    $b2 = New-Object System.Windows.Forms.Button; $b2.Text = "Force Clean (No Backup)"; $b2.DialogResult = "No"; $b2.Location = "50, 165"; $b2.Size = "340, 40"; $b2.BackColor = "IndianRed"; $b2.ForeColor = "White"; $b2.FlatStyle = "Flat"; $f.Controls.Add($b2)
    $b3 = New-Object System.Windows.Forms.Button; $b3.Text = "Cancel"; $b3.DialogResult = "Cancel"; $b3.Location = "50, 220"; $b3.Size = "340, 40"; $b3.BackColor = "DimGray"; $b3.ForeColor = "White"; $b3.FlatStyle = "Flat"; $f.Controls.Add($b3)
    
    return $f.ShowDialog()
}

# =========================================================
# 2. INVOKE-REGISTRYTASK (Full Function)
# =========================================================

function Invoke-RegistryTask {
    param([string]$Action)

    $bkDir = Join-Path (Get-DataPath) "RegistryBackups"
    if (-not (Test-Path $bkDir)) { New-Item -Path $bkDir -ItemType Directory | Out-Null }

    # --- PRIVILEGE BOOSTER ---
    # Only adding the type if it doesn't exist to prevent errors on re-run
    if (-not ([System.Management.Automation.PSTypeName]'Win32.TokenManipulator').Type) {
        Add-Type -TypeDefinition @"
        using System;
        using System.Runtime.InteropServices;
        public class Win32 {
            public class TokenManipulator {
                [DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
                internal static extern bool AdjustTokenPrivileges(IntPtr htok, bool disall, ref TokPriv1Luid newst, int len, IntPtr prev, IntPtr relen);
                [DllImport("kernel32.dll", ExactSpelling = true)]
                internal static extern IntPtr GetCurrentProcess();
                [DllImport("advapi32.dll", ExactSpelling = true, SetLastError = true)]
                internal static extern bool OpenProcessToken(IntPtr h, int acc, ref IntPtr phtok);
                [DllImport("advapi32.dll", SetLastError = true)]
                internal static extern bool LookupPrivilegeValue(string host, string name, ref long pluid);
                [StructLayout(LayoutKind.Sequential, Pack = 1)]
                internal struct TokPriv1Luid { public int Count; public long Luid; public int Attr; }
                internal const int SE_PRIVILEGE_ENABLED = 0x00000002;
                internal const int TOKEN_ADJUST_PRIVILEGES = 0x00000020;
                internal const int TOKEN_QUERY = 0x00000008;
                public static bool EnablePrivilege(string privilege) {
                    try {
                        IntPtr htok = IntPtr.Zero;
                        if (!OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, ref htok)) return false;
                        TokPriv1Luid tp; tp.Count = 1; tp.Attr = SE_PRIVILEGE_ENABLED; tp.Luid = 0;
                        if (!LookupPrivilegeValue(null, privilege, ref tp.Luid)) return false;
                        if (!AdjustTokenPrivileges(htok, false, ref tp, 0, IntPtr.Zero, IntPtr.Zero)) return false;
                        return true;
                    } catch { return false; }
                }
            }
        }
"@
    }
    try {
        [Win32.TokenManipulator]::EnablePrivilege("SeTakeOwnershipPrivilege") | Out-Null
        [Win32.TokenManipulator]::EnablePrivilege("SeRestorePrivilege") | Out-Null
    } catch {}

    # --- MAIN SCAN LOGIC ---
    if ($Action -eq "DeepClean") {
        $selectedScans = Show-RegScanSelection
        if (-not $selectedScans) { return }

        # Progress UI
        $pForm = New-Object System.Windows.Forms.Form; $pForm.Text="Scanning Registry"; $pForm.Size="500,120"; $pForm.StartPosition="CenterScreen"; $pForm.ControlBox=$false
        $pForm.BackColor=[System.Drawing.Color]::FromArgb(30,30,30); $pForm.ForeColor="White"
        $pLabel = New-Object System.Windows.Forms.Label; $pLabel.Location="20,15"; $pLabel.Size="460,20"; $pLabel.Text="Initializing Background Scan..."; $pForm.Controls.Add($pLabel)
        $pBar = New-Object System.Windows.Forms.ProgressBar; $pBar.Location="20,45"; $pBar.Size="440,20"; $pForm.Controls.Add($pBar)
        $pForm.Show()

        # Shared Data for Thread
        $syncHash = [hashtable]::Synchronized(@{
            Findings = [System.Collections.ArrayList]::new()
            Status = "Starting..."
            Progress = 0
            IsCompleted = $false
            Error = $null
        })

        # --- RUNSPACE CONFIGURATION ---
        $iss = [System.Management.Automation.Runspaces.InitialSessionState]::CreateDefault()
        $rs = [System.Management.Automation.Runspaces.RunspaceFactory]::CreateRunspace($iss)
        # STA Mode is critical for Registry (HKCR) scans to work correctly
        $rs.ApartmentState = "STA"
        $rs.ThreadOptions = "ReuseThread"
        $rs.Open()
        
        $rs.SessionStateProxy.SetVariable("SyncHash", $syncHash)
        $rs.SessionStateProxy.SetVariable("SelectedScans", $selectedScans)

        $ps = [PowerShell]::Create()
        $ps.Runspace = $rs
        
        # --- SCANNING SCRIPT BLOCK ---
        [void]$ps.AddScript({
            Import-Module Microsoft.PowerShell.Management
            Import-Module Microsoft.PowerShell.Security
            $SelectedScans = @($SelectedScans)

            # Internal Helper: Check if path exists
            function Test-PathExists {
                param($Path)
                if ([string]::IsNullOrWhiteSpace($Path)) { return $false }
                if ($Path -match "%.*%" -or $Path -match "\$\(.*\)") { return $true }
                if (Test-Path -Path $Path) { return $true }
                if ($Path -match "(?i)System32") {
                    $nativePath = $Path -replace "(?i)System32", "Sysnative"
                    if (Test-Path -Path $nativePath) { return $true }
                }
                return $false
            }

            # Internal Helper: Whitelist
            function Test-IsWhitelisted {
                param($Path)
                if ([string]::IsNullOrWhiteSpace($Path)) { return $false }
                $SafeList = @("TetheringSettingHandler", "CrossDevice", "Windows.Media.Protection", "psmachine", "WebView2", "System.Data.dll", "System.EnterpriseServices", "rundll32", "explorer.exe", "svchost", "dllhost", "wmiprvse", "mmgaserver", "pickerhost", "castsrv", "uihelper", "backgroundtaskhost", "smartscreen", "runtimebroker", "mousocoreworker", "spatialaudiolicensesrv", "speechruntime", "mstsc.exe", "searchprotocolhost", "AppX", "WindowsApps", "UIEOrchestrator", "control.exe", "sdclt.exe", "provtool.exe", "perfmon", "Diagnostic.Perfmon")
                foreach ($safe in $SafeList) { if ($Path -match "(?i)$safe") { return $true } }
                return $false
            }

            # Internal Helper: Clean Path Strings
            function Get-RealExePath {
                param($RawString)
                if ([string]::IsNullOrWhiteSpace($RawString)) { return $null }
                $clean = $RawString.Trim()
                if ($clean -match "^(.*?),\s*-?\d+$") { $clean = $matches[1].Trim() }
                if ($clean.StartsWith('"')) {
                    $endQuote = $clean.IndexOf('"', 1)
                    if ($endQuote -gt 1) {
                        $quotedPart = $clean.Substring(1, $endQuote - 1)
                        if (Test-PathExists $quotedPart) { return $quotedPart }
                        $clean = $quotedPart 
                    }
                }
                if (Test-PathExists $clean) { return $clean }
                if ($clean.Contains(" ")) {
                    $parts = $clean -split " "
                    $candidate = $parts[0]
                    $Check = { param($p) if (Test-PathExists $p) { return $true }; if (Test-PathExists "$p.exe") { return $true }; return $false }
                    if (& $Check $candidate) { return $candidate }
                    for ($i = 1; $i -lt $parts.Count; $i++) {
                        $candidate += " " + $parts[$i]
                        if (& $Check $candidate) { return $candidate }
                    }
                }
                return $clean
            }

            try {
                $categoryWeight = 100.0 / ($SelectedScans.Count)
                $currentBaseProgress = 0.0

                # Tick Helper
                $Tick = {
                    param($DetailText, $CurrentIndex, $TotalEstimated)
                    if ($CurrentIndex % 50 -eq 0) {
                        $SyncHash.Status = $DetailText
                        $fraction = [math]::Min(($CurrentIndex / $TotalEstimated), 1.0)
                        $realVal = $currentBaseProgress + ($fraction * $categoryWeight)
                        $SyncHash.Progress = [int]$realVal
                    }
                }
                $EndCategory = {
                    $currentBaseProgress += $categoryWeight
                    $SyncHash.Progress = [int]$currentBaseProgress
                }

                # 1. ACTIVEX & COM
                if ($SelectedScans -contains "ActiveX") { 
                    $SyncHash.Status = "Scanning ActiveX/COM..."
                    $root = [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::ClassesRoot, [Microsoft.Win32.RegistryView]::Default)
                    $clsidKey = $root.OpenSubKey("CLSID", $false)
                    if ($clsidKey) { 
                        $subKeys = $clsidKey.GetSubKeyNames(); $total = $subKeys.Count; $i = 0
                        foreach ($id in $subKeys) { 
                            $i++; & $Tick "Scanning CLSID: $id" $i $total
                            try { 
                                $sub = $clsidKey.OpenSubKey("$id\InProcServer32", $false)
                                if ($sub) { 
                                    $dll = $sub.GetValue($null)
                                    if ($dll -and $dll -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $dll)) { 
                                        $cleanDll = Get-RealExePath $dll
                                        if (-not (Test-PathExists $cleanDll)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="ActiveX Issue"; Data=$cleanDll; DisplayKey=$id; RegPath="HKCR:\CLSID\$id\InProcServer32"; ValueName=$null; Type="Key" }) }
                                    }
                                    $sub.Close() 
                                }
                                $sub2 = $clsidKey.OpenSubKey("$id\LocalServer32", $false)
                                if ($sub2) {
                                    $exe = $sub2.GetValue($null)
                                    if ($exe -and $exe -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $exe)) {
                                        $cleanExe = Get-RealExePath $exe
                                        if (-not (Test-PathExists $cleanExe)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="ActiveX Issue"; Data=$cleanExe; DisplayKey=$id; RegPath="HKCR:\CLSID\$id\LocalServer32"; ValueName=$null; Type="Key" }) }
                                    }
                                    $sub2.Close()
                                }
                            } catch {} 
                        }
                        $clsidKey.Close() 
                    }
                    & $EndCategory
                }

                # 2. FILE EXTENSIONS
                if ($SelectedScans -contains "Ext") { 
                    $SyncHash.Status = "Scanning File Extensions..."
                    $root = [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::ClassesRoot, [Microsoft.Win32.RegistryView]::Default)
                    $names = $root.GetSubKeyNames(); $total = $names.Count; $i = 0
                    foreach ($ext in $names) { 
                        $i++; & $Tick "Scanning Ext: $ext" $i $total
                        if($ext.StartsWith(".")) { 
                            try { 
                                $sub=$root.OpenSubKey($ext)
                                if($sub.SubKeyCount -eq 0 -and $null -eq $sub.GetValue($null)){ 
                                    [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Unused Extension"; Data=$ext; DisplayKey=$ext; RegPath="HKCR:\$ext"; ValueName=$null; Type="Key" }) 
                                }; $sub.Close() 
                            } catch {} 
                        } 
                    }
                    & $EndCategory
                }

                # 3. USER FILE EXTS
                if ($SelectedScans -contains "FileExts") {
                    $SyncHash.Status = "Scanning User File Associations..."
                    $path = "Software\Microsoft\Windows\CurrentVersion\Explorer\FileExts"
                    $root = [Microsoft.Win32.Registry]::CurrentUser.OpenSubKey($path)
                    if ($root) {
                        $names = $root.GetSubKeyNames(); $total = $names.Count; $i = 0
                        foreach ($ext in $names) {
                            $i++; & $Tick "Scanning UserExt: $ext" $i $total
                            $owl = $root.OpenSubKey("$ext\OpenWithList")
                            if ($owl) {
                                foreach ($valName in $owl.GetValueNames()) {
                                    if ($valName -match "^[a-z]$") { 
                                        $val = $owl.GetValue($valName)
                                        if ($val -match '^[a-zA-Z]:\\') {
                                            $cleanPath = Get-RealExePath $val
                                            if (-not (Test-PathExists $cleanPath)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Invalid FileExt MRU"; Data=$cleanPath; DisplayKey=$ext; RegPath="HKCU:\$path\$ext\OpenWithList"; ValueName=$valName; Type="Value" }) }
                                        }
                                    }
                                }
                                $owl.Close()
                            }
                        }
                        $root.Close()
                    }
                    & $EndCategory
                }

                # 4. APP PATHS
                if ($SelectedScans -contains "AppPaths") {
                    $SyncHash.Status = "Scanning App Paths..."
                    $key = "SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths"
                    $root = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($key)
                    if ($root) {
                        $names = $root.GetSubKeyNames(); $total = $names.Count; $i = 0
                        foreach ($app in $names) {
                            $i++; & $Tick "Scanning AppPath: $app" $i $total
                            try {
                                $sub = $root.OpenSubKey($app)
                                $path = $sub.GetValue($null)
                                if ($path -and $path -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $path)) {
                                    $clean = Get-RealExePath $path
                                    if (-not (Test-PathExists $clean)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Missing App Path"; Data=$clean; DisplayKey=$app; RegPath="HKLM:\$key\$app"; ValueName=$null; Type="Key" }) }
                                }
                                $sub.Close()
                            } catch {}
                        }
                        $root.Close()
                    }
                    & $EndCategory
                }

                # 5. APPLICATIONS & PROGIDs
                if ($SelectedScans -contains "Apps" -or $SelectedScans -contains "ProgIDs") {
                    $SyncHash.Status = "Scanning Apps & ProgIDs..."
                    $searchRoots = @()
                    if ($SelectedScans -contains "Apps") { 
                        $cr = [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::ClassesRoot, [Microsoft.Win32.RegistryView]::Default)
                        $searchRoots += $cr.OpenSubKey("Applications") 
                    }
                    if ($SelectedScans -contains "ProgIDs") { 
                        $searchRoots += [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::ClassesRoot, [Microsoft.Win32.RegistryView]::Default)
                    }

                    foreach ($root in $searchRoots) {
                        if ($root) {
                            $names = $root.GetSubKeyNames(); $total = $names.Count; $i = 0
                            foreach ($app in $names) {
                                $i++; & $Tick "Scanning App: $app" $i $total
                                try {
                                    $appKey = $root.OpenSubKey($app); $shellKey = $appKey.OpenSubKey("shell")
                                    if ($shellKey) {
                                        foreach ($verb in $shellKey.GetSubKeyNames()) {
                                            try {
                                                $cmdKey = $shellKey.OpenSubKey("$verb\command")
                                                if ($cmdKey) {
                                                    $cmd = $cmdKey.GetValue($null)
                                                    if ($cmd) {
                                                        $clean = Get-RealExePath $cmd
                                                        if ($clean -and $clean -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $clean)) {
                                                            if (-not (Test-PathExists $clean) -and $clean -notmatch "%1") {
                                                                [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Invalid App Command ($verb)"; Data=$clean; DisplayKey=$app; RegPath="$($root.Name)\$app\shell\$verb\command"; ValueName=$null; Type="Key" })
                                                            }
                                                        }
                                                    }
                                                    $cmdKey.Close()
                                                }
                                            } catch {}
                                        }
                                        $shellKey.Close()
                                    }
                                    $appKey.Close()
                                } catch {}
                            }
                        }
                    }
                    & $EndCategory
                }

                # 6. UNINSTALLERS
                if ($SelectedScans -contains "Uninstall") {
                    $SyncHash.Status = "Scanning Uninstallers..."
                    $paths = @("SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall", "SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall")
                    $hives = @([Microsoft.Win32.Registry]::LocalMachine, [Microsoft.Win32.Registry]::CurrentUser)
                    $total = 500; $i = 0
                    foreach ($h in $hives) {
                        foreach ($p in $paths) {
                            try {
                                $rk = $h.OpenSubKey($p)
                                if ($rk) {
                                    foreach ($subName in $rk.GetSubKeyNames()) {
                                        $i++; & $Tick "Scanning Uninstaller: $subName" $i $total
                                        $sub = $rk.OpenSubKey($subName)
                                        $uString = $sub.GetValue("UninstallString")
                                        if ($uString -and $uString -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $uString)) {
                                            if ($uString -notmatch "(?i)MsiExec.exe") {
                                                $clean = Get-RealExePath $uString
                                                if (-not (Test-PathExists $clean)) {
                                                    $rootName = if ($h.Name -match "LocalMachine") { "HKLM" } else { "HKCU" }
                                                    [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Missing Uninstaller"; Data=$clean; DisplayKey=$subName; RegPath="$rootName`:\$p\$subName"; ValueName=$null; Type="Key" })
                                                }
                                            }
                                        }
                                        $sub.Close()
                                    }
                                    $rk.Close()
                                }
                            } catch {}
                        }
                    }
                    & $EndCategory
                }

                # 7. MUI CACHE
                if ($SelectedScans -contains "MuiCache") {
                    $SyncHash.Status = "Scanning MuiCache..."
                    $key = "Software\Classes\Local Settings\Software\Microsoft\Windows\Shell\MuiCache"
                    $root = [Microsoft.Win32.Registry]::CurrentUser.OpenSubKey($key)
                    if ($root) {
                        $names = $root.GetValueNames(); $total = $names.Count; $i = 0
                        foreach ($valName in $names) {
                            $i++; & $Tick "Scanning MuiCache: $valName" $i $total
                            $cleanPath = $valName -replace '\.(FriendlyAppName|ApplicationCompany)$', ''
                            if ($cleanPath -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $cleanPath)) {
                                if (-not (Test-PathExists $cleanPath)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Obsolete MuiCache"; Data=$cleanPath; DisplayKey="MuiCache"; RegPath="HKCU:\$key"; ValueName=$valName; Type="Value" }) }
                            }
                        }
                        $root.Close()
                    }
                    & $EndCategory
                }

                # 8. APPCOMPAT FLAGS
                if ($SelectedScans -contains "AppCompat") {
                    $SyncHash.Status = "Scanning Compatibility Store..."
                    $key = "Software\Microsoft\Windows NT\CurrentVersion\AppCompatFlags\Compatibility Assistant\Store"
                    $root = [Microsoft.Win32.Registry]::CurrentUser.OpenSubKey($key)
                    if ($root) {
                        $names = $root.GetValueNames(); $total = $names.Count; $i = 0
                        foreach ($valName in $names) {
                            $i++; & $Tick "Scanning AppCompat: $valName" $i $total
                            if ($valName -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $valName)) {
                                if (-not (Test-PathExists $valName)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Obsolete Compatibility Ref"; Data=$valName; DisplayKey="AppCompat"; RegPath="HKCU:\$key"; ValueName=$valName; Type="Value" }) }
                            }
                        }
                        $root.Close()
                    }
                    & $EndCategory
                }

                # 9. FIREWALL
                if ($SelectedScans -contains "Firewall") {
                    $SyncHash.Status = "Scanning Firewall Rules..."
                    $key = "SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\FirewallRules"
                    $root = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($key)
                    if ($root) {
                        $names = $root.GetValueNames(); $total = $names.Count; $i = 0
                        foreach ($valName in $names) {
                            $i++; & $Tick "Scanning Firewall: $valName" $i $total
                            $data = $root.GetValue($valName)
                            if ($data -match "App=([^|]+)") {
                                $appPath = $matches[1]; $expanded = [Environment]::ExpandEnvironmentVariables($appPath)
                                if ($expanded -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $expanded)) {
                                    if (-not (Test-PathExists $expanded)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Invalid Firewall Rule"; Data=$expanded; DisplayKey=$valName; RegPath="HKLM:\$key"; ValueName=$valName; Type="Value" }) }
                                }
                            }
                        }
                        $root.Close()
                    }
                    & $EndCategory
                }

                # 10. SERVICES
                if ($SelectedScans -contains "Services") {
                    $SyncHash.Status = "Scanning Services..."
                    $key = "SYSTEM\CurrentControlSet\Services"
                    $root = [Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($key)
                    if ($root) {
                        $names = $root.GetSubKeyNames(); $total = $names.Count; $i = 0
                        foreach ($svc in $names) {
                            $i++; & $Tick "Scanning Service: $svc" $i $total
                            try {
                                $sub = $root.OpenSubKey($svc)
                                $img = $sub.GetValue("ImagePath")
                                if ($img -and $img -match '^[a-zA-Z]:\\' -and $img -notmatch "\\drivers\\") {
                                    $clean = Get-RealExePath $img
                                    if (-not (Test-PathExists $clean)) { [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Missing Service Binary"; Data=$clean; DisplayKey=$svc; RegPath="HKLM:\$key\$svc"; ValueName="ImagePath"; Type="Value" }) }
                                }
                                $sub.Close()
                            } catch {}
                        }
                        $root.Close()
                    }
                    & $EndCategory
                }

                # 11. TYPE LIBRARIES
                if ($SelectedScans -contains "TypeLib") {
                    $SyncHash.Status = "Scanning Type Libraries..."
                    $root = [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::ClassesRoot, [Microsoft.Win32.RegistryView]::Default)
                    $tlKey = $root.OpenSubKey("TypeLib", $false)
                    if ($tlKey) {
                        $names = $tlKey.GetSubKeyNames(); $total = $names.Count; $i = 0
                        foreach ($guid in $names) {
                            $i++; & $Tick "Scanning TypeLib: $guid" $i $total
                            try {
                                $verKey = $tlKey.OpenSubKey($guid)
                                if ($verKey) {
                                    foreach ($ver in $verKey.GetSubKeyNames()) {
                                        $numKey = $verKey.OpenSubKey($ver)
                                        $helpDir = $numKey.GetValue("HELPDIR")
                                        if ($helpDir -and $helpDir -match '^[a-zA-Z]:\\' -and -not (Test-PathExists $helpDir)) {
                                            [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Missing HelpDir"; Data=$helpDir; DisplayKey="$guid"; RegPath="HKCR:\TypeLib\$guid\$ver"; ValueName="HELPDIR"; Type="Value" }) 
                                        }
                                        $numKey.Close()
                                    }
                                    $verKey.Close()
                                }
                            } catch {}
                        }
                        $tlKey.Close()
                    }
                    & $EndCategory
                }

                # 12. DEFAULT ICONS
                if ($SelectedScans -contains "Icons") {
                    $SyncHash.Status = "Scanning Default Icons..."
                    $root = [Microsoft.Win32.RegistryKey]::OpenBaseKey([Microsoft.Win32.RegistryHive]::ClassesRoot, [Microsoft.Win32.RegistryView]::Default)
                    $names = $root.GetSubKeyNames(); $total = $names.Count; $i = 0
                    foreach ($ext in $names) {
                        $i++; & $Tick "Scanning Icon: $ext" $i $total
                        try {
                            $iconKey = $root.OpenSubKey("$ext\DefaultIcon")
                            if ($iconKey) {
                                $val = $iconKey.GetValue($null)
                                if ($val) {
                                    $cleanPath = Get-RealExePath $val
                                    if ($cleanPath -match '^[a-zA-Z]:\\' -and -not (Test-IsWhitelisted $cleanPath) -and -not (Test-PathExists $cleanPath) -and $cleanPath -notmatch "%1") {
                                        [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Invalid Default Icon"; Data=$cleanPath; DisplayKey=$ext; RegPath="HKCR:\$ext\DefaultIcon"; ValueName=$null; Type="Key" })
                                    }
                                }
                                $iconKey.Close()
                            }
                        } catch {}
                    }
                    & $EndCategory
                }

                # 13. SHARED DLLs
                if ($SelectedScans -contains "SharedDLLs") { 
                    $SyncHash.Status = "Scanning Shared DLLs..."
                    $keys = @("SOFTWARE\Microsoft\Windows\CurrentVersion\SharedDlls", "SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\SharedDlls")
                    foreach ($k in $keys) {
                        $rk=[Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($k)
                        if($rk){ 
                            $names = $rk.GetValueNames(); $total = $names.Count; $i = 0
                            foreach($val in $names){ 
                                $i++; & $Tick "Scanning DLL: $val" $i $total
                                if($val -match '^[a-zA-Z]:\\' -and -not(Test-PathExists $val)){ [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Missing Shared Ref"; Data=$val; DisplayKey="SharedDlls"; RegPath="HKLM:\$k"; ValueName=$val; Type="Value" }) } 
                            }; $rk.Close() 
                        }
                    }
                    & $EndCategory
                }

                # 14. STARTUP
                if ($SelectedScans -contains "Startup") { 
                    $SyncHash.Status = "Scanning Startup Items..."
                    $paths = @("HKCU:\Software\Microsoft\Windows\CurrentVersion\Run", "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Run", "HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Run")
                    foreach ($p in $paths) { 
                        if(Test-Path $p){ 
                            $props = Get-ItemProperty $p
                            $names = $props.PSObject.Properties.Name; $total = $names.Count; $i = 0
                            foreach($n in $names){ 
                                $i++; & $Tick "Scanning Startup: $n" $i $total
                                $v=$props.$n
                                if($v -is [string] -and $v -match '^[a-zA-Z]:\\'){ 
                                    $cleanExe = Get-RealExePath $v
                                    if(-not (Test-PathExists $cleanExe)){ [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Broken Startup"; Data=$cleanExe; DisplayKey=$n; RegPath=$p; ValueName=$n; Type="Value" }) } 
                                } 
                            } 
                        } 
                    } 
                    & $EndCategory
                }

                # 15. INSTALLER FOLDERS
                if ($SelectedScans -contains "Installer") { 
                    $SyncHash.Status = "Scanning Installer Folders..."
                    $k="SOFTWARE\Microsoft\Windows\CurrentVersion\Installer\Folders"
                    $rk=[Microsoft.Win32.Registry]::LocalMachine.OpenSubKey($k)
                    if($rk){ 
                        $names = $rk.GetValueNames(); $total = $names.Count; $i = 0
                        foreach($val in $names){ 
                            $i++; & $Tick "Scanning Installer: $val" $i $total
                            if($val -match '^[a-zA-Z]:\\' -and -not(Test-PathExists $val)){ [void]$SyncHash.Findings.Add([PSCustomObject]@{ Problem="Missing Installer Folder"; Data=$val; DisplayKey="Installer"; RegPath="HKLM:\$k"; ValueName=$val; Type="Value" }) } 
                        }; $rk.Close() 
                    } 
                    & $EndCategory
                }

                $SyncHash.IsCompleted = $true
            } catch {
                $SyncHash.Error = $_.Exception.Message
                $SyncHash.IsCompleted = $true
            }
        })

        [void]$ps.BeginInvoke()
        
        # --- UI TIMER (Main Thread) ---
        $timer = New-Object System.Windows.Threading.DispatcherTimer
        $timer.Interval = [TimeSpan]::FromMilliseconds(100)
        
        $timer.Add_Tick({
            $pLabel.Text = $syncHash.Status
            $pBar.Value = $syncHash.Progress
            
            if ($syncHash.IsCompleted) {
                $timer.Stop()
                $pForm.Close()
                $ps.Dispose()
                $rs.Dispose()

                if ($syncHash.Error) { [System.Windows.Forms.MessageBox]::Show("Scan Error: $($syncHash.Error)", "Error", "OK", "Error"); return }

                $findings = $syncHash.Findings
                
                # --- RESULTS PROCESSING ---
                if ($findings.Count -eq 0) {
                    [System.Windows.Forms.MessageBox]::Show("Registry Cleaner: No issues found!", "Scan Complete", "OK", "Information") | Out-Null
                    return
                }
                
                $rawSelection = Show-RegistryCleaner -ScanResults ($findings | Select-Object *)
                
                $toDelete = @()
                if ($rawSelection) {
                    $toDelete = $rawSelection | Where-Object { $null -ne $_ -and $null -ne $_.RegPath }
                }

                if ($toDelete.Count -eq 0) { return }

                # --- SAFETY PROMPT ---
                $res = Show-SafetyDialog -Count $toDelete.Count
                if ($res -eq "Cancel") { return }
                if ($res -eq "Yes") {
                    Invoke-UiCommand { try { Checkpoint-Computer -Description "WMT DeepClean" -RestorePointType "MODIFY_SETTINGS" -ErrorAction Stop; "Restore Point created." } catch { "Restore Point failed (Disabled?). Continuing..." } } "Creating Restore Point..."
                }

                # --- EXECUTE FIX (Main Thread) ---
                Invoke-UiCommand {
                    param($toDelete, $bkDir) 
                    
                    $bkFile = Join-Path $bkDir ("DeepClean_Backup_{0}.reg" -f (Get-Date -Format "yyyyMMdd_HHmm"))
                    $fixed=0; $skipped=0; $backedUpKeys=@()
                    
                    Set-Content -Path $bkFile -Value "Windows Registry Editor Version 5.00`r`n`r`n" -Encoding Unicode

                    foreach ($item in $toDelete) {
                        $uid = "$($item.RegPath):$($item.ValueName)"
                        if ($uid -notin $backedUpKeys) { Backup-RegKey -ItemObj $item -FilePath $bkFile; $backedUpKeys += $uid }
                        
                        Write-GuiLog "Removing: $($item.DisplayKey)"
                        $isKey = ($item.Type -eq "Key")
                        $success = Remove-RegKeyForced -Path $item.RegPath -IsKey $isKey -ValName $item.ValueName
                        
                        if ($success) { $fixed++ } else { $skipped++ }
                    }
                    
                    $finalMsg = "Cleanup Complete.`n`nFixed: $fixed item(s)"
                    if ($skipped -gt 0) { $finalMsg += "`nSkipped: $skipped (System Protected)" }
                    $finalMsg += "`n`nBackup: $bkFile"
                    [System.Windows.Forms.MessageBox]::Show($finalMsg, "Result", "OK", "Information") | Out-Null
                    
                } "Deep Cleaning..." -ArgumentList $toDelete, $bkDir
            }
        }.GetNewClosure())
        
        $timer.Start()
    }
}

function Invoke-SSDTrim {
    Invoke-UiCommand {
        $ssds = Get-PhysicalDisk | Where-Object MediaType -eq 'SSD'
        if (-not $ssds) { Write-Output "No SSDs detected."; return }
        $log = Join-Path (Get-DataPath) ("SSD_OPTIMIZE_{0}.log" -f (Get-Date -Format "yyyy-MM-dd_HHmmss"))
        $out = @("SSD Optimize Log - $(Get-Date)")
        foreach ($ssd in $ssds) {
            $disk = Get-Disk | Where-Object { $_.FriendlyName -eq $ssd.FriendlyName }
            if ($disk) {
                # FIX: Use scriptblock for null check
                $vols = $disk | Get-Partition | Get-Volume | Where-Object { $null -ne $_.DriveLetter }
                foreach ($v in $vols) {
                    $out += "Optimizing $($v.DriveLetter):"
                    $out += Optimize-Volume -DriveLetter $v.DriveLetter -ReTrim -Verbose 4>&1
                }
            }
        }
        $out | Out-File -FilePath $log -Encoding UTF8
        Write-Output "SSD optimization complete. Log: $log"
    } "Running SSD Trim/ReTrim..."
}
function Show-BrokenShortcuts {
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Broken Shortcut Manager"
    $f.Size = "1100, 650"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $f.ForeColor = "White"

    # 1. SETUP GRID
    $dg = New-Object System.Windows.Forms.DataGridView
    $dg.Dock = "Top"; $dg.Height = 480
    $dg.BackgroundColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $dg.ForeColor = "Black"
    $dg.AutoSizeColumnsMode = "Fill"
    $dg.SelectionMode = "FullRowSelect"
    $dg.MultiSelect = $true
    $dg.ReadOnly = $true
    $dg.RowHeadersVisible = $false
    $dg.AllowUserToAddRows = $false
    $dg.BorderStyle = "None"
    
    $dg.EnableHeadersVisualStyles = $false
    $dg.ColumnHeadersDefaultCellStyle.BackColor = [System.Drawing.Color]::FromArgb(50, 50, 50)
    $dg.ColumnHeadersDefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.ColumnHeadersDefaultCellStyle.Padding = (New-Object System.Windows.Forms.Padding 4)
    $dg.ColumnHeadersHeight = 35
    $dg.DefaultCellStyle.BackColor = [System.Drawing.Color]::FromArgb(30, 30, 30)
    $dg.DefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.DefaultCellStyle.SelectionBackColor = [System.Drawing.Color]::FromArgb(0, 122, 204)
    $dg.DefaultCellStyle.SelectionForeColor = "White"

    $f.Controls.Add($dg)

    # 2. STATUS
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Text = "Initializing..."
    $lblStatus.AutoSize = $true
    $lblStatus.ForeColor = "Yellow"
    $lblStatus.Location = "20, 490"
    $f.Controls.Add($lblStatus)

    # 3. BUTTONS
    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Dock = "Bottom"; $pnl.Height = 80
    $pnl.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $f.Controls.Add($pnl)

    # --- CHANGED: Delete Button ---
    $btnDelete = New-Object System.Windows.Forms.Button
    $btnDelete.Text = "Delete Selected"
    $btnDelete.Location = "20, 20"; $btnDelete.Width = 150; $btnDelete.Height = 35
    $btnDelete.BackColor = "IndianRed"; $btnDelete.ForeColor = "White"; $btnDelete.FlatStyle = "Flat"
    $pnl.Controls.Add($btnDelete)

    $btnApply = New-Object System.Windows.Forms.Button
    $btnApply.Text = "Apply Fixes" # Renamed slightly to be clearer
    $btnApply.Location = "780, 20"; $btnApply.Width = 150; $btnApply.Height = 35
    $btnApply.BackColor = "SeaGreen"; $btnApply.ForeColor = "White"; $btnApply.FlatStyle = "Flat"
    $btnApply.DialogResult = [System.Windows.Forms.DialogResult]::OK
    $pnl.Controls.Add($btnApply)

    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Close"
    $btnCancel.Location = "950, 20"; $btnCancel.Width = 100; $btnCancel.Height = 35
    $btnCancel.BackColor = "DimGray"; $btnCancel.ForeColor = "White"; $btnCancel.FlatStyle = "Flat"
    $btnCancel.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
    $pnl.Controls.Add($btnCancel)
    
    $lblInfo = New-Object System.Windows.Forms.Label
    $lblInfo.Text = "Right-click for Deep Search or Manual Browse."
    $lblInfo.AutoSize = $true; $lblInfo.ForeColor = "Gray"
    $lblInfo.Location = "200, 30"
    $pnl.Controls.Add($lblInfo)

    # 4. CONTEXT MENU
    $ctx = New-Object System.Windows.Forms.ContextMenuStrip
    
    # --- Manual Browse ---
    $itemBrowse = $ctx.Items.Add("Browse for target...")
    $itemBrowse.Add_Click({
        if ($dg.SelectedRows.Count -eq 1) {
            $row = $dg.SelectedRows[0]
            $obj = $row.DataBoundItem
            $dlg = New-Object System.Windows.Forms.OpenFileDialog
            $dlg.Filter = "Executables (*.exe)|*.exe|All Files (*.*)|*.*"
            if ($dlg.ShowDialog() -eq "OK") {
                $obj.Action = "Fix"
                $obj.NewTarget = $dlg.FileName
                $obj.Details = "Manual fix: $($dlg.FileName)"
                $dg.Refresh()
                $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::LightGreen
            }
        }
    })

    # --- Deep Search ---
    $itemDeep = $ctx.Items.Add("Deep Search (Slow - Scan All Drives)")
    $itemDeep.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
    $itemDeep.ForeColor = [System.Drawing.Color]::DarkBlue
    
    $itemDeep.Add_Click({
        if ($dg.SelectedRows.Count -ne 1) { return }
        $row = $dg.SelectedRows[0]
        $obj = $row.DataBoundItem
        
        $searchName = $null
        $shell = New-Object -ComObject WScript.Shell
        try {
            $sc = $shell.CreateShortcut($obj.FullPath)
            if ($sc.TargetPath) {
                $searchName = Split-Path $sc.TargetPath -Leaf
            }
        } catch {}

        if (-not $searchName) {
            $base = [System.IO.Path]::GetFileNameWithoutExtension($obj.Shortcut)
            $searchName = "$base.exe"
        }

        $warnMsg = "This will scan ALL local hard drives for:`n`n'$searchName'`n`nDepending on your drive size, this can take 5-10+ minutes.`nDuring this time, the application may appear frozen.`n`nDo you want to continue?"
        $res = [System.Windows.Forms.MessageBox]::Show($warnMsg, "Deep Search Warning", [System.Windows.Forms.MessageBoxButtons]::YesNo, [System.Windows.Forms.MessageBoxIcon]::Warning)
        
        if ($res -eq "Yes") {
            [System.Windows.Forms.Cursor]::Current = [System.Windows.Forms.Cursors]::WaitCursor
            $lblStatus.Text = "Deep Searching for '$searchName'..."
            $f.Update()

            $foundPath = $null
            $drives = [System.IO.DriveInfo]::GetDrives() | Where-Object { $_.DriveType -eq 'Fixed' }
            
            foreach ($d in $drives) {
                if ($foundPath) { break }
                $root = $d.RootDirectory.FullName
                $lblStatus.Text = "Scanning drive $root for '$searchName'..."
                $f.Update()
                try {
                    $match = Get-ChildItem -Path $root -Filter $searchName -Recurse -ErrorAction SilentlyContinue -Force | Select-Object -First 1
                    if ($match) { $foundPath = $match.FullName }
                } catch {}
            }

            if ($foundPath) {
                $obj.Action = "Fix"
                $obj.NewTarget = $foundPath
                $obj.Details = "Deep Search: $foundPath"
                $dg.Refresh()
                $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::LightGreen
                $lblStatus.Text = "Found: $foundPath"
                [System.Windows.Forms.MessageBox]::Show("File found!`n`n$foundPath", "Success", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
            } else {
                $lblStatus.Text = "Deep Search failed."
                [System.Windows.Forms.MessageBox]::Show("Could not find '$searchName' on any local drive.", "Not Found", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
            }
            [System.Windows.Forms.Cursor]::Current = [System.Windows.Forms.Cursors]::Default
        }
    })

    $ctx.Items.Add( (New-Object System.Windows.Forms.ToolStripSeparator) )

    # --- Unmark ---
    $itemUnmark = $ctx.Items.Add("Unmark / Cancel Action")
    $itemUnmark.Add_Click({
        foreach ($row in $dg.SelectedRows) {
            $obj = $row.DataBoundItem
            $obj.Action = "None"
            $obj.Details = "Review Needed"
            $dg.Refresh()
            $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::Orange
        }
    })
    $dg.ContextMenuStrip = $ctx

    # --- CHANGED: DELETE BUTTON LOGIC ---
    $btnDelete.Add_Click({
        $count = $dg.SelectedRows.Count
        if ($count -eq 0) { return }

        $res = [System.Windows.Forms.MessageBox]::Show("Permanently delete $count selected shortcut(s)?`n`nThis action is immediate.", "Confirm Delete", [System.Windows.Forms.MessageBoxButtons]::YesNo, [System.Windows.Forms.MessageBoxIcon]::Warning)
        
        if ($res -eq "Yes") {
            $rows = @($dg.SelectedRows) # Copy collection to avoid modification errors
            $deletedCount = 0
            
            foreach ($row in $rows) {
                try {
                    # Get FullPath from the hidden column we added
                    $path = $row.Cells["FullPath"].Value
                    
                    if ($path -and (Test-Path $path)) {
                        Remove-Item -Path $path -Force -ErrorAction Stop
                    }
                    
                    # Remove from UI
                    $dg.Rows.Remove($row)
                    $deletedCount++
                } catch {
                    [System.Windows.Forms.MessageBox]::Show("Could not delete: $path`n$($_.Exception.Message)", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
                }
            }
            $lblStatus.Text = "Deleted $deletedCount shortcuts."
        }
    })

    $script:ScanResults = @()

    # 5. SCAN LOGIC
    $f.Add_Shown({
        [System.Windows.Forms.Cursor]::Current = [System.Windows.Forms.Cursors]::WaitCursor
        $lblStatus.Text = "Scanning shortcuts... Please wait."
        $f.Text = "Broken Shortcut Manager - Scanning..."
        [System.Windows.Forms.Application]::DoEvents()

        $paths = @(
            "C:\ProgramData\Microsoft\Windows\Start Menu", 
            "$env:APPDATA\Microsoft\Windows\Start Menu",
            "$env:USERPROFILE\Desktop",
            "C:\Users\Public\Desktop",
            "$env:USERPROFILE\OneDrive\Desktop"
        ) | Select-Object -Unique | Where-Object { $_ -and (Test-Path $_) }

        $knownFixes = @{
            "My Computer"   = "::{20D04FE0-3AEA-1069-A2D8-08002B30309D}"
            "This PC"       = "::{20D04FE0-3AEA-1069-A2D8-08002B30309D}"
            "Recycle Bin"   = "::{645FF040-5081-101B-9F08-00AA002F954E}"
            "Control Panel" = "::{21EC2020-3AEA-1069-A2DD-08002B30309D}"
            "Documents"     = "::{450D8FBA-AD25-11D0-98A8-0800361B1103}"
        }

        $shell = New-Object -ComObject WScript.Shell
        
        $tempList = @()
        
        foreach ($path in $paths) {
            Get-ChildItem -Path $path -Filter *.lnk -Recurse -Force -ErrorAction SilentlyContinue | ForEach-Object {
                $lnkPath = $_.FullName
                try {
                    $sc = $shell.CreateShortcut($lnkPath)
                    $target = $sc.TargetPath
                } catch { return }

                if ($target -match '^shell:' -or $target -match '^\s*::{') { return }
                if ($target -match '^\s*$' -or $target.IndexOfAny([System.IO.Path]::GetInvalidPathChars()) -ge 0) { return }

                if (-not (Test-Path $target)) {
                    $action = "None"; $details = "Review Needed"; $newT = $null
                    $baseName = $_.BaseName

                    if ($knownFixes.ContainsKey($baseName)) {
                        $action = "Fix"; $details = "Restore System Path"; $newT = $knownFixes[$baseName]
                    } else {
                        $guessName = if ($target) { Split-Path $target -Leaf } else { ($baseName + ".exe") }
                        $peers = Get-ChildItem $_.DirectoryName -Filter *.lnk -ErrorAction SilentlyContinue | Where-Object { $_.FullName -ne $lnkPath }
                        foreach ($p in $peers) {
                            try {
                                $pt = $shell.CreateShortcut($p.FullName).TargetPath
                                if ($pt -and (Test-Path $pt)) {
                                    $parent = Split-Path $pt -Parent; $candidate = Join-Path $parent $guessName
                                    if (Test-Path $candidate) { $action = "Fix"; $details = "Auto-Found: $parent"; $newT = $candidate; break }
                                }
                            } catch {}
                        }
                    }

                    $tempList += [PSCustomObject]@{
                        Shortcut  = $_.Name
                        Folder    = (Split-Path $_.DirectoryName -Leaf)
                        Action    = $action
                        Details   = $details
                        NewTarget = $newT
                        FullPath  = $lnkPath
                    }
                }
            }
        }

        # 6. BIND
        $dt = New-Object System.Data.DataTable
        # Added 'FullPath' column to the DataTable so we can access it for deletion
        $dt.Columns.Add("Shortcut"); $dt.Columns.Add("Folder"); $dt.Columns.Add("Action"); $dt.Columns.Add("Details"); $dt.Columns.Add("NewTarget"); $dt.Columns.Add("FullPath")
        
        foreach ($item in $tempList) {
            $row = $dt.NewRow()
            $row["Shortcut"]  = $item.Shortcut
            $row["Folder"]    = $item.Folder
            $row["Action"]    = $item.Action
            $row["Details"]   = $item.Details
            $row["NewTarget"] = $item.NewTarget
            $row["FullPath"]  = $item.FullPath
            $dt.Rows.Add($row)
        }
        $dg.DataSource = $dt
        $dg.ClearSelection()
        
        # Hide FullPath from view
        if ($dg.Columns["FullPath"]) { $dg.Columns["FullPath"].Visible = $false }

        # 7. COLORS
        for ($i = 0; $i -lt $dg.Rows.Count; $i++) {
            $row = $dg.Rows[$i]
            $act = $row.Cells["Action"].Value
            if ($act -eq "Fix") { $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::LightGreen; $row.Selected = $true }
            elseif ($act -eq "None") { $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::Orange }
            else { $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::Gray }
        }
        
        $lblStatus.Text = "Scan Complete. Found $($tempList.Count) broken shortcuts."
        $f.Text = "Broken Shortcut Manager"
        [System.Windows.Forms.Cursor]::Current = [System.Windows.Forms.Cursors]::Default

        if ($tempList.Count -eq 0) {
            [System.Windows.Forms.MessageBox]::Show("Scan complete. No broken shortcuts found.", "All Clean", [System.Windows.Forms.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
        }
    })

    # Return Logic: Rebuild list from whatever is left in the grid to avoid processing deleted items
    if ($f.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) { 
        $finalList = @()
        foreach ($row in $dg.Rows) {
            # Map DataRowView back to object
            $finalList += [PSCustomObject]@{
                Shortcut  = $row.Cells["Shortcut"].Value
                Folder    = $row.Cells["Folder"].Value
                Action    = $row.Cells["Action"].Value
                Details   = $row.Cells["Details"].Value
                NewTarget = $row.Cells["NewTarget"].Value
                FullPath  = $row.Cells["FullPath"].Value
            }
        }
        return $finalList
    }
    return $null
}
function Invoke-ShortcutFix {
    $items = Show-BrokenShortcuts
    if (-not $items -or $items.Count -eq 0) { return }

    # 1. ANALYZE PLANNED ACTIONS
    $toFix = $items | Where-Object { $_.Action -eq "Fix" }
    $toDel = $items | Where-Object { $_.Action -eq "Delete" }

    if ($toFix.Count -eq 0 -and $toDel.Count -eq 0) {
        [System.Windows.Forms.MessageBox]::Show("No actions were selected.`n`n(Tip: Select rows and click 'Mark for Delete' or Browse to fix them.)", "No Action", [System.Windows.Forms.MessageBoxButton]::OK, [System.Windows.Forms.MessageBoxImage]::Information)
        return
    }

    # 2. CONFIRMATION PROMPT
    $msg = "You are about to apply the following actions:`n`n"
    if ($toFix.Count -gt 0) { $msg += "Fix: $($toFix.Count) shortcut(s)`n" }
    if ($toDel.Count -gt 0) { $msg += "Delete: $($toDel.Count) shortcut(s)`n" }
    $msg += "`nAre you sure you want to continue?"

    $confirm = [System.Windows.Forms.MessageBox]::Show($msg, "Confirm Actions", [System.Windows.Forms.MessageBoxButton]::YesNo, [System.Windows.Forms.MessageBoxImage]::Warning)
    
    if ($confirm -ne "Yes") { return }

    # 3. EXECUTE
    Invoke-UiCommand {
        param($toFix, $toDel)
        $shell = New-Object -ComObject WScript.Shell
        
        # Apply Fixes
        foreach ($item in $toFix) {
            try {
                $sc = $shell.CreateShortcut($item.FullPath)
                $sc.TargetPath = $item.NewTarget
                $sc.Save()
                Write-Output "Fixed: $($item.Shortcut)"
            } catch { Write-Output "Failed to fix $($item.Shortcut): $($_.Exception.Message)" }
        }

        # Apply Deletes
        foreach ($item in $toDel) {
            try {
                Remove-Item $item.FullPath -Force -ErrorAction Stop
                Write-Output "Deleted: $($item.Shortcut)"
            } catch { Write-Output "Failed to delete $($item.Shortcut): $($_.Exception.Message)" }
        }

        Write-Output "Operation complete."

    } "Applying shortcut fixes..." -ArgumentList $toFix, $toDel
}

# --- FIREWALL TOOLS ---
function Invoke-FirewallExport {
    $target = Join-Path (Get-DataPath) ("firewall_rules_{0}.wfw" -f (Get-Date -Format "yyyyMMdd_HHmmss"))
    Invoke-UiCommand { param($target) netsh advfirewall export "$target" } "Exporting firewall rules..." -ArgumentList $target
}

function Invoke-FirewallImport {
    $dlg = New-Object System.Windows.Forms.OpenFileDialog
    $dlg.Filter = "Windows Firewall Policy (*.wfw)|*.wfw"
    if ($dlg.ShowDialog() -ne "OK") { return }
    $file = $dlg.FileName
    Invoke-UiCommand { param($file) netsh advfirewall import "$file" } "Importing firewall rules..." -ArgumentList $file
}

function Invoke-FirewallDefaults {
    $confirm = [System.Windows.MessageBox]::Show("Restore default Windows Firewall rules? Custom rules will be removed.", "Restore Defaults", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    if ($confirm -ne "Yes") { return }
    Invoke-UiCommand { netsh advfirewall reset } "Restoring default firewall rules..."
}

function Invoke-FirewallPurge {
    $confirm = [System.Windows.MessageBox]::Show("Delete ALL firewall rules? This is destructive.", "Delete All Rules", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    if ($confirm -ne "Yes") { return }
    Invoke-UiCommand { Remove-NetFirewallRule -All } "Deleting all firewall rules..."
}

# --- DRIVER TOOLS ---
function Invoke-DriverReport {
    Invoke-UiCommand {
        $outfile = Join-Path (Get-DataPath) "Installed_Drivers.txt"
        driverquery /v > $outfile
        Write-Output "Driver report saved to $outfile"
    } "Creating driver report..."
}

function Invoke-ExportDrivers {
    Invoke-UiCommand {
        $path = Join-Path (Get-DataPath) ("Drivers_Backup_{0}" -f (Get-Date -Format "yyyyMMdd_HHmm"))
        New-Item -ItemType Directory -Path $path -Force | Out-Null
        $proc = Start-Process pnputil.exe -ArgumentList "/export-driver","*","""$path""" -NoNewWindow -Wait -PassThru
        if ($proc.ExitCode -eq 0) {
            Write-Output "Drivers exported to $path"
            [System.Windows.MessageBox]::Show("Drivers exported to:`n$path","Export Drivers",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Information) | Out-Null
        } else {
            Write-Output "Export failed (pnputil exit $($proc.ExitCode))."
            [System.Windows.MessageBox]::Show("Export failed (pnputil exit $($proc.ExitCode)).","Export Drivers",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Error) | Out-Null
        }
    } "Exporting drivers to data folder..."
}

function Show-GhostDevicesDialog {
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Ghost Devices"
    $f.Size = "800, 500"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.ForeColor = [System.Drawing.Color]::White

    $dg = New-Object System.Windows.Forms.DataGridView
    $dg.Dock = "Top"
    $dg.Height = 380
    $dg.BackgroundColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.ForeColor = [System.Drawing.Color]::White
    $dg.AutoSizeColumnsMode = "Fill"
    $dg.SelectionMode = "FullRowSelect"
    $dg.MultiSelect = $true
    $dg.ReadOnly = $true
    $dg.RowHeadersVisible = $false
    $dg.AllowUserToAddRows = $false
    $dg.BorderStyle = "None"
    $dg.EnableHeadersVisualStyles = $false
    $dg.ColumnHeadersDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
    $dg.ColumnHeadersDefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.ColumnHeadersDefaultCellStyle.Padding = (New-Object System.Windows.Forms.Padding 4)
    $dg.ColumnHeadersBorderStyle = [System.Windows.Forms.DataGridViewHeaderBorderStyle]::Single
    $dg.ColumnHeadersHeight = 28
    $dg.DefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.DefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.DefaultCellStyle.SelectionBackColor = [System.Drawing.ColorTranslator]::FromHtml("#007ACC")
    $dg.DefaultCellStyle.SelectionForeColor = [System.Drawing.Color]::White
    $dg.AlternatingRowsDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#252526")
    $dg.GridColor = [System.Drawing.ColorTranslator]::FromHtml("#333333")
    $f.Controls.Add($dg)

    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Dock = "Bottom"
    $pnl.Height = 80
    $pnl.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.Controls.Add($pnl)

    function New-GDButton($text,$x,$color=$null){
        $b=New-Object System.Windows.Forms.Button
        $b.Text=$text
        $b.Left=$x; $b.Top=20; $b.Width=120; $b.Height=35
        $b.FlatStyle="Flat"
        $b.FlatAppearance.BorderSize=1
        $b.FlatAppearance.BorderColor=[System.Drawing.ColorTranslator]::FromHtml("#444444")
        $b.ForeColor=[System.Drawing.Color]::White
        if($color){
            $b.BackColor=[System.Drawing.ColorTranslator]::FromHtml($color)
            $b.FlatAppearance.MouseOverBackColor=[System.Windows.Forms.ControlPaint]::Light($b.BackColor)
        } else {
            $b.BackColor=[System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
            $b.FlatAppearance.MouseOverBackColor=[System.Drawing.ColorTranslator]::FromHtml("#3E3E42")
        }
        $pnl.Controls.Add($b); return $b
    }

    $btnRefresh = New-GDButton "Refresh" 20
    $btnRemoveSel = New-GDButton "Remove Selected" 160 "#802020"
    $btnRemoveAll = New-GDButton "Remove All" 320 "#A04040"
    $btnClose = New-GDButton "Close" 480

    $Load = {
        $items = Get-PnpDevice -ErrorAction SilentlyContinue | Where-Object { $_.Status -eq 'Unknown' }
        $dt = New-Object System.Data.DataTable
        $dt.Columns.Add("InstanceId")
        $dt.Columns.Add("Class")
        $dt.Columns.Add("FriendlyName")
        foreach($d in $items){
            $r=$dt.NewRow()
            $r["InstanceId"]=$d.InstanceId
            $r["Class"]=$d.Class
            $r["FriendlyName"]=$d.FriendlyName
            $dt.Rows.Add($r)
        }
        $dg.DataSource=$dt
        $dg.ClearSelection()
        if(-not $items -or $items.Count -eq 0){
            [System.Windows.MessageBox]::Show("No hidden/ghost devices found.","Ghost Devices",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Information) | Out-Null
        }
    }

    $btnRefresh.Add_Click({ & $Load })
    $btnRemoveSel.Add_Click({
        if($dg.SelectedRows.Count -eq 0){ return }
        foreach($row in $dg.SelectedRows){
            $id = $row.Cells["InstanceId"].Value
            if($id){ pnputil /remove-device $id | Out-Null }
        }
        & $Load
    })
    $btnRemoveAll.Add_Click({
        if($dg.Rows.Count -eq 0){ return }
        foreach($row in $dg.Rows){
            $id = $row.Cells["InstanceId"].Value
            if($id){ pnputil /remove-device $id | Out-Null }
        }
        & $Load
    })
    $btnClose.Add_Click({ $f.Close() })

    & $Load
    $f.ShowDialog() | Out-Null
}

function Invoke-DriverUpdates {
    param([bool]$Enable)
    $value = if ($Enable) { 0 } else { 1 }
    $msg = if ($Enable) { "Enabled automatic driver updates." } else { "Disabled automatic driver updates." }
    Invoke-UiCommand {
        param($value,$msg)
        $path = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\DriverSearching"
        if (-not (Test-Path $path)) { New-Item -Path $path -Force | Out-Null }
        Set-ItemProperty -Path $path -Name "SearchOrderConfig" -Value $value -Type DWord
        Write-Output $msg
    } "Updating driver update policy..." -ArgumentList $value, $msg
}

function Invoke-DeviceMetadata {
    param([bool]$Enable)
    $value = if ($Enable) { 0 } else { 1 }
    $msg = if ($Enable) { "Device metadata downloads enabled." } else { "Device metadata downloads disabled." }
    Invoke-UiCommand {
        param($value,$msg)
        $path = "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Device Metadata"
        if (-not (Test-Path $path)) { New-Item -Path $path -Force | Out-Null }
        Set-ItemProperty -Path $path -Name "PreventDeviceMetadataFromNetwork" -Value $value -Type DWord
        Write-Output $msg
    } "Updating device metadata policy..." -ArgumentList $value, $msg
}

function Show-DriverCleanupDialog {
    # 1. SCANNING PHASE
    $rawOutput = pnputil.exe /enum-drivers 2>&1
    $drivers = @()
    $current = $null

    foreach ($line in $rawOutput) {
        $line = $line.ToString().Trim()
        
        if ($line.Contains(":")) {
            $parts = $line -split ":", 2
            $key   = $parts[0].Trim()
            $val   = $parts[1].Trim()
            
            # 1. Check for Published Name (Start of new driver block)
            if ($val -match '^(oem\d+\.inf)$') {
                if ($current) { $drivers += [PSCustomObject]$current }
                $current = [ordered]@{ 
                    PublishedName = $val
                    OriginalName  = $null
                    Provider      = "Unknown"
                    Version       = [Version]"0.0.0.0"
                    DisplayVer    = "Unknown"
                    SortDate      = [DateTime]::MinValue
                    DisplayDate   = "Unknown"
                }
            }
            elseif ($current) {
                # 2. Parse by Label with Regex Extraction
                if ($key -match "Original Name") {
                    if ($val -notmatch '^oem\d+\.inf$') { $current.OriginalName = $val }
                }
                elseif ($key -match "Provider") {
                    $current.Provider = $val
                }
                elseif ($key -match "Version") {
                    # EXTRACT Version Number (e.g. 2.2.0.134)
                    if ($val -match '(\d+(\.\d+){1,3})') {
                        $vStr = $matches[1]
                        $current.DisplayVer = $vStr
                        try { $current.Version = [Version]$vStr } catch {}
                    } else {
                        $current.DisplayVer = $val 
                    }

                    # Check if line ALSO contains a Date
                    if ($current.DisplayDate -eq "Unknown" -and $val -match '(\d{2}[/\-]\d{2}[/\-]\d{4})') {
                        $dStr = $matches[1]
                        if ($dStr -as [DateTime]) {
                             $current.DisplayDate = $dStr
                             $current.SortDate = [DateTime]$dStr
                        }
                    }
                }
                elseif ($key -match "Date") {
                    if ($val -as [DateTime]) {
                        $current.DisplayDate = $val
                        $current.SortDate = [DateTime]$val
                    } elseif ($val -match '(\d{2}[/\-]\d{2}[/\-]\d{4})') {
                        $dStr = $matches[1]
                        $current.DisplayDate = $dStr
                        if ($dStr -as [DateTime]) { $current.SortDate = [DateTime]$dStr }
                    }
                }
                # 3. Fallback
                elseif ($null -eq $current.OriginalName -and $val -match '\.inf$') {
                    $current.OriginalName = $val
                }
            }   
        }
    }
    if ($current) { $drivers += [PSCustomObject]$current }

    # 2. FILTERING DUPLICATES
    $grouped = $drivers | Where-Object { $_.OriginalName } | Group-Object OriginalName
    $toDelete = @()

    foreach ($group in $grouped) {
        if ($group.Count -gt 1) {
            $sorted = $group.Group | Sort-Object SortDate, Version -Descending
            $toDelete += $sorted | Select-Object -Skip 1
        }
    }

    if (-not $toDelete -or $toDelete.Count -eq 0) {
        [System.Windows.MessageBox]::Show("Driver store is already clean. No duplicates found.", "Clean Old Drivers", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
        return
    }

    # 3. GUI SETUP (THEMED)
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Clean Old Drivers"
    $f.Size = "950, 600"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.ForeColor = [System.Drawing.Color]::White
    
    # Initialize ToolTip provider
    $tip = New-Object System.Windows.Forms.ToolTip
    $tip.AutoPopDelay = 5000
    $tip.InitialDelay = 500
    $tip.ReshowDelay = 500
    $tip.ShowAlways = $true

    # --- LAYOUT FIX: PANEL FIRST (Dock Bottom) --- 
    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Dock = "Bottom"
    $pnl.Height = 60 # Reduced height for cleaner look
    $pnl.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    
    # Draw a subtle top border on the panel
    $pnl.Add_Paint({
        param($s, $e) # Renamed from $sender to $s
        $pen = New-Object System.Drawing.Pen([System.Drawing.Color]::FromArgb(60,60,60), 1)
        $e.Graphics.DrawLine($pen, 0, 0, $s.Width, 0)
    })
    $f.Controls.Add($pnl)

    # --- GRID SECOND (Dock Fill) ---
    $dg = New-Object System.Windows.Forms.DataGridView
    $dg.Dock = "Fill" 
    $dg.BackgroundColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.ForeColor = [System.Drawing.Color]::White
    $dg.AutoSizeColumnsMode = "Fill"
    $dg.SelectionMode = "FullRowSelect"
    $dg.MultiSelect = $true
    $dg.ReadOnly = $true
    $dg.RowHeadersVisible = $false
    $dg.AllowUserToAddRows = $false
    $dg.BorderStyle = "None"
    $dg.CellBorderStyle = "SingleHorizontal"
    
    # Header Styling
    $dg.EnableHeadersVisualStyles = $false
    $dg.ColumnHeadersDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
    $dg.ColumnHeadersDefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.ColumnHeadersDefaultCellStyle.Padding = (New-Object System.Windows.Forms.Padding 6)
    $dg.ColumnHeadersBorderStyle = [System.Windows.Forms.DataGridViewHeaderBorderStyle]::Single
    $dg.ColumnHeadersHeight = 35
    
    # Row Styling
    $dg.DefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.DefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.DefaultCellStyle.SelectionBackColor = [System.Drawing.ColorTranslator]::FromHtml("#007ACC")
    $dg.DefaultCellStyle.SelectionForeColor = [System.Drawing.Color]::White
    $dg.AlternatingRowsDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#252526")
    $dg.GridColor = [System.Drawing.ColorTranslator]::FromHtml("#333333")
    
    $f.Controls.Add($dg)
    $dg.BringToFront() # Ensures grid fills the remaining space above the panel

    # Helper for Themed Buttons with Tooltips AND Anchor support
    function New-DrvBtn($text, $x, $color=$null, $tooltipText="", $anchor="Top, Left"){
        $b=New-Object System.Windows.Forms.Button
        $b.Text=$text
        $b.Left=$x
        $b.Top=12 # Vertically centered in 60px panel
        $b.Width=160
        $b.Height=35
        $b.FlatStyle="Flat"
        $b.FlatAppearance.BorderSize=1
        $b.Anchor = $anchor
        $b.FlatAppearance.BorderColor=[System.Drawing.ColorTranslator]::FromHtml("#444444")
        $b.ForeColor=[System.Drawing.Color]::White
        if($color){
            $b.BackColor=[System.Drawing.ColorTranslator]::FromHtml($color)
            $b.FlatAppearance.MouseOverBackColor=[System.Windows.Forms.ControlPaint]::Light($b.BackColor)
        } else {
            $b.BackColor=[System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
            $b.FlatAppearance.MouseOverBackColor=[System.Drawing.ColorTranslator]::FromHtml("#3E3E42")
        }
        
        if ($tooltipText) {
            $tip.SetToolTip($b, $tooltipText)
        }

        $pnl.Controls.Add($b); return $b
    }

    # --- CHANGED: Renamed "Remove (Options...)" to "Remove All" ---
    $btnBackupClean = New-DrvBtn "Remove All" 20 "#006600" "Select removal options (Backup vs No Backup) for ALL duplicates."
    
    $btnRemoveSel   = New-DrvBtn "Remove Selected" 190 "#802020" "Removes only the currently highlighted driver(s) from the list."
    
    # Place Close button aligned to the Right
    $closeX = $pnl.Width - 180
    $btnClose = New-DrvBtn "Close" $closeX $null "Close this window." "Top, Right"

    # 4. DATA BINDING
    $script:CurrentList = $toDelete

    $LoadGrid = {
        $dt = New-Object System.Data.DataTable
        $dt.Columns.Add("PublishedName")
        $dt.Columns.Add("OriginalName")
        $dt.Columns.Add("Provider")
        $dt.Columns.Add("Version")
        $dt.Columns.Add("Date")

        foreach($d in $script:CurrentList){
            $r=$dt.NewRow()
            $r["PublishedName"] = $d.PublishedName
            $r["OriginalName"]  = $d.OriginalName
            $r["Provider"]      = $d.Provider
            $r["Version"]       = $d.DisplayVer
            $r["Date"]          = $d.DisplayDate
            $dt.Rows.Add($r)
        }
        $dg.DataSource = $dt
        $dg.ClearSelection()
    }

    # 5. ACTION LOGIC
    $DoRemove = {
        param($items, $CloseWindow)
        
        if(-not $items -or $items.Count -eq 0){ return }
        $count = $items.Count
        
        # --- NEW CUSTOM CONFIRMATION DIALOG ---
        $cf = New-Object System.Windows.Forms.Form
        $cf.Text = "Confirm Driver Cleanup"
        $cf.Size = "450, 240"
        $cf.StartPosition = "CenterParent"
        $cf.FormBorderStyle = "FixedDialog"
        $cf.ControlBox = $false
        $cf.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
        $cf.ForeColor = "White"

        $lbl = New-Object System.Windows.Forms.Label
        $lbl.Text = "You are about to remove $count driver(s).`n`nHow would you like to proceed?"
        $lbl.Location = "20, 20"; $lbl.Size = "400, 50"; $lbl.Font = New-Object System.Drawing.Font("Segoe UI", 10)
        $cf.Controls.Add($lbl)

        $bBackup = New-Object System.Windows.Forms.Button
        $bBackup.Text = "Backup && Clean"
        $bBackup.DialogResult = "Yes"
        $bBackup.Location = "20, 90"; $bBackup.Width = 130; $bBackup.Height = 35
        $bBackup.BackColor = "SeaGreen"; $bBackup.ForeColor = "White"; $bBackup.FlatStyle = "Flat"
        $cf.Controls.Add($bBackup)

        $bNoBackup = New-Object System.Windows.Forms.Button
        $bNoBackup.Text = "Clean (No Backup)"
        $bNoBackup.DialogResult = "No"
        $bNoBackup.Location = "160, 90"; $bNoBackup.Width = 130; $bNoBackup.Height = 35
        $bNoBackup.BackColor = "IndianRed"; $bNoBackup.ForeColor = "White"; $bNoBackup.FlatStyle = "Flat"
        $cf.Controls.Add($bNoBackup)

        $bCancel = New-Object System.Windows.Forms.Button
        $bCancel.Text = "Cancel"
        $bCancel.DialogResult = "Cancel"
        $bCancel.Location = "300, 90"; $bCancel.Width = 110; $bCancel.Height = 35
        $bCancel.BackColor = "DimGray"; $bCancel.ForeColor = "White"; $bCancel.FlatStyle = "Flat"
        $cf.Controls.Add($bCancel)

        $result = $cf.ShowDialog()
        
        if ($result -eq "Cancel") { return }

        # A. BACKUP (Only if Yes selected)
        $backupCount = 0
        $timestamp = Get-Date -f 'yyyyMMdd_HHmm'
        $mainBkPath = Join-Path (Get-DataPath) "Drivers_Backup_$timestamp"

        if ($result -eq "Yes") {
            if (-not (Test-Path $mainBkPath)) { New-Item -Path $mainBkPath -ItemType Directory -Force | Out-Null }
            
            $prog = 1
            foreach($item in $items) {
                $f.Text = "Backing up ($prog/$count): $($item.OriginalName)..."
                $f.Update()
                
                $folderName = if ($item.OriginalName) { $item.OriginalName } else { $item.PublishedName }
                $drvPath = Join-Path $mainBkPath $folderName
                New-Item -Path $drvPath -ItemType Directory -Force | Out-Null

                $proc = Start-Process pnputil.exe -ArgumentList "/export-driver", $item.PublishedName, """$drvPath""" -NoNewWindow -Wait -PassThru
                if ($proc.ExitCode -eq 0) { $backupCount++ }
                $prog++
            }
        }

        $f.Text = "Processing Deletions..."
        $f.Update()

        # B. DELETION
        $deleted = 0
        $failed = 0
        
        foreach($item in $items){
            $name = $item.PublishedName
            
            $p = New-Object System.Diagnostics.Process
            $p.StartInfo.FileName = "pnputil.exe"
            $p.StartInfo.Arguments = "/delete-driver $name /uninstall"
            $p.StartInfo.RedirectStandardOutput = $true
            $p.StartInfo.RedirectStandardError = $true
            $p.StartInfo.UseShellExecute = $false
            $p.StartInfo.CreateNoWindow = $true
            $p.Start() | Out-Null
            $stdOut = $p.StandardOutput.ReadToEnd()
            $stdErr = $p.StandardError.ReadToEnd()
            $p.WaitForExit()
            
            if ($p.ExitCode -eq 0 -or $p.ExitCode -eq 3010) {
                $deleted++
            } else {
                # Force Prompt
                $fullLog = "$stdOut`n$stdErr".Trim()
                $warnMsg = "Driver: $($item.OriginalName) ($name)`n`nError:`n$fullLog`n`nForce Delete?"
                $forceDec = [System.Windows.MessageBox]::Show($warnMsg, "Deletion Failed", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Error)
                
                if ($forceDec -eq "Yes") {
                    $procForce = Start-Process pnputil.exe -ArgumentList "/delete-driver $name /uninstall /force" -NoNewWindow -Wait -PassThru
                    if ($procForce.ExitCode -eq 0 -or $procForce.ExitCode -eq 3010) { $deleted++ } else { $failed++ }
                } else {
                    $failed++
                }
            }
        }

        # C. REPORT & REFRESH
        $resMsg = "Done.`nDeleted: $deleted"
        if ($result -eq "Yes") { $resMsg += "`nBackups: $backupCount`nPath: $mainBkPath" }
        
        [System.Windows.MessageBox]::Show($resMsg, "Result", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null

        if ($deleted -gt 0) {
            # Remove deleted items from the current list
            $script:CurrentList = $script:CurrentList | Where-Object { 
                $obj = $_; -not ($items | Where-Object { $_.PublishedName -eq $obj.PublishedName })
            }
            $dg.DataSource = $null
            $LoadGrid.Invoke()
        }
        
        $f.Text = "Clean Old Drivers"
        if ($CloseWindow) { $f.Close() }
    }

    $btnBackupClean.Add_Click({ $DoRemove.Invoke($script:CurrentList, $true) })
    $btnRemoveSel.Add_Click({
        if($dg.SelectedRows.Count -eq 0){ return }
        $selected = @()
        foreach($row in $dg.SelectedRows){
            $pub = $row.Cells["PublishedName"].Value
            $match = $script:CurrentList | Where-Object { $_.PublishedName -eq $pub } | Select-Object -First 1
            if($match){ $selected += $match }
        }
        $DoRemove.Invoke($selected, $false)
    })
    $btnClose.Add_Click({ $f.Close() })

    $LoadGrid.Invoke()
    $f.ShowDialog() | Out-Null
}

# --- RESTORE DRIVERS ---
function Invoke-RestoreDrivers {
    $dataPath = Get-DataPath
    $backups = @()
    try {
        $backups = Get-ChildItem -Path $dataPath -Directory -ErrorAction SilentlyContinue | Where-Object { $_.Name -match '^DriverBackup_' -or $_.Name -match '^Drivers_Backup_' }
    } catch {}

    $selectedPath = $null
    if ($backups -and $backups.Count -gt 0) {
        $backups = $backups | Sort-Object LastWriteTime -Descending
        $form = New-Object System.Windows.Forms.Form
        $form.Text = "Select Driver Backup"
        $form.Size = "600,400"
        $form.StartPosition = "CenterScreen"
        $form.BackColor = [System.Drawing.Color]::FromArgb(30,30,30)
        $form.ForeColor = "White"

        $lst = New-Object System.Windows.Forms.ListBox
        $lst.Dock = "Top"
        $lst.Height = 280
        $lst.BackColor = [System.Drawing.Color]::FromArgb(20,20,20)
        $lst.ForeColor = "White"
        $lst.BorderStyle = "FixedSingle"
        $items = @()
        foreach ($b in $backups) {
            $items += [PSCustomObject]@{
                Name = $b.Name
                Path = $b.FullName
                Display = "{0}  (modified {1})" -f $b.Name, $b.LastWriteTime
            }
        }
        $lst.DisplayMember = "Display"
        $lst.ValueMember = "Path"
        $lst.DataSource = $items
        $form.Controls.Add($lst)

        $pnl = New-Object System.Windows.Forms.Panel
        $pnl.Dock = "Bottom"
        $pnl.Height = 60
        $pnl.BackColor = [System.Drawing.Color]::FromArgb(30,30,30)
        $form.Controls.Add($pnl)

        $btnUse = New-Object System.Windows.Forms.Button
        $btnUse.Text = "Use Selected"
        $btnUse.Left = 20; $btnUse.Top = 15; $btnUse.Width = 140
        $btnUse.BackColor = "SeaGreen"; $btnUse.ForeColor = "White"
        $btnUse.FlatStyle = "Flat"
        $btnUse.Add_Click({
            if ($lst.SelectedItem) {
                $form.Tag = $lst.SelectedItem.Path
                $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
                $form.Close()
            }
        })
        $pnl.Controls.Add($btnUse)

        $btnBrowse = New-Object System.Windows.Forms.Button
        $btnBrowse.Text = "Browse..."
        $btnBrowse.Left = 180; $btnBrowse.Top = 15; $btnBrowse.Width = 120
        $btnBrowse.BackColor = "DimGray"; $btnBrowse.ForeColor = "White"
        $btnBrowse.FlatStyle = "Flat"
        $btnBrowse.Add_Click({
            $form.Tag = "BROWSE"
            $form.DialogResult = [System.Windows.Forms.DialogResult]::Retry
            $form.Close()
        })
        $pnl.Controls.Add($btnBrowse)

        $btnCancel = New-Object System.Windows.Forms.Button
        $btnCancel.Text = "Cancel"
        $btnCancel.Left = 320; $btnCancel.Top = 15; $btnCancel.Width = 120
        $btnCancel.BackColor = "Gray"; $btnCancel.ForeColor = "White"
        $btnCancel.FlatStyle = "Flat"
        $btnCancel.Add_Click({
            $form.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
            $form.Close()
        })
        $pnl.Controls.Add($btnCancel)

        $res = $form.ShowDialog()
        if ($res -eq [System.Windows.Forms.DialogResult]::OK -and $form.Tag) {
            $selectedPath = $form.Tag
        } elseif ($res -eq [System.Windows.Forms.DialogResult]::Retry) {
            # fall through to browse
            $selectedPath = $null
        } else {
            return
        }
    }

    if (-not $selectedPath) {
        $dlg = New-Object System.Windows.Forms.FolderBrowserDialog
        $dlg.Description = "Select DriverBackup folder"
        if (Test-Path $dataPath) { $dlg.SelectedPath = $dataPath }
        if ($dlg.ShowDialog() -ne "OK") { return }
        $selectedPath = $dlg.SelectedPath
    }

    Invoke-UiCommand {
        param($Path)
        if (-not (Test-Path $Path)) {
            Write-Output "Restore failed: path not found $Path"
            [System.Windows.MessageBox]::Show("Restore failed: path not found.`n$Path","Restore Drivers",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Error) | Out-Null
            return
        }

        $infFiles = Get-ChildItem -Path $Path -Filter *.inf -Recurse -ErrorAction SilentlyContinue
        if (-not $infFiles -or $infFiles.Count -eq 0) {
            Write-Output "Restore aborted: no INF files found in $Path"
            [System.Windows.MessageBox]::Show("No INF files found in:`n$Path","Restore Drivers",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Warning) | Out-Null
            return
        }

        $output = pnputil.exe /add-driver "$Path\*.inf" /subdirs 2>&1
        $code = $LASTEXITCODE
        Write-Output $output
        if ($code -eq 0 -or $code -eq 3010) {
            Write-Output "Drivers restored from $Path"
            [System.Windows.MessageBox]::Show("Drivers restored from:`n$Path","Restore Drivers",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Information) | Out-Null
        } else {
            Write-Output "Restore failed (exit $code)."
            $msg = "Restore failed (exit $code)." + "`n`nOutput:`n" + ($output | Out-String)
            [System.Windows.MessageBox]::Show($msg,"Restore Drivers",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Error) | Out-Null
        }
    # CHANGE IS HERE: Passing the argument explicitly
    } "Restoring drivers..." -ArgumentList $selectedPath
}

# --- UPDATE / REPORT TOOLS ---
function Invoke-WindowsUpdateRepairFull {
    Invoke-UiCommand {
        $services = @('wuauserv','bits','cryptsvc','msiserver','usosvc','trustedinstaller')
        foreach ($svc in $services) { try { Stop-Service -Name $svc -Force -ErrorAction SilentlyContinue } catch {} }
        try { Get-BitsTransfer -AllUsers | Remove-BitsTransfer -Confirm:$false } catch {}
        $suffix = ".bak_{0}" -f (Get-Random -Maximum 99999)
        if (Test-Path "$env:windir\\SoftwareDistribution") { try { Rename-Item "$env:windir\\SoftwareDistribution" ("SoftwareDistribution" + $suffix) -ErrorAction SilentlyContinue; Write-Output "Renamed SoftwareDistribution$suffix" } catch {} }
        if (Test-Path "$env:windir\\System32\\catroot2") { try { Rename-Item "$env:windir\\System32\\catroot2" ("catroot2" + $suffix) -ErrorAction SilentlyContinue; Write-Output "Renamed catroot2$suffix" } catch {} }
        $dlls = @("atl.dll","urlmon.dll","mshtml.dll","shdocvw.dll","browseui.dll","jscript.dll","vbscript.dll","scrrun.dll","msxml.dll","msxml3.dll","msxml6.dll","actxprxy.dll","softpub.dll","wintrust.dll","dssenh.dll","rsaenh.dll","gpkcsp.dll","sccbase.dll","slbcsp.dll","cryptdlg.dll","oleaut32.dll","ole32.dll","shell32.dll","initpki.dll","wuapi.dll","wuaueng.dll","wuaueng1.dll","wucltui.dll","wups.dll","wups2.dll","wuweb.dll","qmgr.dll","qmgrprxy.dll","wucltux.dll","muweb.dll","wuwebv.dll")
        foreach ($dll in $dlls) { try { regsvr32.exe /s $dll } catch {} }
        try { netsh winsock reset | Out-Null } catch {}
        try { netsh winhttp reset proxy | Out-Null } catch {}
        foreach ($svc in $services) { try { Start-Service -Name $svc -ErrorAction SilentlyContinue } catch {} }
        Write-Output "Windows Update repair completed."
    } "Running full Windows Update repair..."
}

function Invoke-SystemReports {
    $dlg = New-Object System.Windows.Forms.FolderBrowserDialog
    $dlg.Description = "Select output folder for system reports"
    if ($dlg.ShowDialog() -ne "OK") { return }
    $outdir = Join-Path $dlg.SelectedPath ("SystemReports_{0}" -f (Get-Date -Format "yyyy-MM-dd_HHmm"))
    if (-not (Test-Path $outdir)) { New-Item -ItemType Directory -Path $outdir | Out-Null }
    
    Invoke-UiCommand {
        param($outdir)
        $date = Get-Date -Format "yyyy-MM-dd"
        $sys = Join-Path $outdir "System_Info_$date.txt"
        $net = Join-Path $outdir "Network_Info_$date.txt"
        $drv = Join-Path $outdir "Driver_List_$date.txt"
        systeminfo | Out-File -FilePath $sys -Encoding UTF8
        ipconfig /all | Out-File -FilePath $net -Encoding UTF8
        driverquery | Out-File -FilePath $drv -Encoding UTF8
        Write-Output "Reports saved to $outdir"
    # CHANGE IS HERE: Passing the argument explicitly
    } "Generating system reports..." -ArgumentList $outdir
}

function Invoke-UpdateServiceReset {
    Invoke-UiCommand {
        try {
            $script:UpdateSvcResult = "OK"
            Stop-Service -Name wuauserv -Force -ErrorAction SilentlyContinue
            Stop-Service -Name cryptsvc -Force -ErrorAction SilentlyContinue
            Start-Service -Name appidsvc -ErrorAction SilentlyContinue
            Start-Service -Name wuauserv -ErrorAction SilentlyContinue
            Start-Service -Name cryptsvc -ErrorAction SilentlyContinue
            Start-Service -Name bits -ErrorAction SilentlyContinue
            Write-Output "Restarted Windows Update related services."
        } catch {
            $script:UpdateSvcResult = "ERR: $($_.Exception.Message)"
            throw
        }
    } "Restarting Windows Update services..."
}

function Set-DotNetRollForward {
    param([string]$Mode)
    switch ($Mode) {
        "Runtime" { [System.Environment]::SetEnvironmentVariable("DOTNET_ROLL_FORWARD", "LatestMajor", "Machine"); Write-Output "Runtime roll-forward enabled (LatestMajor)." }
        "SDK" {
            $latestSdk = & dotnet --list-sdks | Sort-Object -Descending | Select-Object -First 1
            if ($latestSdk) {
                $version = $latestSdk.Split()[0]
                $globalJsonPath = "$env:USERPROFILE\global.json"
                @{
                    sdk = @{
                        version = $version
                        rollForward = "latestMajor"
                    }
                } | ConvertTo-Json -Depth 3 | Out-File -Encoding UTF8 $globalJsonPath
                Write-Output "SDK roll-forward set to $version (global.json at $globalJsonPath)."
            } else { Write-Output "No SDK detected." }
        }
        "Both" {
            [System.Environment]::SetEnvironmentVariable("DOTNET_ROLL_FORWARD", "LatestMajor", "Machine")
            $latestSdk = & dotnet --list-sdks | Sort-Object -Descending | Select-Object -First 1
            if ($latestSdk) {
                $version = $latestSdk.Split()[0]
                $globalJsonPath = "$env:USERPROFILE\global.json"
                @{
                    sdk = @{
                        version = $version
                        rollForward = "latestMajor"
                    }
                } | ConvertTo-Json -Depth 3 | Out-File -Encoding UTF8 $globalJsonPath
            }
            Write-Output "Runtime + SDK roll-forward configured."
        }
        "Disable" {
            [System.Environment]::SetEnvironmentVariable("DOTNET_ROLL_FORWARD", $null, "Machine")
            $globalJsonPath = "$env:USERPROFILE\global.json"
            if (Test-Path $globalJsonPath) {
                try {
                    $json = Get-Content $globalJsonPath -Raw | ConvertFrom-Json
                    if ($json.sdk.rollForward) { $json.sdk.PSObject.Properties.Remove("rollForward"); $json | ConvertTo-Json -Depth 3 | Out-File -Encoding UTF8 $globalJsonPath }
                } catch {}
            }
            Write-Output ".NET roll-forward disabled."
        }
    }
}

function Invoke-MASActivation {
    $masInput = [Microsoft.VisualBasic.Interaction]::InputBox("Type YES, I UNDERSTAND to download and run MAS from massgrave.dev", "MAS Activation Confirmation", "")
    if ($masInput -ne "YES, I UNDERSTAND") { return }
    Invoke-UiCommand {
        $scriptContent = Invoke-RestMethod -Uri "https://get.activated.win"
        Invoke-Expression -Command $scriptContent
        Write-Output "MAS script executed."
    } "Running MAS activation..."
}

function Show-ContextMenuBuilder {
    # Ensure libraries are loaded
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing

    # --- SETUP FORM ---
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Custom Context Menu Builder"
    $f.Size = New-Object System.Drawing.Size(600, 420) # Increased height for description
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $f.ForeColor = "White"
    $f.FormBorderStyle = "FixedDialog"
    $f.MaximizeBox = $false

    # --- HEADER / WARNING ---
    $lblWarn = New-Object System.Windows.Forms.Label
    $lblWarn.Text = "NOTE: This tool replaces the 'Set as desktop background' option to force your custom action into the top-level Windows 11 menu."
    $lblWarn.Location = "20, 10"; $lblWarn.Size = "540, 40"
    $lblWarn.ForeColor = "Yellow" # Highlight the warning
    $lblWarn.Font = New-Object System.Drawing.Font("Segoe UI", 9)
    $f.Controls.Add($lblWarn)

    # --- UI INPUTS ---
    $lbl1 = New-Object System.Windows.Forms.Label; $lbl1.Text = "Menu Name:"; $lbl1.Location = "20, 50"; $lbl1.ForeColor = "LightGray"; $lbl1.AutoSize = $true; $f.Controls.Add($lbl1)
    
    $txtName = New-Object System.Windows.Forms.TextBox; $txtName.Location = "20, 75"; $txtName.Width = 540; 
    $txtName.BackColor = [System.Drawing.Color]::FromArgb(40, 40, 40); $txtName.ForeColor = "White"
    $txtName.Text = "Take Ownership" # DEFAULT VALUE
    $f.Controls.Add($txtName)

    $lbl2 = New-Object System.Windows.Forms.Label; $lbl2.Text = "Command:"; $lbl2.Location = "20, 115"; $lbl2.ForeColor = "LightGray"; $lbl2.AutoSize = $true; $f.Controls.Add($lbl2)
    
    $txtCmd = New-Object System.Windows.Forms.TextBox; $txtCmd.Location = "20, 140"; $txtCmd.Width = 450; 
    $txtCmd.BackColor = [System.Drawing.Color]::FromArgb(40, 40, 40); $txtCmd.ForeColor = "White"
    # DEFAULT COMMAND (Take Ownership)
    $txtCmd.Text = 'powershell -windowstyle hidden -command "Start-Process cmd -ArgumentList ''/c takeown /f \"%1\" /r /d y && icacls \"%1\" /grant administrators:F /t'' -Verb runAs"'
    $f.Controls.Add($txtCmd)
    
    $btnBrowseCmd = New-Object System.Windows.Forms.Button; $btnBrowseCmd.Text = "Browse..."; $btnBrowseCmd.Location = "480, 138"; $btnBrowseCmd.Width = 80; $btnBrowseCmd.BackColor = "DimGray"; $btnBrowseCmd.ForeColor = "White"; $btnBrowseCmd.FlatStyle = "Flat"; $f.Controls.Add($btnBrowseCmd)

    $lbl3 = New-Object System.Windows.Forms.Label; $lbl3.Text = "Icon Path (Optional):"; $lbl3.Location = "20, 180"; $lbl3.ForeColor = "LightGray"; $lbl3.AutoSize = $true; $f.Controls.Add($lbl3)
    
    $txtIcon = New-Object System.Windows.Forms.TextBox; $txtIcon.Location = "20, 205"; $txtIcon.Width = 450; 
    $txtIcon.BackColor = [System.Drawing.Color]::FromArgb(40, 40, 40); $txtIcon.ForeColor = "White"
    $txtIcon.Text = "imageres.dll,-78" # DEFAULT ICON (Shield)
    $f.Controls.Add($txtIcon)
    
    $btnBrowseIcon = New-Object System.Windows.Forms.Button; $btnBrowseIcon.Text = "Browse..."; $btnBrowseIcon.Location = "480, 203"; $btnBrowseIcon.Width = 80; $btnBrowseIcon.BackColor = "DimGray"; $btnBrowseIcon.ForeColor = "White"; $btnBrowseIcon.FlatStyle = "Flat"; $f.Controls.Add($btnBrowseIcon)

    $lblHint = New-Object System.Windows.Forms.Label; $lblHint.Text = "Hint: Use `"%1`" for the selected file.`nExample: `"C:\Apps\App.exe`" `"%1`""; $lblHint.Location = "20, 240"; $lblHint.AutoSize = $true; $lblHint.ForeColor = "Gray"; $f.Controls.Add($lblHint)

    # --- BUTTONS ---
    $btnApply = New-Object System.Windows.Forms.Button; $btnApply.Text = "Apply to Menu"; $btnApply.Location = "20, 290"; $btnApply.Width = 160; $btnApply.Height = 35; $btnApply.BackColor = "SeaGreen"; $btnApply.ForeColor = "White"; $btnApply.FlatStyle = "Flat"; $f.Controls.Add($btnApply)
    
    $btnRemove = New-Object System.Windows.Forms.Button; $btnRemove.Text = "Remove from Menu"; $btnRemove.Location = "200, 290"; $btnRemove.Width = 160; $btnRemove.Height = 35; $btnRemove.BackColor = "IndianRed"; $btnRemove.ForeColor = "White"; $btnRemove.FlatStyle = "Flat"; $f.Controls.Add($btnRemove)
    
    $btnClose = New-Object System.Windows.Forms.Button; $btnClose.Text = "Cancel"; $btnClose.Location = "460, 290"; $btnClose.Width = 100; $btnClose.Height = 35; $btnClose.BackColor = "DimGray"; $btnClose.ForeColor = "White"; $btnClose.FlatStyle = "Flat"; $f.Controls.Add($btnClose)

    # --- LOGIC ---
    $f.AcceptButton = $btnApply 
    $f.CancelButton = $btnClose

    $btnBrowseCmd.Add_Click({
        $dlg = New-Object System.Windows.Forms.OpenFileDialog; $dlg.Filter = "Programs|*.exe;*.bat;*.cmd|All Files|*.*"
        if ($dlg.ShowDialog() -eq "OK") { $txtCmd.Text = "`"$($dlg.FileName)`" `"%1`"" }
    })
    $btnBrowseIcon.Add_Click({
        $dlg = New-Object System.Windows.Forms.OpenFileDialog; $dlg.Filter = "Icons|*.ico;*.exe;*.dll|All Files|*.*"
        if ($dlg.ShowDialog() -eq "OK") { $txtIcon.Text = $dlg.FileName }
    })

    $btnApply.Add_Click({
        if ([string]::IsNullOrWhiteSpace($txtName.Text) -or [string]::IsNullOrWhiteSpace($txtCmd.Text)) { [System.Windows.Forms.MessageBox]::Show("Name and Command are required."); return }
        
        $targets = @("HKCU:\Software\Classes\*\shell\SetDesktopWallpaper", "HKCU:\Software\Classes\Directory\shell\SetDesktopWallpaper")
        try {
            foreach ($key in $targets) {
                if (-not (Test-Path -LiteralPath $key)) { New-Item -Path $key -Force | Out-Null }
                Set-ItemProperty -LiteralPath $key -Name "MUIVerb" -Value $txtName.Text
                Set-ItemProperty -LiteralPath $key -Name "MultiSelectModel" -Value "Player"
                if (-not [string]::IsNullOrWhiteSpace($txtIcon.Text)) { Set-ItemProperty -LiteralPath $key -Name "Icon" -Value $txtIcon.Text }
                
                $cmdKey = Join-Path $key "command"
                if (-not (Test-Path -LiteralPath $cmdKey)) { New-Item -Path $cmdKey -Force | Out-Null }
                Set-Item -LiteralPath $cmdKey -Value $txtCmd.Text
            }
            [System.Windows.Forms.MessageBox]::Show("Context Menu updated successfully!", "Success", "OK", "Information")
            $f.Close()
        } catch { [System.Windows.Forms.MessageBox]::Show($_.Exception.Message) }
    })

    $btnRemove.Add_Click({
        if ([System.Windows.Forms.MessageBox]::Show("Remove the custom menu item?", "Confirm", "YesNo") -eq "Yes") {
            $targets = @("HKCU:\Software\Classes\*\shell\SetDesktopWallpaper", "HKCU:\Software\Classes\Directory\shell\SetDesktopWallpaper")
            foreach ($key in $targets) { if (Test-Path -LiteralPath $key) { Remove-Item -LiteralPath $key -Recurse -Force -ErrorAction SilentlyContinue } }
            [System.Windows.Forms.MessageBox]::Show("Item removed.", "Success")
            $f.Close()
        }
    })
    
    $btnClose.Add_Click({ $f.Close() })
    $f.ShowDialog() | Out-Null
}

# --- FIREWALL RULE DIALOG ---
function Show-RuleDialog {
    param($Title, $RuleObj=$null) 
    $f = New-Object System.Windows.Forms.Form
    $f.Text = $Title; $f.Size = "450, 450"; $f.StartPosition = "CenterScreen"; $f.BackColor = [System.Drawing.Color]::FromArgb(40,40,40); $f.ForeColor = "White"
    function New-Input { param($L, $Y, $V="", $Opts=$null)
        $lbl=New-Object System.Windows.Forms.Label; $lbl.Text=$L; $lbl.Top=$Y; $lbl.Left=20; $lbl.AutoSize=$true; $f.Controls.Add($lbl)
        if ($Opts) { $c=New-Object System.Windows.Forms.ComboBox; $c.DropDownStyle="DropDownList"; foreach($opt in $Opts){ [void]$c.Items.Add($opt) }; if($V){$c.SelectedItem=$V}else{$c.SelectedIndex=0} } 
        else { $c=New-Object System.Windows.Forms.TextBox; $c.Text=$V }
        $c.Top=$Y+20; $c.Left=20; $c.Width=380; $c.BackColor=[System.Drawing.Color]::FromArgb(60,60,60); $c.ForeColor="White"; $f.Controls.Add($c); return $c
    }
    $vName=""; $vDir="Inbound"; $vAct="Block"; $vProt="TCP"; $vPort=""
    if ($RuleObj) { $vName=$RuleObj.DisplayName; $vDir=$RuleObj.Direction; $vAct=$RuleObj.Action; $vProt=$RuleObj.Protocol; $vPort=$RuleObj.LocalPort }
    $iName = New-Input "Rule Name" 10 $vName
    if ($RuleObj) { $iName.ReadOnly=$true; $iName.BackColor=[System.Drawing.Color]::FromArgb(30,30,30) }
    $iDir = New-Input "Direction" 60 $vDir @("Inbound", "Outbound")
    $iAct = New-Input "Action" 110 $vAct @("Allow", "Block")
    $iProt = New-Input "Protocol" 160 $vProt @("TCP", "UDP", "Any")
    $iPort = New-Input "Local Port (e.g. 80)" 210 $vPort
    $btn = New-Object System.Windows.Forms.Button; $btn.Text="Save"; $btn.Top=300; $btn.Left=20; $btn.Width=380; $btn.Height=40; $btn.BackColor="SeaGreen"; $btn.ForeColor="White"; $btn.DialogResult="OK"; $f.Controls.Add($btn)
    $tip = New-Object System.Windows.Forms.ToolTip; $tip.SetToolTip($btn, "Confirm and save this firewall rule")
    if ($f.ShowDialog() -eq "OK") { return @{ Name=$iName.Text; Direction=$iDir.SelectedItem; Action=$iAct.SelectedItem; Protocol=$iProt.SelectedItem; Port=$iPort.Text } }
    return $null
}

# --- TASK SCHEDULER MANAGER (styled to match Firewall UI) ---
function Show-TaskManager {
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Task Scheduler Manager"
    $f.Size = "900, 600"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.ForeColor = [System.Drawing.Color]::White

    $dg = New-Object System.Windows.Forms.DataGridView
    $dg.Dock = "Top"
    $dg.Height = 450
    $dg.BackgroundColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.ForeColor = [System.Drawing.Color]::White
    $dg.AutoSizeColumnsMode = "Fill"
    $dg.SelectionMode = "FullRowSelect"
    $dg.MultiSelect = $false
    $dg.ReadOnly = $true
    $dg.RowHeadersVisible = $false
    $dg.AllowUserToAddRows = $false
    $dg.BorderStyle = "None"
    $dg.EnableHeadersVisualStyles = $false
    $dg.ColumnHeadersDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
    $dg.ColumnHeadersDefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.ColumnHeadersDefaultCellStyle.Padding = (New-Object System.Windows.Forms.Padding 4)
    $dg.ColumnHeadersBorderStyle = [System.Windows.Forms.DataGridViewHeaderBorderStyle]::Single
    $dg.ColumnHeadersHeight = 30
    $dg.DefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $dg.DefaultCellStyle.ForeColor = [System.Drawing.Color]::White
    $dg.DefaultCellStyle.SelectionBackColor = [System.Drawing.ColorTranslator]::FromHtml("#007ACC")
    $dg.DefaultCellStyle.SelectionForeColor = [System.Drawing.Color]::White
    $dg.AlternatingRowsDefaultCellStyle.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#252526")
    $dg.GridColor = [System.Drawing.ColorTranslator]::FromHtml("#333333")
    $f.Controls.Add($dg)

    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Dock = "Bottom"
    $pnl.Height = 80
    $pnl.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#1E1E1E")
    $f.Controls.Add($pnl)

    function New-StyledBtn ($Text, $X, $Color=$null) {
        $b = New-Object System.Windows.Forms.Button
        $b.Text = $Text
        $b.Top = 20; $b.Left = $X; $b.Width = 100; $b.Height = 35
        $b.FlatStyle = "Flat"
        $b.FlatAppearance.BorderSize = 1
        $b.FlatAppearance.BorderColor = [System.Drawing.ColorTranslator]::FromHtml("#444444")
        $b.ForeColor = [System.Drawing.Color]::White
        if ($Color) {
            $b.BackColor = [System.Drawing.ColorTranslator]::FromHtml($Color)
            $b.FlatAppearance.MouseOverBackColor = [System.Windows.Forms.ControlPaint]::Light($b.BackColor)
        } else {
            $b.BackColor = [System.Drawing.ColorTranslator]::FromHtml("#2D2D30")
            $b.FlatAppearance.MouseOverBackColor = [System.Drawing.ColorTranslator]::FromHtml("#3E3E42")
        }
        $pnl.Controls.Add($b)
        return $b
    }

    $btnRef = New-StyledBtn "Refresh" 20
    $btnEn  = New-StyledBtn "Enable" 130 "#006600"
    $btnDis = New-StyledBtn "Disable" 240 "#CCAA00"
    $btnDel = New-StyledBtn "Delete" 350 "#802020"

    $dg.Add_RowPrePaint({
        param($src, $e)
        $row = $src.Rows[$e.RowIndex]
        if ($row.Cells["State"].Value) {
            $state = $row.Cells["State"].Value.ToString()
            if ($state -eq "Running") {
                $row.DefaultCellStyle.ForeColor = [System.Drawing.ColorTranslator]::FromHtml("#00FF00")
            } elseif ($state -eq "Ready") {
                $row.DefaultCellStyle.ForeColor = [System.Drawing.ColorTranslator]::FromHtml("#FFFF00")
            } elseif ($state -eq "Disabled") {
                $row.DefaultCellStyle.ForeColor = [System.Drawing.ColorTranslator]::FromHtml("#FF3333")
            } else {
                $row.DefaultCellStyle.ForeColor = [System.Drawing.Color]::LightGray
            }
        }
    })

    $LoadTasks = {
        $tasks = Get-ScheduledTask | Select-Object TaskName, State, Author, TaskPath
        $dt = New-Object System.Data.DataTable
        $dt.Columns.Add("TaskName"); $dt.Columns.Add("State"); $dt.Columns.Add("Author"); $dt.Columns.Add("Path")
        foreach ($t in $tasks) {
            $r=$dt.NewRow()
            $r["TaskName"]=$t.TaskName; $r["State"]=$t.State; $r["Author"]=$t.Author; $r["Path"]=$t.TaskPath
            $dt.Rows.Add($r)
        }
        $dg.DataSource = $dt
        $dg.ClearSelection()
    }

    $btnRef.Add_Click({ & $LoadTasks })
    $btnEn.Add_Click({ if($dg.SelectedRows.Count -gt 0){ $n=$dg.SelectedRows[0].Cells["TaskName"].Value; Enable-ScheduledTask -TaskName $n -ErrorAction SilentlyContinue; & $LoadTasks } })
    $btnDis.Add_Click({ if($dg.SelectedRows.Count -gt 0){ $n=$dg.SelectedRows[0].Cells["TaskName"].Value; Disable-ScheduledTask -TaskName $n -ErrorAction SilentlyContinue; & $LoadTasks } })
    $btnDel.Add_Click({ if($dg.SelectedRows.Count -gt 0){ $n=$dg.SelectedRows[0].Cells["TaskName"].Value; if([System.Windows.Forms.MessageBox]::Show("Delete $n?","Confirm",[System.Windows.Forms.MessageBoxButtons]::YesNo) -eq "Yes"){ Unregister-ScheduledTask -TaskName $n -Confirm:$false; & $LoadTasks } } })
    & $LoadTasks; $f.ShowDialog()
}

# ==========================================
# 3. XAML GUI
# ==========================================
[xml]$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
        Title="Windows Maintenance Tool v$AppVersion" Height="820" Width="1280" MinHeight="620" MinWidth="960"
        WindowStartupLocation="CenterScreen" Background="#0D1117" Foreground="#E6EDF3"
        FontFamily="Segoe UI Variable Display, Segoe UI, Arial" FontSize="13"
        TextOptions.TextFormattingMode="Display"
        TextOptions.TextRenderingMode="ClearType"
        UseLayoutRounding="True"
        SnapsToDevicePixels="True">

    <Window.Resources>
        <!-- Modern Color Palette (GitHub Dark inspired) -->
        <SolidColorBrush x:Key="BgDark" Color="#0D1117"/>
        <SolidColorBrush x:Key="BgPanel" Color="#161B22"/>
        <SolidColorBrush x:Key="BgElevated" Color="#21262D"/>
        <SolidColorBrush x:Key="BgHover" Color="#30363D"/>
        <SolidColorBrush x:Key="BorderBrush" Color="#30363D"/>
        <SolidColorBrush x:Key="BorderAccent" Color="#58A6FF"/>
        <SolidColorBrush x:Key="Accent" Color="#58A6FF"/>
        <SolidColorBrush x:Key="AccentHover" Color="#79C0FF"/>
        <SolidColorBrush x:Key="TextPrimary" Color="#E6EDF3"/>
        <SolidColorBrush x:Key="TextSecondary" Color="#8B949E"/>
        <SolidColorBrush x:Key="TextMuted" Color="#6E7681"/>
        <SolidColorBrush x:Key="Success" Color="#238636"/>
        <SolidColorBrush x:Key="SuccessHover" Color="#2EA043"/>
        <SolidColorBrush x:Key="Danger" Color="#DA3633"/>
        <SolidColorBrush x:Key="DangerHover" Color="#F85149"/>
        <SolidColorBrush x:Key="Warning" Color="#D29922"/>
        <SolidColorBrush x:Key="Info" Color="#1F6FEB"/>

        <!-- Subtle Shadow Effects (reduced for clarity) -->
        <DropShadowEffect x:Key="CardShadow" ShadowDepth="1" BlurRadius="4" Opacity="0.15" Color="#000000"/>

        <!-- Modern TextBox (crisp text) -->
        <Style TargetType="TextBox">
            <Setter Property="Background" Value="#0D1117"/>
            <Setter Property="Foreground" Value="#E6EDF3"/>
            <Setter Property="BorderBrush" Value="#30363D"/>
            <Setter Property="BorderThickness" Value="1"/>
            <Setter Property="Padding" Value="10,6"/>
            <Setter Property="FontSize" Value="13"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="UseLayoutRounding" Value="True"/>
            <Setter Property="TextOptions.TextFormattingMode" Value="Display"/>
            <Setter Property="TextOptions.TextRenderingMode" Value="ClearType"/>
            <Style.Triggers>
                <Trigger Property="IsFocused" Value="True">
                    <Setter Property="BorderBrush" Value="#58A6FF"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Modern Navigation Button (crisp text) -->
        <Style TargetType="Button" x:Key="NavBtn">
            <Setter Property="Background" Value="Transparent"/>
            <Setter Property="Foreground" Value="#8B949E"/>
            <Setter Property="BorderThickness" Value="0"/>
            <Setter Property="Height" Value="38"/>
            <Setter Property="Margin" Value="2"/>
            <Setter Property="HorizontalContentAlignment" Value="Left"/>
            <Setter Property="Padding" Value="12,0"/>
            <Setter Property="FontSize" Value="13"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="UseLayoutRounding" Value="True"/>
            <Setter Property="TextOptions.TextFormattingMode" Value="Display"/>
            <Setter Property="TextOptions.TextRenderingMode" Value="ClearType"/>
            <Setter Property="Template">
                <Setter.Value>
                    <ControlTemplate TargetType="Button">
                        <Grid>
                            <Border Name="Bd" Background="{TemplateBinding Background}" CornerRadius="4" UseLayoutRounding="True">
                                <ContentPresenter VerticalAlignment="Center" Margin="{TemplateBinding Padding}"/>
                            </Border>
                            <!-- Active Indicator Bar (left side) -->
                            <Border Name="Indicator" Width="3" Background="#58A6FF" HorizontalAlignment="Left" 
                                    CornerRadius="2,0,0,2" Visibility="{TemplateBinding Tag}" UseLayoutRounding="True"/>
                        </Grid>
                        <ControlTemplate.Triggers>
                            <Trigger Property="IsMouseOver" Value="True">
                                <Setter TargetName="Bd" Property="Background" Value="#21262D"/>
                                <Setter Property="Foreground" Value="#E6EDF3"/>
                            </Trigger>
                        </ControlTemplate.Triggers>
                    </ControlTemplate>
                </Setter.Value>
            </Setter>
        </Style>

        <!-- Modern Action Button (clean, no blur) -->
        <Style TargetType="Button" x:Key="ActionBtn">
            <Setter Property="Background" Value="#21262D"/>
            <Setter Property="Foreground" Value="#E6EDF3"/>
            <Setter Property="Height" Value="32"/>
            <Setter Property="Margin" Value="4"/>
            <Setter Property="FontSize" Value="12"/>
            <Setter Property="FontWeight" Value="Normal"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="UseLayoutRounding" Value="True"/>
            <Setter Property="TextOptions.TextFormattingMode" Value="Display"/>
            <Setter Property="TextOptions.TextRenderingMode" Value="ClearType"/>
            <Setter Property="Template">
                <Setter.Value>
                    <ControlTemplate TargetType="Button">
                        <Border Name="Bd" Background="{TemplateBinding Background}" CornerRadius="4" 
                                BorderBrush="#30363D" BorderThickness="1">
                            <ContentPresenter HorizontalAlignment="Center" VerticalAlignment="Center" Margin="16,0"/>
                        </Border>
                        <ControlTemplate.Triggers>
                            <Trigger Property="IsMouseOver" Value="True">
                                <Setter TargetName="Bd" Property="Background" Value="#30363D"/>
                                <Setter TargetName="Bd" Property="BorderBrush" Value="#8B949E"/>
                            </Trigger>
                            <Trigger Property="IsPressed" Value="True">
                                <Setter TargetName="Bd" Property="Background" Value="#161B22"/>
                            </Trigger>
                            <Trigger Property="IsEnabled" Value="False">
                                <Setter TargetName="Bd" Property="Background" Value="#161B22"/>
                                <Setter TargetName="Bd" Property="Opacity" Value="0.5"/>
                                <Setter Property="Foreground" Value="#6E7681"/>
                            </Trigger>
                        </ControlTemplate.Triggers>
                    </ControlTemplate>
                </Setter.Value>
            </Setter>
        </Style>

        <!-- Success/Green Button -->
        <Style TargetType="Button" x:Key="PositiveBtn" BasedOn="{StaticResource ActionBtn}">
            <Setter Property="Background" Value="#238636"/>
            <Setter Property="BorderBrush" Value="#2EA043"/>
            <Setter Property="Foreground" Value="#FFFFFF"/>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#2EA043"/>
                    <Setter Property="BorderBrush" Value="#3FB950"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Danger/Red Button -->
        <Style TargetType="Button" x:Key="DestructiveBtn" BasedOn="{StaticResource ActionBtn}">
            <Setter Property="Background" Value="#DA3633"/>
            <Setter Property="BorderBrush" Value="#F85149"/>
            <Setter Property="Foreground" Value="#FFFFFF"/>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#F85149"/>
                    <Setter Property="BorderBrush" Value="#FF7B72"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Warning/Yellow Button -->
        <Style TargetType="Button" x:Key="WarningBtn" BasedOn="{StaticResource ActionBtn}">
            <Setter Property="Background" Value="#D29922"/>
            <Setter Property="BorderBrush" Value="#E3B341"/>
            <Setter Property="Foreground" Value="#0D1117"/>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#E3B341"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Info/Blue Button -->
        <Style TargetType="Button" x:Key="UtilityBtn" BasedOn="{StaticResource ActionBtn}">
            <Setter Property="Background" Value="#1F6FEB"/>
            <Setter Property="BorderBrush" Value="#58A6FF"/>
            <Setter Property="Foreground" Value="#FFFFFF"/>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#58A6FF"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Accent Button -->
        <Style TargetType="Button" x:Key="AccentBtn" BasedOn="{StaticResource ActionBtn}">
            <Setter Property="Background" Value="#58A6FF"/>
            <Setter Property="BorderBrush" Value="#79C0FF"/>
            <Setter Property="Foreground" Value="#0D1117"/>
            <Setter Property="FontWeight" Value="Bold"/>
            <Style.Triggers>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#79C0FF"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Modern ListView Item (crisp text) -->
        <Style x:Key="FwItem" TargetType="ListViewItem">
            <Setter Property="BorderThickness" Value="0"/>
            <Setter Property="Padding" Value="12,6"/>
            <Setter Property="Margin" Value="2,1"/>
            <Setter Property="Background" Value="Transparent"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="UseLayoutRounding" Value="True"/>
            <Setter Property="TextOptions.TextFormattingMode" Value="Display"/>
            <Setter Property="TextOptions.TextRenderingMode" Value="ClearType"/>
            <Style.Triggers>
                <Trigger Property="ItemsControl.AlternationIndex" Value="0">
                    <Setter Property="Background" Value="#161B22"/>
                </Trigger>
                <Trigger Property="ItemsControl.AlternationIndex" Value="1">
                    <Setter Property="Background" Value="#0D1117"/>
                </Trigger>
                <Trigger Property="IsSelected" Value="True">
                    <Setter Property="Background" Value="#1F6FEB"/>
                    <Setter Property="Foreground" Value="#FFFFFF"/>
                    <Setter Property="FontWeight" Value="Medium"/>
                </Trigger>
                <Trigger Property="IsMouseOver" Value="True">
                    <Setter Property="Background" Value="#30363D"/>
                </Trigger>
            </Style.Triggers>
        </Style>

        <!-- Card Style Border (clean, crisp) -->
        <Style x:Key="CardStyle" TargetType="Border">
            <Setter Property="Background" Value="#161B22"/>
            <Setter Property="BorderBrush" Value="#30363D"/>
            <Setter Property="BorderThickness" Value="1"/>
            <Setter Property="CornerRadius" Value="4"/>
            <Setter Property="Margin" Value="6"/>
            <Setter Property="Padding" Value="16"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="UseLayoutRounding" Value="True"/>
        </Style>

        <!-- Section Header Text (crisp) -->
        <Style x:Key="SectionHeader" TargetType="TextBlock">
            <Setter Property="FontSize" Value="26"/>
            <Setter Property="FontWeight" Value="SemiBold"/>
            <Setter Property="Foreground" Value="#E6EDF3"/>
            <Setter Property="Margin" Value="0,0,0,16"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="TextOptions.TextFormattingMode" Value="Display"/>
            <Setter Property="TextOptions.TextRenderingMode" Value="ClearType"/>
        </Style>

        <!-- Subsection Header (crisp) -->
        <Style x:Key="SubHeader" TargetType="TextBlock">
            <Setter Property="FontSize" Value="11"/>
            <Setter Property="FontWeight" Value="Bold"/>
            <Setter Property="Foreground" Value="#8B949E"/>
            <Setter Property="Margin" Value="0,0,0,10"/>
            <Setter Property="FontFamily" Value="Segoe UI, Arial"/>
            <Setter Property="SnapsToDevicePixels" Value="True"/>
            <Setter Property="TextOptions.TextFormattingMode" Value="Display"/>
        </Style>
    </Window.Resources>

    <Grid>
        <Grid.ColumnDefinitions>
            <ColumnDefinition Width="300"/>
            <ColumnDefinition Width="*"/>
        </Grid.ColumnDefinitions>

        <!-- Sidebar -->
        <Border Grid.Column="0" Background="{StaticResource BgPanel}" BorderBrush="{StaticResource BorderBrush}" BorderThickness="0,0,1,0">
            <Grid>
                <Grid.RowDefinitions>
                    <RowDefinition Height="Auto"/> 
                    <RowDefinition Height="Auto"/> 
                    <RowDefinition Height="*"/>    
                    <RowDefinition Height="6"/>
                    <RowDefinition Height="240"/>
                </Grid.RowDefinitions>

                <!-- Header/Search -->
                <Border Name="bdQuickFind" Grid.Row="0" Background="{StaticResource BgDark}" Margin="16,20,16,12" CornerRadius="8" BorderBrush="{StaticResource BorderBrush}" BorderThickness="1" Cursor="IBeam">
                    <StackPanel Margin="12">
                        <TextBlock Text="Quick Find" FontSize="11" Foreground="{StaticResource TextMuted}" FontWeight="SemiBold" Margin="0,0,0,8"/>
                        <TextBox Name="txtGlobalSearch" Height="36" ToolTip="Search any function..." VerticalContentAlignment="Center"
                                 Background="{StaticResource BgPanel}" Foreground="{StaticResource TextPrimary}"
                                 BorderBrush="{StaticResource BorderBrush}" BorderThickness="1" Padding="10,0"/>
                    </StackPanel>
                </Border>

                <!-- Navigation -->
                <StackPanel Grid.Row="1" Margin="8,8,8,0">
                    <TextBlock Text="NAVIGATION" Style="{StaticResource SubHeader}" Margin="12,0,0,4"/>
                    <StackPanel Name="pnlNavButtons">
                        <Button Name="btnTabUpdates" Content="Updates" Style="{StaticResource NavBtn}" Tag="pnlUpdates"/>
                        <Button Name="btnTabTweaks" Content="Tweaks" Style="{StaticResource NavBtn}" Tag="pnlTweaks"/>
                        <Button Name="btnTabHealth" Content="System Health" Style="{StaticResource NavBtn}" Tag="pnlHealth"/>
                        <Button Name="btnTabNetwork" Content="Network &amp; DNS" Style="{StaticResource NavBtn}" Tag="pnlNetwork"/>
                        <Button Name="btnTabFirewall" Content="Firewall" Style="{StaticResource NavBtn}" Tag="pnlFirewall"/>
                        <Button Name="btnTabDrivers" Content="Drivers" Style="{StaticResource NavBtn}" Tag="pnlDrivers"/>
                        <Button Name="btnTabCleanup" Content="Cleanup" Style="{StaticResource NavBtn}" Tag="pnlCleanup"/>
                        <Button Name="btnTabUtils" Content="Utilities" Style="{StaticResource NavBtn}" Tag="pnlUtils"/>
                        <Button Name="btnTabSupport" Content="Support" Style="{StaticResource NavBtn}" Tag="pnlSupport"/>
                        <Button Name="btnNavDownloads" Content="Download Stats" Style="{StaticResource NavBtn}" ToolTip="Show latest release download counts"/>
                    </StackPanel>
                </StackPanel>
                
                <ListBox Name="lstSearchResults" Grid.Row="2" Background="{StaticResource BgDark}" BorderThickness="0" Foreground="{StaticResource Accent}" Visibility="Collapsed" Margin="8"/>

                <GridSplitter Grid.Row="3" Height="8" Margin="12,0" HorizontalAlignment="Stretch" VerticalAlignment="Center"
                              Background="Transparent" Cursor="SizeNS" ResizeDirection="Rows" ResizeBehavior="PreviousAndNext"
                              ShowsPreview="True"/>

                <!-- Log Panel -->
                <Border Grid.Row="4" Background="{StaticResource BgDark}" Margin="12" CornerRadius="8" BorderBrush="{StaticResource BorderBrush}" BorderThickness="1" MinHeight="140">
                    <Grid>
                        <Grid.RowDefinitions>
                            <RowDefinition Height="Auto"/>
                            <RowDefinition Height="*"/>
                        </Grid.RowDefinitions>
                        <Border Grid.Row="0" Background="{StaticResource BgPanel}" CornerRadius="8,8,0,0" Padding="12,8">
                            <TextBlock Text="Activity Log" FontSize="11" Foreground="{StaticResource TextMuted}" FontWeight="SemiBold"/>
                        </Border>
                        <ScrollViewer Name="svLog" Grid.Row="1" VerticalScrollBarVisibility="Auto" HorizontalScrollBarVisibility="Disabled" Margin="8" UseLayoutRounding="True">
                            <TextBox Name="LogBox" IsReadOnly="True" TextWrapping="Wrap" FontFamily="Consolas, monospace" FontSize="12" 
                                     Background="Transparent" Foreground="#3FB950" BorderThickness="0" Padding="4"
                                     SnapsToDevicePixels="True" TextOptions.TextFormattingMode="Display"/>
                        </ScrollViewer>
                    </Grid>
                </Border>
            </Grid>
        </Border>

        <Border Grid.Column="1" Background="{StaticResource BgDark}">
            <Grid Margin="20">
                
                <!-- UPDATES PANEL -->
                <Grid Name="pnlUpdates" Visibility="Visible">
                    <Grid.RowDefinitions>
                        <RowDefinition Height="Auto"/>
                        <RowDefinition Height="*"/>
                        <RowDefinition Height="Auto"/>
                    </Grid.RowDefinitions>
                    
                    <!-- Header Card -->
                    <Border Grid.Row="0" Style="{StaticResource CardStyle}" Margin="0,0,0,12">
                        <Grid>
                            <Grid.ColumnDefinitions>
                                <ColumnDefinition Width="*"/>
                                <ColumnDefinition Width="320"/>
                            </Grid.ColumnDefinitions>
                            <StackPanel VerticalAlignment="Center">
                                <StackPanel>
                                    <TextBlock Name="lblWingetTitle" Text="Package Updates" Style="{StaticResource SectionHeader}" Margin="0"/>
                                    <TextBlock Name="lblWingetStatus" Text="Ready to scan" Foreground="#D29922" FontSize="13" Visibility="Visible"/>
                                    <StackPanel Orientation="Horizontal" Margin="0,8,0,0" VerticalAlignment="Center">
                                        <ProgressBar Name="pbWingetProgress" Width="260" Height="8" Minimum="0" Maximum="100" Value="0" Visibility="Collapsed"/>
                                        <TextBlock Name="lblWingetProgress" Text="" Margin="10,0,0,0" Foreground="{StaticResource TextMuted}" FontSize="12" Visibility="Collapsed"/>
                                    </StackPanel>
                                </StackPanel>
                            </StackPanel>
                            <Grid Grid.Column="1">
                                <Grid.ColumnDefinitions>
                                    <ColumnDefinition Width="*"/>
                                    <ColumnDefinition Width="Auto"/>
                                </Grid.ColumnDefinitions>
                                <TextBox Name="txtWingetSearch" Grid.Column="0" Height="40" VerticalContentAlignment="Center" Text="Search packages..."/>
                                <Button Name="btnWingetFind" Grid.Column="1" Content="Search" Width="100" Height="40" Margin="8,0,0,0" Style="{StaticResource AccentBtn}"/>
                            </Grid>
                        </Grid>
                    </Border>

                    <!-- List Card -->
                    <Border Grid.Row="1" Style="{StaticResource CardStyle}" Padding="0">
                        <ListView Name="lstWinget" Background="Transparent" Foreground="{StaticResource TextPrimary}" BorderThickness="0" 
                                  SelectionMode="Extended" AlternationCount="2" ItemContainerStyle="{StaticResource FwItem}">
                            <ListView.View>
                                <GridView>
                                    <GridViewColumn Header="Source" Width="90" DisplayMemberBinding="{Binding Source}"/>
                                    <GridViewColumn Header="Package Name" Width="280" DisplayMemberBinding="{Binding Name}"/>
                                    <GridViewColumn Header="ID" Width="220" DisplayMemberBinding="{Binding Id}"/>
                                    <GridViewColumn Header="Installed" Width="110" DisplayMemberBinding="{Binding Version}"/>
                                    <GridViewColumn Header="Latest" Width="110" DisplayMemberBinding="{Binding Available}"/>
                                </GridView>
                            </ListView.View>
                        </ListView>
                    </Border>

                    <!-- Action Bar -->
                    <Border Grid.Row="2" Style="{StaticResource CardStyle}" Margin="0,12,0,0">
                        <StackPanel Orientation="Horizontal" HorizontalAlignment="Right">
                            <Button Name="btnManageProviders" Content="Providers" Style="{StaticResource ActionBtn}" ToolTip="Manage package sources"/>
                            <Button Name="btnShowCatalog" Content="Software Catalog" Style="{StaticResource ActionBtn}" ToolTip="Browse our curated catalog of popular free applications. Install multiple apps at once with one click."/>
                            <Button Name="btnWingetScan" Content="Refresh All" Style="{StaticResource AccentBtn}"/>
                            <Button Name="btnWingetUpdateSel" Content="Update Selected" Style="{StaticResource PositiveBtn}"/>
                            <Button Name="btnWingetInstall" Content="Install" Style="{StaticResource PositiveBtn}" Visibility="Collapsed"/>
                            <Button Name="btnWingetUninstall" Content="Uninstall" Style="{StaticResource DestructiveBtn}"/>
                            <Button Name="btnWingetIgnore" Content="Ignore" Style="{StaticResource WarningBtn}"/>
                            <Button Name="btnWingetUnignore" Content="Manage Ignored" Style="{StaticResource ActionBtn}"/>
                        </StackPanel>
                    </Border>
                </Grid>

                <!-- SOFTWARE CATALOG PANEL (Initially Hidden) -->
                <Grid Name="pnlCatalog" Visibility="Collapsed">
                    <Grid.RowDefinitions>
                        <RowDefinition Height="Auto"/>
                        <RowDefinition Height="Auto"/>
                        <RowDefinition Height="*"/>
                        <RowDefinition Height="Auto"/>
                    </Grid.RowDefinitions>
                    
                    <!-- Header -->
                    <Border Grid.Row="0" Style="{StaticResource CardStyle}" Margin="0,0,0,12">
                        <Grid>
                            <Grid.ColumnDefinitions>
                                <ColumnDefinition Width="*"/>
                                <ColumnDefinition Width="300"/>
                            </Grid.ColumnDefinitions>
                            <StackPanel>
                                <TextBlock Text="Software Catalog" Style="{StaticResource SectionHeader}" Margin="0"/>
                                <TextBlock Text="Curated selection of popular applications" Foreground="{StaticResource TextSecondary}" FontSize="13"/>
                            </StackPanel>
                            <Grid Grid.Column="1">
                                <Grid.ColumnDefinitions>
                                    <ColumnDefinition Width="*"/>
                                    <ColumnDefinition Width="Auto"/>
                                </Grid.ColumnDefinitions>
                                <TextBox Name="txtCatalogSearch" Grid.Column="0" Height="40" VerticalContentAlignment="Center" Text="Search catalog..." ToolTip="Type to filter applications by name or description"/>
                                <Button Name="btnCatalogSearch" Grid.Column="1" Content="Search" Width="100" Height="40" Margin="8,0,0,0" Style="{StaticResource AccentBtn}" ToolTip="Search the catalog"/>
                            </Grid>
                        </Grid>
                    </Border>

                    <!-- Category Filter -->
                    <StackPanel Grid.Row="1" Orientation="Horizontal" Margin="6,0,6,12">
                        <Button Name="btnCatAll" Content="All" Width="80" Style="{StaticResource AccentBtn}" ToolTip="Show all applications"/>
                        <Button Name="btnCatBrowsers" Content="Browsers" Width="90" Style="{StaticResource ActionBtn}" ToolTip="Filter: Web browsers"/>
                        <Button Name="btnCatDev" Content="Development" Width="100" Style="{StaticResource ActionBtn}" ToolTip="Filter: Developer tools, IDEs, runtimes"/>
                        <Button Name="btnCatUtils" Content="Utilities" Width="90" Style="{StaticResource ActionBtn}" ToolTip="Filter: System utilities and tools"/>
                        <Button Name="btnCatMedia" Content="Multimedia" Width="100" Style="{StaticResource ActionBtn}" ToolTip="Filter: Media players, editors, streaming"/>
                        <Button Name="btnCatGames" Content="Gaming" Width="80" Style="{StaticResource ActionBtn}" ToolTip="Filter: Game platforms and gaming tools"/>
                        <Button Name="btnCatSecurity" Content="Security" Width="90" Style="{StaticResource ActionBtn}" ToolTip="Filter: Antivirus, password managers, security tools"/>
                    </StackPanel>

                    <!-- Catalog List -->
                    <Border Grid.Row="2" Style="{StaticResource CardStyle}" Padding="0">
                        <ListView Name="lstCatalog" Background="Transparent" Foreground="{StaticResource TextPrimary}" BorderThickness="0" 
                                  SelectionMode="Extended" AlternationCount="2" ItemContainerStyle="{StaticResource FwItem}">
                            <ListView.View>
                                <GridView>
                                    <GridViewColumn Header="Category" Width="100" DisplayMemberBinding="{Binding Category}"/>
                                    <GridViewColumn Header="Name" Width="200" DisplayMemberBinding="{Binding Name}"/>
                                    <GridViewColumn Header="Description" Width="350" DisplayMemberBinding="{Binding Description}"/>
                                    <GridViewColumn Header="Source" Width="80" DisplayMemberBinding="{Binding Source}"/>
                                </GridView>
                            </ListView.View>
                        </ListView>
                    </Border>

                    <!-- Actions -->
                    <Border Grid.Row="3" Style="{StaticResource CardStyle}" Margin="0,12,0,0">
                        <StackPanel Orientation="Horizontal" HorizontalAlignment="Right">
                            <Button Name="btnBackToUpdates" Content="Back to Updates" Style="{StaticResource ActionBtn}" ToolTip="Return to the package updates view"/>
                            <Button Name="btnCatalogInstall" Content="Install Selected" Style="{StaticResource PositiveBtn}" ToolTip="Install all selected applications using winget. May take several minutes depending on app size."/>
                            <Button Name="btnCatalogSelectAll" Content="Select All" Style="{StaticResource ActionBtn}" ToolTip="Select all visible applications in the list"/>
                            <Button Name="btnCatalogClear" Content="Clear Selection" Style="{StaticResource ActionBtn}" ToolTip="Unselect all applications"/>
                        </StackPanel>
                    </Border>
                </Grid>

                <!-- TWEAKS PANEL -->
                <ScrollViewer Name="pnlTweaks" Visibility="Collapsed" VerticalScrollBarVisibility="Auto" HorizontalScrollBarVisibility="Disabled">
                    <StackPanel>
                    <!-- Performance Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <StackPanel Margin="0,0,0,12">
                                <TextBlock Text="Performance Tweaks" Style="{StaticResource SectionHeader}" Margin="0"/>
                            </StackPanel>
                            <TextBlock Text="POWER &amp; PERFORMANCE" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnPerfServicesManual" Content="Services to Manual" Style="{StaticResource ActionBtn}" ToolTip="Optimize 100+ Windows services by setting them to Manual startup. Improves boot time and reduces background RAM usage."/>
                                <Button Name="btnPerfServicesRevert" Content="Revert Services" Style="{StaticResource WarningBtn}" ToolTip="Restore all services to their default startup type (Automatic/Manual/Disabled). Use this if you experience issues after optimization."/>
                                <Button Name="btnPerfDisableHibernate" Content="Disable Hibernation" Style="{StaticResource ActionBtn}" ToolTip="Disable hibernation and delete hiberfil.sys. Frees up several GB of disk space equal to your RAM size."/>
                                <Button Name="btnPerfEnableHibernate" Content="Enable Hibernation" Style="{StaticResource ActionBtn}" ToolTip="Re-enable hibernation mode. Allows your PC to save state and power off completely, resuming faster than a full boot."/>
                                <Button Name="btnPerfDisableSuperfetch" Content="Disable Superfetch" Style="{StaticResource ActionBtn}" ToolTip="Disable SysMain (Superfetch) service. Prevents Windows from pre-loading apps into RAM. Can help on systems with low RAM or SSDs."/>
                                <Button Name="btnPerfEnableSuperfetch" Content="Enable Superfetch" Style="{StaticResource ActionBtn}" ToolTip="Enable SysMain (Superfetch) service. Pre-loads frequently used apps into RAM for faster launch times on HDDs."/>
                                <Button Name="btnPerfDisableMemCompress" Content="Disable Mem Compression" Style="{StaticResource ActionBtn}" ToolTip="Disable memory compression. RAM stores data uncompressed. May improve performance on high-RAM systems."/>
                                <Button Name="btnPerfEnableMemCompress" Content="Enable Mem Compression" Style="{StaticResource ActionBtn}" ToolTip="Enable memory compression. Windows compresses inactive RAM pages to free up physical memory for active apps."/>
                                <Button Name="btnPerfUltimatePower" Content="Ultimate Performance" Style="{StaticResource PositiveBtn}" ToolTip="Enable the Ultimate Performance power plan. Removes all power throttling for maximum performance. Best for desktops and high-performance laptops."/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- AppX Bloatware Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="APPX BLOATWARE REMOVAL" Style="{StaticResource SubHeader}" ToolTip="Remove pre-installed Windows apps (UWP/Modern apps) that you don't use. Frees disk space and reduces background processes."/>
                            <TextBlock Text="Select apps to remove (use Ctrl+Click for multiple)" Foreground="{StaticResource TextSecondary}" Margin="0,0,0,8"/>
                            <ListView Name="lstAppxPackages" Height="200" Background="{StaticResource BgDark}" Foreground="{StaticResource TextPrimary}" BorderBrush="{StaticResource BorderBrush}" BorderThickness="1" SelectionMode="Multiple">
                                <ListView.View>
                                    <GridView>
                                        <GridViewColumn Header="App Name" Width="250" DisplayMemberBinding="{Binding Name}"/>
                                        <GridViewColumn Header="Package" Width="300" DisplayMemberBinding="{Binding Package}"/>
                                    </GridView>
                                </ListView.View>
                            </ListView>
                            <WrapPanel Margin="0,12,0,0">
                                <Button Name="btnAppxLoad" Content="Load Apps" Style="{StaticResource ActionBtn}" ToolTip="Scan for installed UWP/Modern apps that can be removed. Populates the list above with removable bloatware."/>
                                <Button Name="btnAppxRemoveSel" Content="Remove Selected" Style="{StaticResource DestructiveBtn}" ToolTip="Remove the selected apps from your system. These apps can be reinstalled from the Microsoft Store if needed later."/>
                                <Button Name="btnAppxRemoveAll" Content="Remove All" Style="{StaticResource DestructiveBtn}" ToolTip="Remove ALL listed apps at once. This is faster but be careful - only click if you're sure you don't need any of these apps!"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- Windows Features Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="WINDOWS OPTIONAL FEATURES" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnFeatHyperV" Content="Hyper-V" Style="{StaticResource ActionBtn}" ToolTip="Microsoft's hardware virtualization platform. Create and run virtual machines. Requires Pro/Enterprise edition and CPU virtualization support."/>
                                <Button Name="btnFeatWSL" Content="WSL" Style="{StaticResource ActionBtn}" ToolTip="Windows Subsystem for Linux. Run Linux command-line tools and apps directly on Windows without a VM. Popular for developers."/>
                                <Button Name="btnFeatSandbox" Content="Sandbox" Style="{StaticResource ActionBtn}" ToolTip="Windows Sandbox - a lightweight, temporary desktop environment to safely run untrusted apps. Discards all changes when closed. Requires Pro/Enterprise."/>
                                <Button Name="btnFeatDotNet35" Content=".NET 3.5" Style="{StaticResource ActionBtn}" ToolTip=".NET Framework 3.5 (includes 2.0 and 3.0). Required for many older Windows applications and some games."/>
                                <Button Name="btnFeatNFS" Content="NFS Client" Style="{StaticResource ActionBtn}" ToolTip="Network File System client. Allows Windows to connect to Linux/UNIX NFS file shares and NAS devices."/>
                                <Button Name="btnFeatTelnet" Content="Telnet" Style="{StaticResource ActionBtn}" ToolTip="Telnet client for command-line remote connections. Not secure (unencrypted) - use SSH when possible."/>
                                <Button Name="btnFeatIIS" Content="IIS" Style="{StaticResource ActionBtn}" ToolTip="Internet Information Services - Microsoft's web server. Host websites locally, useful for web development."/>
                                <Button Name="btnFeatLegacy" Content="Legacy Media" Style="{StaticResource ActionBtn}" ToolTip="Windows Media Player and DirectPlay. Required for some older games and media playback."/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- Services Management Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="SERVICES MANAGEMENT" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnSvcOptimize" Content="Optimize Services" Style="{StaticResource PositiveBtn}" ToolTip="Set 100+ non-essential Windows services to Manual startup. Significantly improves boot time and reduces background resource usage."/>
                                <Button Name="btnSvcRestore" Content="Restore Defaults" Style="{StaticResource WarningBtn}" ToolTip="Restore ALL services to their original Windows default settings. Use this if you experience system issues after optimization."/>
                                <Button Name="btnSvcView" Content="View Services" Style="{StaticResource ActionBtn}" ToolTip="Open a grid view of all Windows services showing their current startup type and status. Useful for manual troubleshooting."/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- Scheduled Tasks Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="SCHEDULED TASKS" Style="{StaticResource SubHeader}"/>
                            <TextBlock Text="Disable telemetry and tracking tasks" Foreground="{StaticResource TextSecondary}" Margin="0,0,0,8"/>
                            <WrapPanel>
                                <Button Name="btnTasksDisableTelemetry" Content="Disable Telemetry Tasks" Style="{StaticResource DestructiveBtn}" ToolTip="Disable Windows telemetry scheduled tasks including: CEIP (Customer Experience), Error Reporting, Compatibility Appraiser. Reduces background activity and privacy concerns."/>
                                <Button Name="btnTasksRestore" Content="Restore Tasks" Style="{StaticResource WarningBtn}" ToolTip="Re-enable all telemetry and diagnostic scheduled tasks. Restores Windows default behavior for diagnostics and feedback."/>
                                <Button Name="btnTasksView" Content="View Tasks" Style="{StaticResource ActionBtn}" ToolTip="View telemetry-related scheduled tasks in a grid. Shows task name, path, and current enabled/disabled state."/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- Windows Update Presets Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="WINDOWS UPDATE PRESETS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnWUDefault" Content="Default" Style="{StaticResource ActionBtn}" ToolTip="Standard Windows Update behavior. Feature and security updates install automatically. Recommended for most users."/>
                                <Button Name="btnWUSecurity" Content="Security Only" Style="{StaticResource UtilityBtn}" ToolTip="Defer feature updates for 1 year, install security updates only. Get new features later while staying secure. Good for stability."/>
                                <Button Name="btnWUDisable" Content="Disable All" Style="{StaticResource DestructiveBtn}" ToolTip="Completely disable Windows Update. NOT RECOMMENDED - your system will become vulnerable to security threats. Use with caution."/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                    </StackPanel>
                </ScrollViewer>

                <!-- SYSTEM HEALTH PANEL -->
                <StackPanel Name="pnlHealth" Visibility="Collapsed">
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <StackPanel Margin="0,0,0,20">
                                <TextBlock Text="System Health" Style="{StaticResource SectionHeader}" Margin="0"/>
                            </StackPanel>
                            <TextBlock Text="WINDOWS REPAIR TOOLS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnSFC" Content="SFC Scan" Style="{StaticResource ActionBtn}" ToolTip="System File Checker - repairs corrupted system files"/>
                                <Button Name="btnDISMCheck" Content="DISM Check" Style="{StaticResource ActionBtn}" ToolTip="Check Windows image health"/>
                                <Button Name="btnDISMRestore" Content="DISM Restore" Style="{StaticResource UtilityBtn}" ToolTip="Repair Windows image"/>
                                <Button Name="btnCHKDSK" Content="CHKDSK" Style="{StaticResource ActionBtn}" ToolTip="Check disk for errors (requires reboot)"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                </StackPanel>

                <!-- NETWORK PANEL -->
                <StackPanel Name="pnlNetwork" Visibility="Collapsed">
                    <!-- Header Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="Network &amp; DNS" Style="{StaticResource SectionHeader}" Margin="0"/>
                        </StackPanel>
                    </Border>

                    <!-- General Tools Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="DIAGNOSTIC TOOLS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnNetInfo" Content="IP Config" Style="{StaticResource ActionBtn}" ToolTip="Display network configuration"/>
                                <Button Name="btnFlushDNS" Content="Flush DNS" Style="{StaticResource ActionBtn}" ToolTip="Clear DNS cache"/>
                                <Button Name="btnResetWifi" Content="Restart Wi-Fi" Style="{StaticResource ActionBtn}" ToolTip="Reset wireless adapters"/>
                                <Button Name="btnNetRepair" Content="Full Repair" Style="{StaticResource WarningBtn}" ToolTip="Comprehensive network reset"/>
                                <Button Name="btnRouteTable" Content="Save Routes" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnRouteView" Content="View Routes" Style="{StaticResource ActionBtn}"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- DNS Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="DNS SERVERS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnDnsGoogle" Content="Google DNS" Style="{StaticResource ActionBtn}" ToolTip="8.8.8.8 / 8.8.4.4"/>
                                <Button Name="btnDnsCloudflare" Content="Cloudflare" Style="{StaticResource ActionBtn}" ToolTip="1.1.1.1 / 1.0.0.1"/>
                                <Button Name="btnDnsQuad9" Content="Quad9" Style="{StaticResource ActionBtn}" ToolTip="9.9.9.9"/>
                                <Button Name="btnDnsAuto" Content="Auto (DHCP)" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnDnsCustom" Content="Custom..." Style="{StaticResource UtilityBtn}"/>
                            </WrapPanel>
                            
                            <TextBlock Text="DNS OVER HTTPS" Style="{StaticResource SubHeader}" Margin="0,16,0,8"/>
                            <WrapPanel>
                                <Button Name="btnDohAuto" Content="Enable DoH" Style="{StaticResource PositiveBtn}" ToolTip="Enable DNS encryption for all providers"/>
                                <Button Name="btnDohDisable" Content="Disable DoH" Style="{StaticResource DestructiveBtn}" ToolTip="Disable DNS encryption"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- Hosts File Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="HOSTS FILE MANAGER" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnHostsUpdate" Content="Download AdBlock" Style="{StaticResource PositiveBtn}" ToolTip="Download and merge ad-blocking hosts"/>
                                <Button Name="btnHostsEdit" Content="Edit Hosts" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnHostsBackup" Content="Backup" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnHostsRestore" Content="Restore" Style="{StaticResource ActionBtn}"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                </StackPanel>

                <!-- FIREWALL PANEL -->
                <Grid Name="pnlFirewall" Visibility="Collapsed">
                    <Grid.RowDefinitions>
                        <RowDefinition Height="Auto"/>
                        <RowDefinition Height="*"/>
                        <RowDefinition Height="Auto"/>
                    </Grid.RowDefinitions>
                    
                    <!-- Header Card -->
                    <Border Grid.Row="0" Style="{StaticResource CardStyle}" Margin="0,0,0,12">
                        <Grid>
                            <Grid.ColumnDefinitions>
                                <ColumnDefinition Width="*"/>
                                <ColumnDefinition Width="240"/>
                            </Grid.ColumnDefinitions>
                            <StackPanel>
                                <TextBlock Text="Firewall Manager" Style="{StaticResource SectionHeader}" Margin="0"/>
                                <TextBlock Name="lblFwStatus" Text="Ready" Foreground="#D29922" FontSize="13" Visibility="Collapsed"/>
                            </StackPanel>
                            <TextBox Name="txtFwSearch" Grid.Column="1" Text="Search rules..." ToolTip="Filter by name or port"/>
                        </Grid>
                    </Border>
                    
                    <!-- Rules List Card -->
                    <Border Grid.Row="1" Style="{StaticResource CardStyle}" Padding="0">
                        <ListView Name="lstFirewall" Background="Transparent" Foreground="{StaticResource TextPrimary}" BorderThickness="0" AlternationCount="2" ItemContainerStyle="{StaticResource FwItem}">
                            <ListView.View>
                                <GridView>
                                    <GridViewColumn Header="Rule Name" Width="360" DisplayMemberBinding="{Binding Name}"/>
                                    <GridViewColumn Header="Direction" Width="70" DisplayMemberBinding="{Binding Direction}"/>
                                    <GridViewColumn Header="Action" Width="70">
                                        <GridViewColumn.CellTemplate>
                                            <DataTemplate>
                                                <TextBlock Text="{Binding Action}" FontWeight="Bold">
                                                    <TextBlock.Style>
                                                        <Style TargetType="TextBlock">
                                                            <Setter Property="Foreground" Value="{StaticResource TextSecondary}"/>
                                                            <Style.Triggers>
                                                                <DataTrigger Binding="{Binding Action}" Value="Allow"><Setter Property="Foreground" Value="#3FB950"/></DataTrigger>
                                                                <DataTrigger Binding="{Binding Action}" Value="Block"><Setter Property="Foreground" Value="#F85149"/></DataTrigger>
                                                            </Style.Triggers>
                                                        </Style>
                                                    </TextBlock.Style>
                                                </TextBlock>
                                            </DataTemplate>
                                        </GridViewColumn.CellTemplate>
                                    </GridViewColumn>
                                    <GridViewColumn Header="Status" Width="70">
                                        <GridViewColumn.CellTemplate>
                                            <DataTemplate>
                                                <TextBlock Text="{Binding Enabled}" FontWeight="Bold">
                                                    <TextBlock.Style>
                                                        <Style TargetType="TextBlock">
                                                            <Setter Property="Foreground" Value="{StaticResource TextSecondary}"/>
                                                            <Style.Triggers>
                                                                <DataTrigger Binding="{Binding Enabled}" Value="True"><Setter Property="Foreground" Value="#3FB950"/></DataTrigger>
                                                                <DataTrigger Binding="{Binding Enabled}" Value="False"><Setter Property="Foreground" Value="#F85149"/></DataTrigger>
                                                            </Style.Triggers>
                                                        </Style>
                                                    </TextBlock.Style>
                                                </TextBlock>
                                            </DataTemplate>
                                        </GridViewColumn.CellTemplate>
                                    </GridViewColumn>
                                    <GridViewColumn Header="Protocol" Width="70" DisplayMemberBinding="{Binding Protocol}"/>
                                    <GridViewColumn Header="Port" Width="90" DisplayMemberBinding="{Binding LocalPort}"/>
                                </GridView>
                            </ListView.View>
                        </ListView>
                    </Border>

                    <!-- Actions Card -->
                    <Border Grid.Row="2" Style="{StaticResource CardStyle}" Margin="0,12,0,0">
                        <StackPanel Orientation="Horizontal" HorizontalAlignment="Right">
                            <Button Name="btnFwRefresh" Content="Reload" Style="{StaticResource ActionBtn}"/>
                            <Button Name="btnFwAdd" Content="Add Rule" Style="{StaticResource PositiveBtn}"/>
                            <Button Name="btnFwEdit" Content="Edit" Style="{StaticResource ActionBtn}"/>
                            <Button Name="btnFwEnable" Content="Enable" Style="{StaticResource ActionBtn}"/>
                            <Button Name="btnFwDisable" Content="Disable" Style="{StaticResource ActionBtn}"/>
                            <Button Name="btnFwDelete" Content="Delete" Style="{StaticResource DestructiveBtn}"/>
                            <Button Name="btnFwExport" Content="Export" Style="{StaticResource ActionBtn}"/>
                            <Button Name="btnFwImport" Content="Import" Style="{StaticResource ActionBtn}"/>
                            <Button Name="btnFwDefaults" Content="Defaults" Style="{StaticResource WarningBtn}"/>
                            <Button Name="btnFwPurge" Content="Delete All" Style="{StaticResource DestructiveBtn}"/>
                        </StackPanel>
                    </Border>
                </Grid>

                <!-- DRIVERS PANEL -->
                <StackPanel Name="pnlDrivers" Visibility="Collapsed">
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <StackPanel Margin="0,0,0,20">
                                <TextBlock Text="Driver Management" Style="{StaticResource SectionHeader}" Margin="0"/>
                            </StackPanel>
                            <TextBlock Text="DRIVER TOOLS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnDrvReport" Content="Generate Report" Style="{StaticResource ActionBtn}" ToolTip="Create detailed driver list"/>
                                <Button Name="btnDrvBackup" Content="Export Drivers" Style="{StaticResource ActionBtn}" ToolTip="Backup all drivers to folder"/>
                                <Button Name="btnDrvGhost" Content="Remove Ghosts" Style="{StaticResource WarningBtn}" ToolTip="Remove disconnected devices"/>
                                <Button Name="btnDrvClean" Content="Clean Old" Style="{StaticResource WarningBtn}" ToolTip="Remove old driver versions"/>
                                <Button Name="btnDrvRestore" Content="Restore" Style="{StaticResource ActionBtn}" ToolTip="Restore from backup"/>
                            </WrapPanel>
                            <TextBlock Text="WINDOWS UPDATE SETTINGS" Style="{StaticResource SubHeader}" Margin="0,16,0,8"/>
                            <WrapPanel>
                                <Button Name="btnDrvDisableWU" Content="Disable Auto-Drivers" Style="{StaticResource DestructiveBtn}" ToolTip="Stop Windows Update from installing drivers"/>
                                <Button Name="btnDrvEnableWU" Content="Enable Auto-Drivers" Style="{StaticResource PositiveBtn}" ToolTip="Allow Windows Update to install drivers"/>
                                <Button Name="btnDrvDisableMeta" Content="Disable Metadata" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnDrvEnableMeta" Content="Enable Metadata" Style="{StaticResource ActionBtn}"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                </StackPanel>

                <!-- CLEANUP PANEL -->
                <StackPanel Name="pnlCleanup" Visibility="Collapsed">
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <StackPanel Margin="0,0,0,20">
                                <TextBlock Text="System Cleanup" Style="{StaticResource SectionHeader}" Margin="0"/>
                            </StackPanel>
                            <TextBlock Text="CLEANUP TOOLS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnCleanDisk" Content="Disk Cleanup" Style="{StaticResource ActionBtn}" ToolTip="Windows built-in cleanup utility"/>
                                <Button Name="btnCleanTemp" Content="Delete Temp Files" Style="{StaticResource ActionBtn}" ToolTip="Clear temp folders"/>
                                <Button Name="btnCleanShortcuts" Content="Fix Shortcuts" Style="{StaticResource ActionBtn}" ToolTip="Remove broken shortcuts"/>
                                <Button Name="btnCleanReg" Content="Clean Registry" Style="{StaticResource WarningBtn}" ToolTip="Remove obsolete registry entries"/>
                                <Button Name="btnCleanXbox" Content="Clean Xbox Data" Style="{StaticResource ActionBtn}" ToolTip="Clear Xbox app cache"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                </StackPanel>

                <!-- UTILITIES PANEL -->
                <StackPanel Name="pnlUtils" Visibility="Collapsed">
                    <!-- System Tools Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <StackPanel Margin="0,0,0,20">
                                <TextBlock Text="System Tools" Style="{StaticResource SectionHeader}" Margin="0"/>
                            </StackPanel>
                            <TextBlock Text="SYSTEM INFO &amp; MAINTENANCE" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnUtilSysInfo" Content="System Report" Style="{StaticResource ActionBtn}" ToolTip="Generate detailed system report"/>
                                <Button Name="btnUtilTrim" Content="Trim SSD" Style="{StaticResource ActionBtn}" ToolTip="Optimize SSD performance"/>
                                <Button Name="btnUtilMas" Content="MAS Activation" Style="{StaticResource UtilityBtn}" ToolTip="Microsoft Activation Scripts"/>
                                <Button Name="btnTaskManager" Content="Task Scheduler" Style="{StaticResource ActionBtn}" ToolTip="Manage scheduled tasks"/>
                                <Button Name="btnCtxBuilder" Content="Context Menu" Style="{StaticResource ActionBtn}" ToolTip="Customize right-click menu"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>

                    <!-- Repairs Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="REPAIRS &amp; SETTINGS" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnUpdateRepair" Content="Reset Windows Update" Style="{StaticResource WarningBtn}" ToolTip="Fix Windows Update issues"/>
                                <Button Name="btnUpdateServices" Content="Restart Services" Style="{StaticResource ActionBtn}" ToolTip="Restart update-related services"/>
                                <Button Name="btnDotNetEnable" Content="Enable .NET RollForward" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnDotNetDisable" Content="Disable .NET RollForward" Style="{StaticResource ActionBtn}"/>
                                <Button Name="btnInstallGpedit" Content="Install Gpedit" Style="{StaticResource UtilityBtn}" ToolTip="Add Group Policy to Home editions"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                </StackPanel>

                <!-- SUPPORT PANEL -->
                <StackPanel Name="pnlSupport" Visibility="Collapsed">
                    <!-- Header Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <StackPanel Margin="0,0,0,12">
                                <TextBlock Text="Support &amp; Credits" Style="{StaticResource SectionHeader}" Margin="0"/>
                                <TextBlock Text="Windows Maintenance Tool v$AppVersion" FontSize="14" Foreground="{StaticResource TextSecondary}" FontWeight="SemiBold"/>
                            </StackPanel>
                        </StackPanel>
                    </Border>

                    <!-- Credits Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="CONTRIBUTORS" Style="{StaticResource SubHeader}"/>
                            <StackPanel Margin="0,8,0,0">
                                <StackPanel Orientation="Horizontal" Margin="0,4">
                                    <TextBlock Text="Author: " Foreground="{StaticResource TextSecondary}" Width="120"/>
                                    <Button Name="btnCreditLilBattiCLI" Content="Lil_Batti" Style="{StaticResource ActionBtn}" Height="26" Padding="8,2"/>
                                </StackPanel>
                                <StackPanel Orientation="Horizontal" Margin="0,4">
                                    <TextBlock Text="GUI &amp; Features: " Foreground="{StaticResource TextSecondary}" Width="120"/>
                                    <Button Name="btnCreditChaythonGUI" Content="Chaython" Style="{StaticResource ActionBtn}" Height="26" Padding="8,2"/>
                                </StackPanel>
                            </StackPanel>
                            <TextBlock Text="MIT License - Copyright (c) 2026" Foreground="{StaticResource TextMuted}" FontSize="11" Margin="0,16,0,0"/>
                        </StackPanel>
                    </Border>

                    <!-- Support Actions Card -->
                    <Border Style="{StaticResource CardStyle}">
                        <StackPanel>
                            <TextBlock Text="GET INVOLVED" Style="{StaticResource SubHeader}"/>
                            <WrapPanel>
                                <Button Name="btnSupportDiscord" Content="Join Discord" Style="{StaticResource UtilityBtn}" ToolTip="Community support server"/>
                                <Button Name="btnSupportIssue" Content="Report Issue" Style="{StaticResource ActionBtn}" ToolTip="Submit bug reports on GitHub"/>
                                <Button Name="btnToggleTheme" Content="Toggle Theme" Style="{StaticResource ActionBtn}" ToolTip="Switch between dark and light theme"/>
                                <Button Name="btnDonateIos12" Content="Sponsor Lil_Batti" Style="{StaticResource PositiveBtn}"/>
                                <Button Name="btnDonate" Content="Sponsor Chaython" Style="{StaticResource PositiveBtn}"/>
                            </WrapPanel>
                        </StackPanel>
                    </Border>
                </StackPanel>

            </Grid>
        </Border>
    </Grid>
</Window>
"@

# ==========================================
# 4. INIT & HELPERS
# ==========================================
$reader = (New-Object System.Xml.XmlNodeReader $xaml)
$window = [Windows.Markup.XamlReader]::Load($reader)

function Get-Ctrl { param($Name) return $window.FindName($Name) }

# --- THEME PALETTES ---
$script:CurrentTheme = "dark"
$script:ThemePalettes = @{
    dark = @{
        BgDark      = "#0D1117"
        BgPanel     = "#161B22"
        BgElevated  = "#21262D"
        BgHover     = "#30363D"
        BorderBrush = "#30363D"
        BorderAccent= "#58A6FF"
        Accent      = "#58A6FF"
        AccentHover = "#79C0FF"
        TextPrimary = "#E6EDF3"
        TextSecondary = "#8B949E"
        TextMuted   = "#6E7681"
        Success     = "#238636"
        SuccessHover= "#2EA043"
        Danger      = "#DA3633"
        DangerHover = "#F85149"
        Warning     = "#D29922"
        Info        = "#1F6FEB"
        LogText     = "#3FB950"
    }
    light = @{
        BgDark      = "#F5F7FA"
        BgPanel     = "#FFFFFF"
        BgElevated  = "#EEF2F7"
        BgHover     = "#E2E8F0"
        BorderBrush = "#CBD5E1"
        BorderAccent= "#2563EB"
        Accent      = "#2563EB"
        AccentHover = "#3B82F6"
        TextPrimary = "#0F172A"
        TextSecondary = "#334155"
        TextMuted   = "#64748B"
        Success     = "#15803D"
        SuccessHover= "#16A34A"
        Danger      = "#B91C1C"
        DangerHover = "#DC2626"
        Warning     = "#B45309"
        Info        = "#1D4ED8"
        LogText     = "#166534"
    }
}

function Set-WmtTheme {
    param([string]$Theme = "dark")
    if (-not $script:ThemePalettes.ContainsKey($Theme)) { $Theme = "dark" }
    $palette = $script:ThemePalettes[$Theme]

    foreach ($key in $palette.Keys) {
        if ($key -eq "LogText") { continue }
        $brush = $window.Resources[$key]
        if ($brush -is [System.Windows.Media.SolidColorBrush]) {
            $brush.Color = [System.Windows.Media.ColorConverter]::ConvertFromString($palette[$key])
        }
    }

    $window.Background = $window.Resources["BgDark"]
    $window.Foreground = $window.Resources["TextPrimary"]

    $logBoxCtrl = Get-Ctrl "LogBox"
    if ($logBoxCtrl) { $logBoxCtrl.Foreground = $palette.LogText }

    $quickFindCtrl = Get-Ctrl "txtGlobalSearch"
    if ($quickFindCtrl -and $quickFindCtrl.Text -eq $script:QuickFindPlaceholder) {
        $quickFindCtrl.Foreground = $window.Resources["TextMuted"]
    }

    $toggleBtn = Get-Ctrl "btnToggleTheme"
    if ($toggleBtn) {
        $toggleBtn.Content = if ($Theme -eq "dark") { "Light Mode" } else { "Dark Mode" }
    }

    $script:CurrentTheme = $Theme
}

# --- ICON INJECTION SYSTEM (SILENT + TOOLTIPS) ---
function Set-ButtonIcon {
    param($BtnName, $PathData, $Text, $Tooltip="", $Scale=16, $Color="White")
    $btn = Get-Ctrl $BtnName
    if (-not $btn) { return }
    
    # Visuals
    $sp = New-Object System.Windows.Controls.StackPanel; $sp.Orientation="Horizontal"
    $path = New-Object System.Windows.Shapes.Path
    $path.Data = [System.Windows.Media.Geometry]::Parse($PathData)
    $path.Fill = [System.Windows.Media.BrushConverter]::new().ConvertFromString($Color)
    $path.Stretch = "Uniform"; $path.Height=$Scale; $path.Width=$Scale; $path.Margin="0,0,10,0"
    $txt = New-Object System.Windows.Controls.TextBlock; $txt.Text=$Text; $txt.VerticalAlignment="Center"
    [void]$sp.Children.Add($path); [void]$sp.Children.Add($txt)
    $btn.Content = $sp
    
    # Tooltip
    if ($Tooltip) { $btn.ToolTip = $Tooltip }
}

# --- ICONS & TOOLTIPS ---
Set-ButtonIcon "btnTabUpdates" "M5,20H19V18H5M19,9H15V3H9V9H5L12,16L19,9Z" "Updates (Winget)" "Manage Windows software updates via Winget" 18 "#00FF00"
# Set-ButtonIcon "btnTabHealth" - CUSTOM LOGIC BELOW
Set-ButtonIcon "btnTabNetwork" "M5,3A2,2 0 0,0 3,5V15A2,2 0 0,0 5,17H8V15H5V5H19V15H16V17H19A2,2 0 0,0 21,15V5A2,2 0 0,0 19,3H5M11,15H13V17H11V15M11,11H13V13H11V11M11,7H13V9H11V7Z" "Network & DNS" "DNS, IP Config, Network Repair tools" 18
Set-ButtonIcon "btnTabFirewall" "M12,1L3,5V11C3,16.55 6.84,21.74 12,23C17.16,21.74 21,16.55 21,11V5L12,1M12,11.95L7,12.2V11.2L12,10.95L17,11.2V12.2L12,11.95Z" "Firewall Manager" "View and manage Windows Firewall rules" 18 "#FF5555"
Set-ButtonIcon "btnTabDrivers" "M7,17L10.5,12.5L5,9.6V17H7M12,21L14.6,16.3L9.5,13.6L12,21M17,17V9.6L11.5,12.5L15,17H17M20.2,4.8L12,1L3.8,4.8C2.7,5.4 2,6.5 2,7.7V17C2,19.8 4.2,22 7,22H17C19.8,22 22,19.8 22,17V7.7C22,6.5 21.3,5.4 20.2,4.8Z" "Drivers" "Backup, Restore, and Clean drivers" 18
Set-ButtonIcon "btnTabCleanup" "M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z" "Cleanup" "Disk cleanup, Temp files, Shortcuts, Registry" 18
Set-ButtonIcon "btnTabUtils" "M22.7,19L13.6,9.9C14.5,7.6 14,4.9 12.1,3C10.1,1 7.1,0.6 4.7,1.7L9,6L6,9L1.6,4.7C0.4,7.1 0.9,10.1 2.9,12.1C4.8,14 7.5,14.5 9.8,13.6L18.9,22.7C19.3,23.1 19.9,23.1 20.3,22.7L22.7,20.3C23.1,19.9 23.1,19.3 22.7,19Z" "Utilities" "System Info, SSD Trim, Activation, Task Scheduler" 18
Set-ButtonIcon "btnTabSupport" "M10,19H13V22H10V19M12,2C17.35,2.22 19.68,7.62 16.5,11.67C15.67,12.67 14.33,13.33 13.67,14.17C13,15 13,16 13,17H10C10,15.33 10,13.92 10.67,12.92C11.33,11.92 12.67,11.33 13.5,10.67C15.92,8.43 15.32,5.26 12,5A3,3 0 0,0 9,8H6A6,6 0 0,1 12,2Z" "Support & Credits" "Links to Discord and GitHub" 18
# Tweaks tab icon (lightning bolt / flash icon)
Set-ButtonIcon "btnTabTweaks" "M7,2V13H10V22L17,10H13L17,2H7M10,4H14L11,10H15L10.5,17V12H7V4H10Z" "Tweaks" "System optimization and performance tweaks" 18 "#FFD700"
$btnWingetIgnore = Get-Ctrl "btnWingetIgnore"
$btnWingetUnignore = Get-Ctrl "btnWingetUnignore"
# (Ban Icon for Ignore)
Set-ButtonIcon "btnWingetIgnore" "M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M12,4A8,8 0 0,0 4,12C4,13.85 4.63,15.55 5.68,16.91L16.91,5.68C15.55,4.63 13.85,4 12,4M12,20A8,8 0 0,0 20,12C20,10.15 19.37,8.45 18.32,7.09L7.09,18.32C8.45,19.37 10.15,20 12,20Z" "Ignore Selected" "Hide selected updates from future scans" 16 "#FFD700"
# (List/Restore Icon for Unignore)
Set-ButtonIcon "btnWingetUnignore" "M2,5H22V7H2V5M2,9H22V11H2V9M2,13H22V15H2V13M2,17H22V19H2V17" "Manage Ignored" "View and restore ignored updates"

# --- CUSTOM HEALTH ICON (Red Squircle with White Cross) ---
$btnHealth = Get-Ctrl "btnTabHealth"
if ($btnHealth) {
    $grid = New-Object System.Windows.Controls.Grid
    $grid.Width = 18; $grid.Height = 18; $grid.Margin = "0,0,10,0"
    
    # Red Squircle
    $rect = New-Object System.Windows.Shapes.Rectangle
    $rect.RadiusX=4; $rect.RadiusY=4
    $rect.Fill = [System.Windows.Media.BrushConverter]::new().ConvertFromString("#FF3333")
    [void]$grid.Children.Add($rect)
    
    # White Cross (Plus shape)
    $path = New-Object System.Windows.Shapes.Path
    $path.Data = [System.Windows.Media.Geometry]::Parse("M8,4H10V8H14V10H10V14H8V10H4V8H8V4Z")
    $path.Fill = [System.Windows.Media.Brushes]::White
    $path.Stretch = "Uniform"; $path.Margin = "3"
    [void]$grid.Children.Add($path)
    
    $sp = New-Object System.Windows.Controls.StackPanel; $sp.Orientation="Horizontal"
    [void]$sp.Children.Add($grid)
    $txt = New-Object System.Windows.Controls.TextBlock; $txt.Text="System Health"; $txt.VerticalAlignment="Center"
    [void]$sp.Children.Add($txt)
    
    $btnHealth.Content = $sp
    $btnHealth.ToolTip = "System integrity checks (SFC, DISM, CHKDSK)"
}

Set-ButtonIcon "btnNetRepair" "M20,12H19.5C19.5,14.5 17.5,16.5 15,16.5H9V18.5H15C18.6,18.5 21.5,15.6 21.5,12H21C21,15 19,17.5 16,18V16L13,19L16,22V20C19.9,19.4 23,16 23,12M3,12H3.5C3.5,9.5 5.5,7.5 8,7.5H14V5.5H8C4.4,5.5 1.5,8.4 1.5,12H2C2,9 4,6.5 7,6V8L10,5L7,2V4C3.1,4.6 0,8 0,12H3Z" "Full Net Repair" "Full network stack reset (Winsock, IP, Flush DNS)"
Set-ButtonIcon "btnRouteTable" "M19,15L13,21L11.58,19.58L15.17,16H4V4H6V14H15.17L11.58,10.42L13,9L19,15Z" "Save Route Table" "Exports the current IP routing table to the data folder"
Set-ButtonIcon "btnRouteView" "M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M12,17C14.76,17 17,14.76 17,12C17,9.24 14.76,7 12,7C9.24,7 7,9.24 7,12C7,14.76 9.24,17 12,17M12,9A3,3 0 0,1 15,12A3,3 0 0,1 12,15A3,3 0 0,1 9,12A3,3 0 0,1 12,9Z" "View Route Table" "Displays the routing table in the log"
Set-ButtonIcon "btnCleanReg" "M5,3H19A2,2 0 0,1 21,5V19A2,2 0 0,1 19,21H5A2,2 0 0,1 3,19V5A2,2 0 0,1 5,3M7,7V9H9V7H7M11,7V9H13V7H11M15,7V9H17V7H15M7,11V13H9V11H7M11,11V13H13V11H11M15,11V13H17V11H15M7,15V17H9V15H7M11,15V17H13V15H11M15,15V17H17V15H15Z" "Clean Reg Keys" "Backs up & deletes obsolete Uninstall registry keys"
Set-ButtonIcon "btnCleanXbox" "M6.4,4.8L12,10.4L17.6,4.8L19.2,6.4L13.6,12L19.2,17.6L17.6,19.2L12,13.6L6.4,19.2L4.8,17.6L10.4,12L4.8,6.4L6.4,4.8Z" "Clean Xbox Data" "Removes Xbox Live credentials to fix login loops" 18 "#107C10"
Set-ButtonIcon "btnUpdateRepair" "M21,10.12H14.22L16.96,7.3C14.55,4.61 10.54,4.42 7.85,6.87C5.16,9.32 5.35,13.33 7.8,16.03C10.25,18.72 14.26,18.91 16.95,16.46C17.65,15.82 18.2,15.05 18.56,14.21L20.62,15.05C19.79,16.89 18.3,18.42 16.39,19.34C13.4,20.78 9.77,20.21 7.37,17.96C4.96,15.71 4.54,12.06 6.37,9.32C8.2,6.59 11.83,5.65 14.65,7.09L17.38,4.35H10.63V2.35H21V10.12Z" "Reset Update Svc" "Stops services, clears cache, and resets Windows Update components"
Set-ButtonIcon "btnUpdateServices" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12H20A8,8 0 0,1 12,20A8,8 0 0,1 4,12A8,8 0 0,1 12,4C14.23,4 16.24,4.82 17.76,6.24L14,10H22V2L19.36,4.64C17.5,2.89 14.89,2 12,2Z" "Restart Update Svcs" "Restarts update-related services"
Set-ButtonIcon "btnDotNetEnable" "M14.6,16.6L19.2,12L14.6,7.4L16,6L22,12L16,18L14.6,16.6M9.4,16.6L4.8,12L9.4,7.4L8,6L2,12L8,18L9.4,16.6Z" "Set .NET RollFwd" "Sets DOTNET_ROLL_FORWARD=LatestMajor (Force apps to use newest .NET)"
Set-ButtonIcon "btnDotNetDisable" "M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z" "Reset .NET RollFwd" "Removes the DOTNET_ROLL_FORWARD environment variable"
Set-ButtonIcon "btnTaskManager" "M14,10H2V12H14V10M14,6H2V8H14V6M2,16H10V14H2V16M21.5,11.5L23,13L16,20L11.5,15.5L13,14L16,17L21.5,11.5Z" "Task Scheduler" "View, Enable, Disable, or Delete Windows Scheduled Tasks"
Set-ButtonIcon "btnInstallGpedit" "M6,2C4.89,2 4,2.89 4,4V20A2,2 0 0,0 6,22H18A2,2 0 0,0 20,20V8L14,2H6M6,4H13V9H18V20H6V4M8,12V14H16V12H8M8,16V18H13V16H8Z" "Install Gpedit" "Installs the Group Policy Editor on Windows Home editions"
Set-ButtonIcon "btnSFC" "M15.5,14L20.5,19L19,20.5L14,15.5V14.71L13.73,14.43C12.59,15.41 11.11,16 9.5,16A6.5,6.5 0 0,1 3,9.5A6.5,6.5 0 0,1 9.5,3A6.5,6.5 0 0,1 16,9.5C16,11.11 15.41,12.59 14.43,13.73L14.71,14H15.5M9.5,14C12,14 14,12 14,9.5C14,7 12,5 9.5,5C7,5 5,7 5,9.5C5,12 7,14 9.5,14Z" "SFC Scan" "Scans system files for corruption and repairs them"
Set-ButtonIcon "btnDISMCheck" "M22,10V9C22,5.1 18.9,2 15,2C11.1,2 8,5.1 8,9V10H22M19.5,12.5C19.5,11.1 20.6,10 22,10H8V15H19.5V12.5Z" "DISM Check" "Checks the health of the Windows Image (dism /checkhealth)"
Set-ButtonIcon "btnDISMRestore" "M19.5,12.5C19.5,11.1 20.6,10 22,10V9C22,5.1 18.9,2 15,2C11.1,2 8,5.1 8,9V10C9.4,10 10.5,11.1 10.5,12.5C10.5,13.9 9.4,15 8,15V19H12V22H8C6.3,22 5,20.7 5,19V15C3.6,15 2.5,13.9 2.5,12.5C2.5,11.1 3.6,10 5,10V9C5,3.5 9.5,-1 15,-1C20.5,-1 25,3.5 25,9V10C26.4,10 27.5,11.1 27.5,12.5C27.5,13.9 26.4,15 25,15V19C25,20.7 23.7,22 22,22H17V19H22V15C20.6,15 19.5,13.9 19.5,12.5Z" "DISM Restore" "Attempts to repair the Windows Image (dism /restorehealth)"
Set-ButtonIcon "btnCHKDSK" "M6,2H18C19.1,2 20,2.9 20,4V20C20,21.1 19.1,22 18,22H6C4.9,22 4,21.1 4,20V4C4,2.9 4.9,2 6,2M6,4V20H18V4H6M11,17C11,17.55 11.45,18 12,18C12.55,18 13,17.55 13,17C13,16.45 12.55,16 12,16C11.45,16 11,16.45 11,17M7,17C7,17.55 7.45,18 8,18C8.55,18 9,17.55 9,17C9,16.45 8.55,16 8,16C7.45,16 7,16.45 7,17M15,17C15,17.55 15.45,18 16,18C16.55,18 17,17.55 17,17C17,16.45 16.55,16 16,16C15.45,16 15,16.45 15,17Z" "Check Disk" "Scans all drives for filesystem errors (requires reboot)"
Set-ButtonIcon "btnFlushDNS" "M2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2A10,10 0 0,0 2,12M4,12A8,8 0 0,1 12,4A8,8 0 0,1 20,12A8,8 0 0,1 12,20A8,8 0 0,1 4,12M10,17L15,12L10,7V17Z" "Flush DNS" "Clears the client DNS resolver cache"
Set-ButtonIcon "btnNetInfo" "M13,9H11V7H13M13,17H11V11H13M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2Z" "Show IP Config" "Displays full IP configuration for all adapters"
Set-ButtonIcon "btnResetWifi" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M12,4A8,8 0 0,1 20,12A8,8 0 0,1 12,20A8,8 0 0,1 4,12A8,8 0 0,1 12,4M11,16.5L18,9.5L16.59,8.09L11,13.67L7.91,10.59L6.5,12L11,16.5Z" "Restart Wi-Fi" "Disables and Re-Enables Wi-Fi adapters"
Set-ButtonIcon "btnCleanDisk" "M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z" "Disk Cleanup" "Opens the built-in Windows Disk Cleanup utility"
Set-ButtonIcon "btnCleanTemp" "M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z" "Delete Temp Files" "Deletes temporary files from User and System Temp folders"
Set-ButtonIcon "btnCleanShortcuts" "M19,3H5C3.89,3 3,3.89 3,5V19A2,2 0 0,0 5,21H19A2,2 0 0,0 21,19V5C21,3.89 20.1,3 19,3M19,19H5V5H19V19M10,17L5,12L6.41,10.59L10,14.17L17.59,6.58L19,8L10,17Z" "Fix Shortcuts" "Scans for and fixes broken .lnk shortcuts"
(Get-Ctrl "btnWingetFind").Width = 100
Set-ButtonIcon "btnWingetFind" "M9.5,3A6.5,6.5 0 0,1 16,9.5C16,11.11 15.41,12.59 14.44,13.73L14.71,14H15.5L20.5,19L19,20.5L14,15.5V14.71L13.73,14.44C12.59,15.41 11.11,16 9.5,16A6.5,6.5 0 0,1 3,9.5A6.5,6.5 0 0,1 9.5,3M9.5,5C7,5 5,7 5,9.5C5,12 7,14 9.5,14C12,14 14,12 14,9.5C14,7 12,5 9.5,5Z" "Search" "Search Winget"
Set-ButtonIcon "btnWingetScan" "M12,18A6,6 0 0,1 6,12C6,11 6.25,10.03 6.7,9.2L5.24,7.74C4.46,8.97 4,10.43 4,12A8,8 0 0,0 12,20V23L16,19L12,15V18M12,4V1L8,5L12,9V6A6,6 0 0,1 18,12C18,13 17.75,13.97 17.3,14.8L18.76,16.26C19.54,15.03 20,13.57 20,12A8,8 0 0,0 12,4Z" "Refresh Updates" "Checks the Winget repository for available application updates"
Set-ButtonIcon "btnWingetUpdateSel" "M5,20H19V18H5M19,9H15V3H9V9H5L12,16L19,9Z" "Update Selected" "Updates the selected applications"
Set-ButtonIcon "btnWingetInstall" "M19,13H13V19H11V13H5V11H11V5H13V11H19V13Z" "Install Selected" "Installs the selected applications"
Set-ButtonIcon "btnWingetUninstall" "M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z" "Uninstall Selected" "Uninstalls the selected applications"
Set-ButtonIcon "btnSupportDiscord" "M19.27 5.33C17.94 4.71 16.5 4.26 15 4a.09.09 0 0 0-.07.03c-.18.33-.39.76-.53 1.09a16.09 16.09 0 0 0-4.8 0c-.14-.34-.35-.76-.54-1.09c-.01-.02-.04-.03-.07-.03c-1.5.26-2.93.71-4.27 1.33c-.01 0-.02.01-.03.02c-2.72 4.07-3.47 8.03-3.1 11.95c0 .02.01.04.03.05c1.8 1.32 3.53 2.12 5.2 2.65c.03.01.06 0 .07-.02c.4-.55.76-1.13 1.07-1.74c.02-.04 0-.08-.04-.09c-.57-.22-1.11-.48-1.64-.78c-.04-.02-.04-.08.01-.11c.11-.08.22-.17.33-.25c.02-.02.05-.02.07-.01c3.44 1.57 7.15 1.57 10.55 0c.02-.01.05-.01.07.01c.11.09.22.17.33.26c.04.03.04.09-.01.11c-.52.31-1.07.56-1.64.78c-.04.01-.05.06-.04.09c.32.61.68 1.19 1.07 1.74c.03.01.06.02.09.01c1.67-.53 3.4-1.33 5.2-2.65c.02-.01.03-.03.03-.05c.44-4.53-.73-8.46-3.1-11.95c-.01-.01-.02-.02-.04-.02z" "Join Discord" "Opens the community support Discord server"
Set-ButtonIcon "btnSupportIssue" "M12 .297c-6.63 0-12 5.373-12 12 0 5.303 3.438 9.8 8.205 11.385.6.113.82-.258.82-.577 0-.285-.01-1.04-.015-2.04-3.338.724-4.042-1.61-4.042-1.61C4.422 18.07 3.633 17.7 3.633 17.7c-1.087-.744.084-.729.084-.729 1.205.084 1.838 1.236 1.838 1.236 1.07 1.835 2.809 1.305 3.495.998.108-.776.417-1.305.76-1.605-2.665-.3-5.466-1.332-5.466-5.93 0-1.31.465-2.38 1.235-3.22-.135-.303-.54-1.523.105-3.176 0 0 1.005-.322 3.3 1.23.96-.267 1.98-.399 3-.405 1.02.006 2.04.138 3 .405 2.28-1.552 3.285-1.23 3.285-1.23.645 1.653.24 2.873.12 3.176.765.84 1.23 1.91 1.23 3.22 0 4.61-2.805 5.625-5.475 5.92.42.36.81 1.096.81 2.22 0 1.606-.015 2.896-.015 3.286 0 .315.21.69.825.57C20.565 22.092 24 17.592 24 12.297c0-6.627-5.373-12-12-12" "Report Issue" "Opens the GitHub Issues page to report bugs"
Set-ButtonIcon "btnNavDownloads" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M11,6H13V12H11V6M11,14H13V16H11V14Z" "Release Downloads" "Show latest release download counts"
Set-ButtonIcon "btnDonateIos12" "M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.9,13.5 12,13.5C8.36,13.5 6,12.28 6,10C6,7.24 8.7,5 12,5V3H14V5C15.68,5.37 16.86,6.31 17.38,7.5H15.32C14.93,6.85 13.95,6.2 12,6.2C10.37,6.2 9,7.11 9,8.2C9,9.3 10.1,9.7 12,9.7C15.64,9.7 18,10.92 18,13.2C18,15.96 15.3,18.2 12,18.2V20H10V18.2C8.32,17.83 7.14,16.89 6.62,15.7L8.68,15Z" "Sponsor Lil_Batti" "Support Lil_Batti via GitHub Sponsors" "#00FF00"
Set-ButtonIcon "btnDonate" "M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.9,13.5 12,13.5C8.36,13.5 6,12.28 6,10C6,7.24 8.7,5 12,5V3H14V5C15.68,5.37 16.86,6.31 17.38,7.5H15.32C14.93,6.85 13.95,6.2 12,6.2C10.37,6.2 9,7.11 9,8.2C9,9.3 10.1,9.7 12,9.7C15.64,9.7 18,10.92 18,13.2C18,15.96 15.3,18.2 12,18.2V20H10V18.2C8.32,17.83 7.14,16.89 6.62,15.7L8.68,15Z" "Sponsor Chaython" "Support Chaython via GitHub Sponsors" "#00FF00"
Set-ButtonIcon "btnDnsGoogle" "M21.35,11.1H12.18V13.83H18.69C18.36,17.64 15.19,19.27 12.19,19.27C8.36,19.27 5,16.25 5,12C5,7.9 8.2,4.73 12.2,4.73C15.29,4.73 17.1,6.7 17.1,6.7L19,4.72C19,4.72 16.56,2 12.1,2C6.42,2 2.03,6.8 2.03,12C2.03,17.05 6.16,22 12.25,22C17.6,22 21.5,18.33 21.5,12.91C21.5,11.76 21.35,11.1 21.35,11.1V11.1Z" "Google" "Sets DNS to 8.8.8.8 & 8.8.4.4"
Set-ButtonIcon "btnDnsCloudflare" "M19.35,10.04C18.67,6.59 15.64,4 12,4C9.11,4 6.6,5.64 5.35,8.04C2.34,8.36 0,10.91 0,14A6,6 0 0,0 6,20H19A5,5 0 0,0 24,15C24,12.36 21.95,10.22 19.35,10.04Z" "Cloudflare" "Sets DNS to 1.1.1.1 & 1.0.0.1"
Set-ButtonIcon "btnDnsQuad9" "M12,1L3,5V11C3,16.55 6.84,21.74 12,23C17.16,21.74 21,16.55 21,11V5L12,1M10,17L6,13L7.41,11.59L10,14.17L16.59,7.58L18,9L10,17Z" "Quad9" "Sets DNS to 9.9.9.9 (Malware Blocking)"
Set-ButtonIcon "btnDnsAuto" "M12,18A6,6 0 0,1 6,12C6,11 6.25,10.03 6.7,9.2L5.24,7.74C4.46,8.97 4,10.43 4,12A8,8 0 0,0 12,20V23L16,19L12,15V18M12,4V1L8,5L12,9V6A6,6 0 0,1 18,12C18,13 17.75,13.97 17.3,14.8L18.76,16.26C19.54,15.03 20,13.57 20,12A8,8 0 0,0 12,4Z" "Auto (DHCP)" "Resets DNS settings to DHCP (Automatic)"
Set-ButtonIcon "btnDnsCustom" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M17,7L12,12L7,7H17Z" "Custom DNS" "Set custom DNS addresses across active adapters"
Set-ButtonIcon "btnHostsUpdate" "M5,20H19V18H5M19,9H15V3H9V9H5L12,16L19,9Z" "Download AdBlock" "Updates Hosts file with AdBlocking list"
Set-ButtonIcon "btnHostsEdit" "M14.06,9L15,9.94L5.92,19H5V18.08L14.06,9M17.66,3C17.41,3 17.15,3.1 16.96,3.29L15.13,5.12L18.88,8.87L20.71,7.04C21.1,6.65 21.1,6 20.71,5.63L18.37,3.29C18.17,3.09 17.92,3 17.66,3M14.06,6.19L3,17.25V21H6.75L17.81,9.94L14.06,6.19Z" "Edit Hosts" "Opens the Hosts File Editor"
Set-ButtonIcon "btnHostsBackup" "M19,9H15V3H9V9H5L12,16L19,9Z" "Backup Hosts" "Backs up the current hosts file to the data folder"
Set-ButtonIcon "btnHostsRestore" "M13,3A9,9 0 0,0 4,12H1L4.89,15.89L4.96,16.03L9,12H6A7,7 0 0,1 13,5A7,7 0 0,1 20,12A7,7 0 0,1 13,19C11.07,19 9.32,18.21 8.06,16.94L6.64,18.36C8.27,20 10.5,21 13,21A9,9 0 0,0 22,12A9,9 0 0,0 13,3Z" "Restore Hosts" "Restores a previous hosts file backup"
Set-ButtonIcon "btnDohAuto" "M12,1L3,5V11C3,16.55 6.84,21.74 12,23C17.16,21.74 21,16.55 21,11V5L12,1M10,17L6,13L7.41,11.59L10,14.17L16.59,7.58L18,9L10,17Z" "Enable DoH (All)" "Enables DNS over HTTPS for all supported providers" "#00FFFF"
Set-ButtonIcon "btnDohDisable" "M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z" "Disable DoH" "Disables DNS over HTTPS" "#FF5555"
Set-ButtonIcon "btnFwRefresh" "M17.65,6.35C16.2,4.9 14.21,4 12,4A8,8 0 0,0 4,12A8,8 0 0,0 12,20C15.73,20 18.84,17.45 19.73,14H17.65C16.83,16.33 14.61,18 12,18A6,6 0 0,1 6,12A6,6 0 0,1 12,6C13.66,6 15.14,6.69 16.22,7.78L13,11H20V4L17.65,6.35Z" "Reload" "Refreshes the firewall rule list"
Set-ButtonIcon "btnFwAdd" "M19,13H13V19H11V13H5V11H11V5H13V11H19V13Z" "Add Rule" "Create a new firewall rule"
Set-ButtonIcon "btnFwEdit" "M14.06,9L15,9.94L5.92,19H5V18.08L14.06,9M17.66,3C17.41,3 17.15,3.1 16.96,3.29L15.13,5.12L18.88,8.87L20.71,7.04C21.1,6.65 21.1,6 20.71,5.63L18.37,3.29C18.17,3.09 17.92,3 17.66,3M14.06,6.19L3,17.25V21H6.75L17.81,9.94L14.06,6.19Z" "Modify" "Edit the selected firewall rule"
Set-ButtonIcon "btnFwEnable" "M10,17L6,13L7.41,11.59L10,14.17L16.59,7.58L18,9L10,17Z" "Enable" "Enable selected rule"
Set-ButtonIcon "btnFwDisable" "M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z" "Disable" "Disable selected rule"
Set-ButtonIcon "btnFwDelete" "M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z" "Delete" "Delete selected rule"
Set-ButtonIcon "btnFwExport" "M15,14H14V10H10V14H9L12,17L15,14M12,3L4.5,8V14C4.5,17.93 7.36,21.43 12,23C16.64,21.43 19.5,17.93 19.5,14V8L12,3Z" "Export" "Export firewall policy to the data folder"
Set-ButtonIcon "btnFwImport" "M12,3L4.5,8V14C4.5,17.93 7.36,21.43 12,23C16.64,21.43 19.5,17.93 19.5,14V8L12,3M12,6.15L17.5,10.2V14C17.5,16.96 15.56,19.5 12,20.82C8.44,19.5 6.5,16.96 6.5,14V10.2L12,6.15M12,9L8,13H11V17H13V13H16L12,9Z" "Import" "Import firewall policy (.wfw)"
Set-ButtonIcon "btnFwDefaults" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12C22,6.47 17.5,2 12,2M7,9H9V13H11V9H13V13H15V9H17V15H7V9Z" "Restore Defaults" "Reset firewall to default rules"
Set-ButtonIcon "btnFwPurge" "M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z" "Delete All" "Delete all firewall rules"
Set-ButtonIcon "btnDrvReport" "M13,9H18.5L13,3.5V9M6,2H14L20,8V20A2,2 0 0,1 18,22H6C4.89,22 4,21.1 4,20V4C4,2.89 4.89,2 6,2M15,18V16H6V18H15M18,14V12H6V14H18Z" "Generate Driver Report" "Saves a list of all installed drivers to the data folder"
Set-ButtonIcon "btnDrvGhost" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M12,4A8,8 0 0,1 20,12A8,8 0 0,1 12,20A8,8 0 0,1 4,12A8,8 0 0,1 12,4M11,16.5L18,9.5L16.59,8.09L11,13.67L7.91,10.59L6.5,12L11,16.5Z" "Remove Ghost Devices" "Removes disconnected (ghost) PnP devices"
Set-ButtonIcon "btnDrvBackup" "M13,9H18.5L13,3.5V9M6,2H14L20,8V20A2,2 0 0,1 18,22H6C4.89,22 4,21.1 4,20V4C4,2.89 4.89,2 6,2M15,18V16H6V18H15M18,14V12H6V14H18Z" "Export Drivers" "Exports all drivers to the data folder"
Set-ButtonIcon "btnDrvClean" "M19,4H15.5L14.5,3H9.5L8.5,4H5V6H19M6,19A2,2 0 0,0 8,21H16A2,2 0 0,0 18,19V7H6V19Z" "Clean Old Drivers" "Removes obsolete drivers from the Windows Driver Store"
Set-ButtonIcon "btnDrvRestore" "M12,2L3,7V17L12,22L21,17V7L12,2M12,4.3L18.5,8L12,11.7L5.5,8L12,4.3M5,9.85L12,14L19,9.85V16.15L12,20.3L5,16.15V9.85M7,11H9V14H7V11M15,11H17V14H15V11Z" "Restore Drivers" "Imports drivers from a DriverBackup folder"
Set-ButtonIcon "btnDrvDisableWU" "M19,4H5V6H19M5,20H19V18H5M9,9H15V11H9V9M9,13H15V15H9V13Z" "Disable Driver Updates" "Turn off automatic driver updates"
Set-ButtonIcon "btnDrvEnableWU" "M19,4H5V6H19M5,20H19V18H5M9,9H15V11H9V9M9,13H13V15H9V13Z" "Enable Driver Updates" "Turn on automatic driver updates"
Set-ButtonIcon "btnDrvDisableMeta" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M15,17H9V15H15V17M16.59,11.17L15.17,12.59L12,9.41L8.83,12.59L7.41,11.17L12,6.58L16.59,11.17Z" "Disable Device Metadata" "Block device metadata downloads"
Set-ButtonIcon "btnDrvEnableMeta" "M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M15,17H9V15H15V17M11,14V9H9L12,6L15,9H13V14H11Z" "Enable Device Metadata" "Allow device metadata downloads"
Set-ButtonIcon "btnUtilSysInfo" "M13,9H18.5L13,3.5V9M6,2H14L20,8V20A2,2 0 0,1 18,22H6C4.89,22 4,21.1 4,20V4C4,2.89 4.89,2 6,2M15,18V16H6V18H15M18,14V12H6V14H18Z" "System Info Report" "Generates a full system information report"
Set-ButtonIcon "btnUtilTrim" "M6,2H18A2,2 0 0,1 20,4V20A2,2 0 0,1 18,22H6A2,2 0 0,1 4,20V4A2,2 0 0,1 6,2M12,4A6,6 0 0,0 6,10C6,13.31 8.69,16 12,16A6,6 0 0,0 18,10C18,6.69 15.31,4 12,4M12,14A4,4 0 0,1 8,10A4,4 0 0,1 12,6A4,4 0 0,1 16,10A4,4 0 0,1 12,14Z" "Trim SSD" "Optimizes SSD performance via Trim command"
Set-ButtonIcon "btnUtilMas" "M5,20H19V18H5M19,9H15V3H9V9H5L12,16L19,9Z" "MAS Activation" "Downloads and runs Microsoft Activation Scripts"
Set-ButtonIcon "btnCtxBuilder" "M19,3H5C3.89,3 3,3.89 3,5V19A2,2 0 0,0 5,21H19A2,2 0 0,0 21,19V5C21,3.89 20.1,3 19,3M19,19H5V5H19V19M10,17L5,12L6.41,10.59L10,14.17L17.59,6.58L19,8L10,17Z" "Context Menu Builder" "Create a custom right-click action for Windows 11"
# ==========================================
# 5. LOGIC & EVENTS
# ==========================================
$TabButtons = @("btnTabUpdates","btnTabTweaks","btnTabHealth","btnTabNetwork","btnTabFirewall","btnTabDrivers","btnTabCleanup","btnTabUtils","btnTabSupport")
$Panels     = @("pnlUpdates","pnlCatalog","pnlTweaks","pnlHealth","pnlNetwork","pnlFirewall","pnlDrivers","pnlCleanup","pnlUtils","pnlSupport")

# --- INITIALIZE ALL CONTROLS ---
$btnManageProviders = Get-Ctrl "btnManageProviders"
$btnManageProviders.Add_Click({ Show-ProviderManager })
$btnWingetScan = Get-Ctrl "btnWingetScan"
$btnWingetUpdateSel = Get-Ctrl "btnWingetUpdateSel"
$btnWingetUpdateAll = Get-Ctrl "btnWingetUpdateAll"
$btnWingetInstall = Get-Ctrl "btnWingetInstall"
$btnWingetUninstall = Get-Ctrl "btnWingetUninstall"
$btnWingetFind = Get-Ctrl "btnWingetFind"
$lstWinget = Get-Ctrl "lstWinget"
$txtWingetSearch = Get-Ctrl "txtWingetSearch"
$lblWingetStatus = Get-Ctrl "lblWingetStatus"
$lblWingetTitle = Get-Ctrl "lblWingetTitle"
$pbWingetProgress = Get-Ctrl "pbWingetProgress"
$lblWingetProgress = Get-Ctrl "lblWingetProgress"
$cmbPackageManager = Get-Ctrl "cmbPackageManager"
if ($cmbPackageManager) {
    $cmbPackageManager.Add_SelectionChanged({ 
        $btnWingetScan.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
    })
}

$btnSFC = Get-Ctrl "btnSFC"
$btnDISMCheck = Get-Ctrl "btnDISMCheck"
$btnDISMRestore = Get-Ctrl "btnDISMRestore"
$btnCHKDSK = Get-Ctrl "btnCHKDSK"

$btnPerfServicesManual = Get-Ctrl "btnPerfServicesManual"
$btnPerfServicesRevert = Get-Ctrl "btnPerfServicesRevert"
$btnPerfDisableHibernate = Get-Ctrl "btnPerfDisableHibernate"
$btnPerfEnableHibernate = Get-Ctrl "btnPerfEnableHibernate"
$btnPerfDisableSuperfetch = Get-Ctrl "btnPerfDisableSuperfetch"
$btnPerfEnableSuperfetch = Get-Ctrl "btnPerfEnableSuperfetch"
$btnPerfDisableMemCompress = Get-Ctrl "btnPerfDisableMemCompress"
$btnPerfEnableMemCompress = Get-Ctrl "btnPerfEnableMemCompress"
$btnPerfUltimatePower = Get-Ctrl "btnPerfUltimatePower"

$btnAppxLoad = Get-Ctrl "btnAppxLoad"
$btnAppxRemoveSel = Get-Ctrl "btnAppxRemoveSel"
$btnAppxRemoveAll = Get-Ctrl "btnAppxRemoveAll"
$lstAppxPackages = Get-Ctrl "lstAppxPackages"

$btnFeatHyperV = Get-Ctrl "btnFeatHyperV"
$btnFeatWSL = Get-Ctrl "btnFeatWSL"
$btnFeatSandbox = Get-Ctrl "btnFeatSandbox"
$btnFeatDotNet35 = Get-Ctrl "btnFeatDotNet35"
$btnFeatNFS = Get-Ctrl "btnFeatNFS"
$btnFeatTelnet = Get-Ctrl "btnFeatTelnet"
$btnFeatIIS = Get-Ctrl "btnFeatIIS"
$btnFeatLegacy = Get-Ctrl "btnFeatLegacy"

$btnSvcOptimize = Get-Ctrl "btnSvcOptimize"
$btnSvcRestore = Get-Ctrl "btnSvcRestore"
$btnSvcView = Get-Ctrl "btnSvcView"

$btnTasksDisableTelemetry = Get-Ctrl "btnTasksDisableTelemetry"
$btnTasksRestore = Get-Ctrl "btnTasksRestore"
$btnTasksView = Get-Ctrl "btnTasksView"

$btnWUDefault = Get-Ctrl "btnWUDefault"
$btnWUSecurity = Get-Ctrl "btnWUSecurity"
$btnWUDisable = Get-Ctrl "btnWUDisable"

$btnNetInfo = Get-Ctrl "btnNetInfo"
$btnFlushDNS = Get-Ctrl "btnFlushDNS"
$btnResetWifi = Get-Ctrl "btnResetWifi"
$btnNetRepair = Get-Ctrl "btnNetRepair"
$btnRouteTable = Get-Ctrl "btnRouteTable"
$btnRouteView = Get-Ctrl "btnRouteView"
$btnDnsGoogle = Get-Ctrl "btnDnsGoogle"
$btnDnsCloudflare = Get-Ctrl "btnDnsCloudflare"
$btnDnsQuad9 = Get-Ctrl "btnDnsQuad9"
$btnDnsAuto = Get-Ctrl "btnDnsAuto"
$btnDnsCustom = Get-Ctrl "btnDnsCustom"
$btnDohAuto = Get-Ctrl "btnDohAuto"
$btnDohDisable = Get-Ctrl "btnDohDisable"
$btnHostsUpdate = Get-Ctrl "btnHostsUpdate"
$btnHostsEdit = Get-Ctrl "btnHostsEdit"
$btnHostsBackup = Get-Ctrl "btnHostsBackup"
$btnHostsRestore = Get-Ctrl "btnHostsRestore"

$btnFwRefresh = Get-Ctrl "btnFwRefresh"
$btnFwAdd = Get-Ctrl "btnFwAdd"
$btnFwEdit = Get-Ctrl "btnFwEdit"
$btnFwEnable = Get-Ctrl "btnFwEnable"
$btnFwDisable = Get-Ctrl "btnFwDisable"
$btnFwDelete = Get-Ctrl "btnFwDelete"
$btnFwExport = Get-Ctrl "btnFwExport"
$btnFwImport = Get-Ctrl "btnFwImport"
$btnFwDefaults = Get-Ctrl "btnFwDefaults"
$btnFwPurge = Get-Ctrl "btnFwPurge"
$lstFw = Get-Ctrl "lstFirewall"
$txtFwSearch = Get-Ctrl "txtFwSearch"
$lblFwStatus = Get-Ctrl "lblFwStatus"

$btnDrvReport = Get-Ctrl "btnDrvReport"
$btnDrvBackup = Get-Ctrl "btnDrvBackup"
$btnDrvGhost = Get-Ctrl "btnDrvGhost"
$btnDrvClean = Get-Ctrl "btnDrvClean"
$btnDrvRestore = Get-Ctrl "btnDrvRestore"
$btnDrvDisableWU = Get-Ctrl "btnDrvDisableWU"
$btnDrvEnableWU = Get-Ctrl "btnDrvEnableWU"
$btnDrvDisableMeta = Get-Ctrl "btnDrvDisableMeta"
$btnDrvEnableMeta = Get-Ctrl "btnDrvEnableMeta"

$btnCleanDisk = Get-Ctrl "btnCleanDisk"
$btnCleanTemp = Get-Ctrl "btnCleanTemp"
$btnCleanShortcuts = Get-Ctrl "btnCleanShortcuts"
$btnCleanReg = Get-Ctrl "btnCleanReg"
$btnCleanXbox = Get-Ctrl "btnCleanXbox"

$btnUtilSysInfo = Get-Ctrl "btnUtilSysInfo"
$btnUtilTrim = Get-Ctrl "btnUtilTrim"
$btnUtilMas = Get-Ctrl "btnUtilMas"
$btnUpdateRepair = Get-Ctrl "btnUpdateRepair"
$btnUpdateServices = Get-Ctrl "btnUpdateServices"
$btnDotNetEnable = Get-Ctrl "btnDotNetEnable"
$btnDotNetDisable = Get-Ctrl "btnDotNetDisable"
$btnTaskManager = Get-Ctrl "btnTaskManager"
$btnInstallGpedit = Get-Ctrl "btnInstallGpedit"
$btnCtxBuilder = Get-Ctrl "btnCtxBuilder"

$pnlUpdates = Get-Ctrl "pnlUpdates"
$pnlCatalog = Get-Ctrl "pnlCatalog"
$lstCatalog = Get-Ctrl "lstCatalog"
$txtCatalogSearch = Get-Ctrl "txtCatalogSearch"
$btnShowCatalog = Get-Ctrl "btnShowCatalog"
$btnBackToUpdates = Get-Ctrl "btnBackToUpdates"
$btnCatalogSearch = Get-Ctrl "btnCatalogSearch"
$btnCatalogInstall = Get-Ctrl "btnCatalogInstall"
$btnCatalogSelectAll = Get-Ctrl "btnCatalogSelectAll"
$btnCatalogClear = Get-Ctrl "btnCatalogClear"
$btnCatAll = Get-Ctrl "btnCatAll"
$btnCatBrowsers = Get-Ctrl "btnCatBrowsers"
$btnCatDev = Get-Ctrl "btnCatDev"
$btnCatUtils = Get-Ctrl "btnCatUtils"
$btnCatMedia = Get-Ctrl "btnCatMedia"
$btnCatGames = Get-Ctrl "btnCatGames"
$btnCatSecurity = Get-Ctrl "btnCatSecurity"

$btnSupportDiscord = Get-Ctrl "btnSupportDiscord"
$btnSupportIssue = Get-Ctrl "btnSupportIssue"
$btnToggleTheme = Get-Ctrl "btnToggleTheme"
$btnNavDownloads = Get-Ctrl "btnNavDownloads"
$btnDonateIos12 = Get-Ctrl "btnDonateIos12"
$btnDonate = Get-Ctrl "btnDonate"
$btnCreditLilBattiCLI = Get-Ctrl "btnCreditLilBattiCLI"
$btnCreditChaythonCLI = Get-Ctrl "btnCreditChaythonCLI"
$btnCreditChaythonGUI = Get-Ctrl "btnCreditChaythonGUI"
$btnCreditChaythonFeatures = Get-Ctrl "btnCreditChaythonFeatures"
$btnCreditIos12checker = Get-Ctrl "btnCreditIos12checker"

$bdQuickFind = Get-Ctrl "bdQuickFind"
$txtGlobalSearch = Get-Ctrl "txtGlobalSearch"
$lstSearchResults = Get-Ctrl "lstSearchResults"
$pnlNavButtons = Get-Ctrl "pnlNavButtons"
$svLog = Get-Ctrl "svLog"
$LogBox = Get-Ctrl "LogBox"
if ($LogBox) {
    $LogBox.Add_TextChanged({
        param($s,$e)
        if ($svLog) { $svLog.ScrollToEnd() } else { $s.ScrollToEnd() }
    })
}

# Quick Find placeholder + focus behavior
$script:QuickFindPlaceholder = "Search tools..."
if ($txtGlobalSearch) {
    $txtGlobalSearch.Text = $script:QuickFindPlaceholder
    $txtGlobalSearch.Foreground = $window.Resources["TextMuted"]
    $txtGlobalSearch.Add_GotFocus({
        if ($txtGlobalSearch.Text -eq $script:QuickFindPlaceholder) {
            $txtGlobalSearch.Text = ""
            $txtGlobalSearch.Foreground = $window.Resources["TextPrimary"]
        }
    })
    $txtGlobalSearch.Add_LostFocus({
        if ([string]::IsNullOrWhiteSpace($txtGlobalSearch.Text)) {
            $txtGlobalSearch.Text = $script:QuickFindPlaceholder
            $txtGlobalSearch.Foreground = $window.Resources["TextMuted"]
        }
    })
}
if ($bdQuickFind -and $txtGlobalSearch) {
    $bdQuickFind.Add_MouseLeftButtonDown({ $txtGlobalSearch.Focus() })
}

# --- TABS LOGIC ---
foreach ($btnName in $TabButtons) {
    (Get-Ctrl $btnName).Add_Click({
        param($s,$e)
        # Hide all panels
        foreach ($p in $Panels) { (Get-Ctrl $p).Visibility = "Collapsed" }
        # Reset all nav buttons
        foreach ($b in $TabButtons) { 
            $btn = Get-Ctrl $b
            $btn.ClearValue([System.Windows.Controls.Button]::BackgroundProperty)
            $btn.Foreground = "#8B949E"
            $btn.FontWeight = "Normal"
            $btn.Tag = "Collapsed"  # Hide indicator
        }
        # Show target panel (derive from button name: btnTabUpdates -> pnlUpdates)
        $panelName = $s.Name -replace "btnTab", "pnl"
        if ($s.Name -eq "btnNavDownloads") { $panelName = "pnlSupport" }
        $target = (Get-Ctrl $panelName)
        $target.Visibility = "Visible"
        # Set active state on clicked button
        $s.Background = "#21262D"
        $s.Foreground = "#E6EDF3"
        $s.FontWeight = "SemiBold"
        $s.Tag = "Visible"  # Show indicator
        if ($s.Name -eq "btnTabFirewall") { $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) }
        if ($s.Name -eq "btnTabUpdates") {
             if ($lstWinget.Items.Count -eq 0) { 
                $btnWingetScan.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
             }
        }
    })
}

# --- GLOBAL SEARCH ---
$SearchIndex = @{}
function Add-SearchIndexEntry { param($BtnName, $Desc, $ParentTab) $b=Get-Ctrl $BtnName; if($b){ $SearchIndex[$Desc]=@{Button=$b;Tab=$ParentTab} } }
# --- GLOBAL SEARCH INDEX ---

# 1. Updates (Winget)
Add-SearchIndexEntry "btnWingetScan"        "Check for Updates (Winget)"      "btnTabUpdates"
Add-SearchIndexEntry "btnWingetUpdateSel"   "Update Selected Apps"            "btnTabUpdates"
Add-SearchIndexEntry "btnWingetInstall"     "Install Selected Apps"           "btnTabUpdates"
Add-SearchIndexEntry "btnWingetUninstall"   "Uninstall Selected Apps"         "btnTabUpdates"
Add-SearchIndexEntry "btnWingetFind"        "Search Winget Packages"          "btnTabUpdates"

# 2. System Health
Add-SearchIndexEntry "btnSFC"               "SFC Scan (System File Checker)"  "btnTabHealth"
Add-SearchIndexEntry "btnDISMCheck"         "DISM Check Health"               "btnTabHealth"
Add-SearchIndexEntry "btnDISMRestore"       "DISM Restore Health"             "btnTabHealth"
Add-SearchIndexEntry "btnCHKDSK"            "CHKDSK (Check Disk)"             "btnTabHealth"

# 3. Network & DNS
Add-SearchIndexEntry "btnNetInfo"           "Show IP Config / Network Info"   "btnTabNetwork"
Add-SearchIndexEntry "btnFlushDNS"          "Flush DNS Cache"                 "btnTabNetwork"
Add-SearchIndexEntry "btnResetWifi"         "Restart Wi-Fi Adapter"           "btnTabNetwork"
Add-SearchIndexEntry "btnNetRepair"         "Full Network Repair (Reset IP)"  "btnTabNetwork"
Add-SearchIndexEntry "btnRouteTable"        "Save Routing Table"              "btnTabNetwork"
Add-SearchIndexEntry "btnRouteView"         "View Routing Table"              "btnTabNetwork"

# DNS Presets
Add-SearchIndexEntry "btnDnsGoogle"         "Set DNS: Google (8.8.8.8)"       "btnTabNetwork"
Add-SearchIndexEntry "btnDnsCloudflare"     "Set DNS: Cloudflare (1.1.1.1)"   "btnTabNetwork"
Add-SearchIndexEntry "btnDnsQuad9"          "Set DNS: Quad9 (Malware Block)"  "btnTabNetwork"
Add-SearchIndexEntry "btnDnsAuto"           "Reset DNS to Auto (DHCP)"        "btnTabNetwork"
Add-SearchIndexEntry "btnDnsCustom"         "Set Custom DNS Address"          "btnTabNetwork"

# DNS Encryption & Hosts
Add-SearchIndexEntry "btnDohAuto"           "Enable DoH (DNS over HTTPS)"     "btnTabNetwork"
Add-SearchIndexEntry "btnDohDisable"        "Disable DoH"                     "btnTabNetwork"
Add-SearchIndexEntry "btnHostsUpdate"       "Update Hosts (AdBlock)"          "btnTabNetwork"
Add-SearchIndexEntry "btnHostsEdit"         "Edit Hosts File"                 "btnTabNetwork"
Add-SearchIndexEntry "btnHostsBackup"       "Backup Hosts File"               "btnTabNetwork"
Add-SearchIndexEntry "btnHostsRestore"      "Restore Hosts File"              "btnTabNetwork"

# 4. Firewall
Add-SearchIndexEntry "btnFwRefresh"         "Refresh Firewall Rules"          "btnTabFirewall"
Add-SearchIndexEntry "btnFwAdd"             "Add New Firewall Rule"           "btnTabFirewall"
Add-SearchIndexEntry "btnFwEdit"            "Edit/Modify Firewall Rule"       "btnTabFirewall"
Add-SearchIndexEntry "btnFwExport"          "Export Firewall Policy"          "btnTabFirewall"
Add-SearchIndexEntry "btnFwImport"          "Import Firewall Policy"          "btnTabFirewall"
Add-SearchIndexEntry "btnFwDefaults"        "Restore Default Firewall Rules"  "btnTabFirewall"
Add-SearchIndexEntry "btnFwPurge"           "Delete All Firewall Rules"       "btnTabFirewall"

# 5. Drivers
Add-SearchIndexEntry "btnDrvReport"         "Generate Driver Report"          "btnTabDrivers"
Add-SearchIndexEntry "btnDrvBackup"         "Export Drivers"                  "btnTabDrivers"
Add-SearchIndexEntry "btnDrvGhost"          "Remove Ghost Devices"            "btnTabDrivers"
Add-SearchIndexEntry "btnDrvClean"          "Clean Old Drivers (DriverStore)" "btnTabDrivers"
Add-SearchIndexEntry "btnDrvRestore"        "Restore Drivers from Backup"     "btnTabDrivers"
Add-SearchIndexEntry "btnDrvDisableWU"      "Disable Driver Updates"          "btnTabDrivers"
Add-SearchIndexEntry "btnDrvEnableWU"       "Enable Driver Updates"           "btnTabDrivers"
Add-SearchIndexEntry "btnDrvDisableMeta"    "Disable Device Metadata"         "btnTabDrivers"
Add-SearchIndexEntry "btnDrvEnableMeta"     "Enable Device Metadata"          "btnTabDrivers"

# 6. Cleanup
Add-SearchIndexEntry "btnCleanDisk"         "Disk Cleanup Tool"               "btnTabCleanup"
Add-SearchIndexEntry "btnCleanTemp"         "Clean Temporary Files"           "btnTabCleanup"
Add-SearchIndexEntry "btnCleanShortcuts"    "Fix Broken Shortcuts"            "btnTabCleanup"
Add-SearchIndexEntry "btnCleanReg"          "Registry Cleanup & Backup"       "btnTabCleanup"
Add-SearchIndexEntry "btnCleanXbox"         "Clean Xbox Credentials"          "btnTabCleanup"

# 7. Utilities
Add-SearchIndexEntry "btnUtilSysInfo"       "System Info Report"              "btnTabUtils"
Add-SearchIndexEntry "btnUtilTrim"          "Trim SSD (Optimize)"             "btnTabUtils"
Add-SearchIndexEntry "btnUtilMas"           "MAS Activation"                  "btnTabUtils"
Add-SearchIndexEntry "btnUpdateRepair"      "Reset Windows Update Components" "btnTabUtils"
Add-SearchIndexEntry "btnUpdateServices"    "Restart Update Services"         "btnTabUtils"
Add-SearchIndexEntry "btnDotNetEnable"      "Set .NET RollForward"            "btnTabUtils"
Add-SearchIndexEntry "btnDotNetDisable"     "Reset .NET RollForward"          "btnTabUtils"
Add-SearchIndexEntry "btnTaskManager"       "Task Scheduler Manager"          "btnTabUtils"
Add-SearchIndexEntry "btnInstallGpedit"     "Install Group Policy (Home)"     "btnTabUtils"
Add-SearchIndexEntry "btnCtxBuilder" "Custom Context Menu Builder" "btnTabUtils"

# 8. Support
Add-SearchIndexEntry "btnSupportDiscord"    "Join Discord Support"            "btnTabSupport"
Add-SearchIndexEntry "btnSupportIssue"      "Report an Issue (GitHub)"        "btnTabSupport"

$txtGlobalSearch.Add_TextChanged({
    $q = $txtGlobalSearch.Text
    if ($q.Length -gt 1 -and $q -ne $script:QuickFindPlaceholder) {
        $pnlNavButtons.Visibility = "Collapsed"
        $lstSearchResults.Visibility = "Visible"
        $lstSearchResults.Items.Clear()
        $SearchIndex.GetEnumerator() | Where-Object { $_.Key -match "$q" } | ForEach-Object { [void]$lstSearchResults.Items.Add($_.Key) }
    } else { $pnlNavButtons.Visibility = "Visible"; $lstSearchResults.Visibility = "Collapsed" }
})
$lstSearchResults.Add_SelectionChanged({ if ($lstSearchResults.SelectedItem) { $match=$SearchIndex[$lstSearchResults.SelectedItem]; (Get-Ctrl $match.Tab).RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))); $txtGlobalSearch.Text="" } })


# WINGET CONTEXT MENU (Right-Click)
$ctxMenu = New-Object System.Windows.Controls.ContextMenu

# 1. Update Selected
$miUpdate = New-Object System.Windows.Controls.MenuItem
$miUpdate.Header = "Update Selected"
# We reference the button variable directly to ensure it works
$miUpdate.Add_Click({ 
    $btnWingetUpdateSel.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
})
[void]$ctxMenu.Items.Add($miUpdate)

# 2. Uninstall Selected
$miUninstall = New-Object System.Windows.Controls.MenuItem
$miUninstall.Header = "Uninstall Selected"
$miUninstall.Add_Click({ 
    $btnWingetUninstall.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
})
[void]$ctxMenu.Items.Add($miUninstall)

# --- Separator ---
[void]$ctxMenu.Items.Add((New-Object System.Windows.Controls.Separator))

# 3. Ignore Selected
$miIgnore = New-Object System.Windows.Controls.MenuItem
$miIgnore.Header = "Ignore Selected"
$miIgnore.Add_Click({ 
    $btnWingetIgnore.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
})
[void]$ctxMenu.Items.Add($miIgnore)

# 4. Manage Ignored
$miManage = New-Object System.Windows.Controls.MenuItem
$miManage.Header = "Manage Ignored List..."
$miManage.Add_Click({ 
    $btnWingetUnignore.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
})
[void]$ctxMenu.Items.Add($miManage)

# --- Separator ---
[void]$ctxMenu.Items.Add((New-Object System.Windows.Controls.Separator))

# 5. Refresh Updates
$miRefresh = New-Object System.Windows.Controls.MenuItem
$miRefresh.Header = "Refresh Updates"
$miRefresh.Add_Click({ 
    $btnWingetScan.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
})
[void]$ctxMenu.Items.Add($miRefresh)

# 6. Attach to List
$lstWinget.ContextMenu = $ctxMenu

# --- WINGET ---
$txtWingetSearch.Add_GotFocus({
    if ($txtWingetSearch.Text -in @("Search packages...", "Search new packages...")) { $txtWingetSearch.Text = "" }
})
$txtWingetSearch.Add_TextChanged({
    # If the user starts typing and the only item is our status message, clear it
    if ($lstWinget.Items.Count -eq 1 -and $lstWinget.Items[0].Name -eq "No updates available") {
        $lstWinget.Items.Clear()
    }
})
$txtWingetSearch.Add_KeyDown({ param($s, $e) if ($e.Key -eq "Return") { $btnWingetFind.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) } })

# 2. HELPER TO START JOB
$Script:StartWingetAction = {
    param($ListItems, $ActionName, $CmdTemplate)
    
    if (-not $ListItems -or $ListItems.Count -eq 0) { return }
    $uniqueItems = @($ListItems | Select-Object -Property Source, Name, Id -Unique)
    $totalItems = $uniqueItems.Count
    
    # UI Updates
    $btnWingetScan.IsEnabled = $false
    $btnWingetUpdateSel.IsEnabled = $false
    if ($btnWingetInstall) { $btnWingetInstall.IsEnabled = $false }
    if ($btnWingetUninstall) { $btnWingetUninstall.IsEnabled = $false }
    $lblWingetStatus.Text = "$ActionName in progress..."
    $lblWingetStatus.Visibility = "Visible"
    $script:WingetActiveAction = $ActionName
    $script:WingetActionStartedAt = Get-Date
    $script:WingetProgressTotal = $totalItems
    $script:WingetProgressDone = 0
    $script:WingetProgressSuccess = 0
    $script:WingetProgressSkipped = 0
    $script:WingetProgressFailed = 0
    $script:WingetCurrentIndex = 0
    $script:WingetCurrentItemName = ""
    $script:WingetCompletedIndexes = @{}
    if ($pbWingetProgress) {
        $pbWingetProgress.Minimum = 0
        $pbWingetProgress.Maximum = [Math]::Max(1, $totalItems)
        $pbWingetProgress.Value = 0
        $pbWingetProgress.IsIndeterminate = $false
        $pbWingetProgress.Visibility = "Visible"
    }
    if ($lblWingetProgress) {
        $lblWingetProgress.Text = "0/$totalItems done | 0 success | 0 skipped | 0 failed"
        $lblWingetProgress.Visibility = "Visible"
    }
    
    # 1. Define the Job Arguments
    $jobArgs = @{
        Items = $uniqueItems
        ActionName = $ActionName
        CmdTemplate = $CmdTemplate
        TempPath = $env:TEMP
    }

    # 2. Start the Background Job
    $script:WingetJob = Start-Job -ArgumentList $jobArgs -ScriptBlock {
        param($ArgsDict)
        $items = $ArgsDict.Items
        $act   = $ArgsDict.ActionName
        $tmpl  = $ArgsDict.CmdTemplate
        $temp  = $ArgsDict.TempPath
        
        [Console]::OutputEncoding = [System.Text.UTF8Encoding]::new($false)

        # --- ERROR DICTIONARY ---
        $ErrorCodes = @{
            "0"          = "Success"
            "0x0"        = "Success"
            "3010"       = "Reboot Required"
            "1602"       = "Cancelled by User"
            "1618"       = "Another Installation in Progress"
            
            # Winget Specific
            "0x8a150001" = "Invalid Argument"; "0x8a150002" = "Internal Failure"; "0x8a150003" = "Source Corrupted (Trying auto-fix...)"
            "0x8a150004" = "Installer Failed"; "0x8a150005" = "Hash Mismatch";    "0x8a150006" = "Not Applicable"
            "0x8a150007" = "Launch Failed";    "0x8a150008" = "Manifest Missing"; "0x8a150009" = "Invalid Manifest"
            "0x8a15000a" = "Unsupported Type"; "0x8a15000b" = "Package Not Found";"0x8a15000c" = "Vendor Error"
            "0x8a15000d" = "Download Failed";  "0x8a15000e" = "Installer Hash Mismatch"; "0x8a15000f" = "Data Missing"
            "0x8a150014" = "Network Error"
            "0x80070002" = "File Not Found";   "0x80070003" = "Path Not Found";   "0x80070005" = "Access Denied"
            "0x80070490" = "Element Not Found";"0x80072ee7" = "DNS Lookup Fail";  "0x80072f8f" = "SSL Cert Error"
            "1603"       = "Fatal MSI Error"
        }

        # Helper: Execute Command & Stream Output in Real-Time
        function Invoke-WingetCmd ($command) {
            $pInfo = New-Object System.Diagnostics.ProcessStartInfo
            $pInfo.FileName = "powershell.exe"
            $pInfo.Arguments = "-Command $command"
            $pInfo.RedirectStandardOutput = $true
            $pInfo.RedirectStandardError = $true
            $pInfo.UseShellExecute = $false
            $pInfo.CreateNoWindow = $true
            $proc = [System.Diagnostics.Process]::Start($pInfo)
            
            # Stream output/error in real-time while process runs
            while (-not $proc.HasExited) {
                while ($proc.StandardOutput.Peek() -gt -1) {
                    $line = $proc.StandardOutput.ReadLine()
                    if ($line) { Write-Output "LOG:  > $line" }
                }
                while ($proc.StandardError.Peek() -gt -1) {
                    $line = $proc.StandardError.ReadLine()
                    if ($line) { Write-Output "LOG:  ! $line" }
                }
                Start-Sleep -Milliseconds 100
            }
            
            # Get remaining output after exit
            $remaining = $proc.StandardOutput.ReadToEnd()
            if ($remaining) {
                foreach ($line in ($remaining -split "`r`n")) {
                    if ($line) { Write-Output "LOG:  > $line" }
                }
            }
            
            # Also capture any remaining errors
            $errOutput = $proc.StandardError.ReadToEnd()
            if ($errOutput) {
                foreach ($line in ($errOutput -split "`r`n")) {
                    if ($line) { Write-Output "LOG:  ! $line" }
                }
            }
            
            return $proc
        }

        function Invoke-WingetLive ($argsLine) {
            $pInfo = New-Object System.Diagnostics.ProcessStartInfo
            $pInfo.FileName = "winget"
            $pInfo.Arguments = $argsLine
            $pInfo.RedirectStandardOutput = $true
            $pInfo.RedirectStandardError = $true
            $pInfo.UseShellExecute = $false
            $pInfo.CreateNoWindow = $true
            $proc = [System.Diagnostics.Process]::Start($pInfo)

            $outBuf = New-Object System.Text.StringBuilder
            $errBuf = New-Object System.Text.StringBuilder
            $lastLine = ""
            $lastPct = -1
            $lastBeat = Get-Date

            while ((-not $proc.HasExited) -or $proc.StandardOutput.Peek() -gt -1 -or $proc.StandardError.Peek() -gt -1) {
                $hadData = $false

                while ($proc.StandardOutput.Peek() -gt -1) {
                    $hadData = $true
                    $ch = [char]$proc.StandardOutput.Read()
                    if ($ch -eq "`r" -or $ch -eq "`n") {
                        $line = $outBuf.ToString().Trim()
                        [void]$outBuf.Clear()
                        if ($line) {
                            if ($line -match "(\d+)%") {
                                $pct = [int]$matches[1]
                                if ($pct -ne $lastPct) {
                                    $lastPct = $pct
                                    Write-Output "LOG:  [winget] Progress: $pct%"
                                }
                            }
                            elseif ($line -ne $lastLine) {
                                $lastLine = $line
                                Write-Output "LOG:  > $line"
                            }
                        }
                    } else {
                        [void]$outBuf.Append($ch)
                    }
                }

                while ($proc.StandardError.Peek() -gt -1) {
                    $hadData = $true
                    $ch = [char]$proc.StandardError.Read()
                    if ($ch -eq "`r" -or $ch -eq "`n") {
                        $line = $errBuf.ToString().Trim()
                        [void]$errBuf.Clear()
                        if ($line) {
                            if ($line -match "(\d+)%") {
                                $pct = [int]$matches[1]
                                if ($pct -ne $lastPct) {
                                    $lastPct = $pct
                                    Write-Output "LOG:  [winget] Progress: $pct%"
                                }
                            }
                            elseif ($line -ne $lastLine) {
                                $lastLine = $line
                                Write-Output "LOG:  ! $line"
                            }
                        }
                    } else {
                        [void]$errBuf.Append($ch)
                    }
                }

                $now = Get-Date
                if ((-not $hadData) -and ((New-TimeSpan -Start $lastBeat -End $now).TotalSeconds -ge 8)) {
                    Write-Output "LOG:  [winget] still running..."
                    $lastBeat = $now
                }

                if (-not $hadData) {
                    Start-Sleep -Milliseconds 80
                }
            }

            $tailOut = $outBuf.ToString().Trim()
            if ($tailOut) { Write-Output "LOG:  > $tailOut" }
            $tailErr = $errBuf.ToString().Trim()
            if ($tailErr) { Write-Output "LOG:  ! $tailErr" }

            return $proc
        }

        function Invoke-VisibleCmd ($command, $title) {
            if ([string]::IsNullOrWhiteSpace($title)) { $title = "WMT Package Update" }
            $safeTitle = ($title -replace '"', '') -replace '[\r\n]', ' '
            $cmdLine = "/c title $safeTitle && $command"
            $proc = Start-Process -FilePath "cmd.exe" -ArgumentList $cmdLine -PassThru -Wait -WindowStyle Normal
            return $proc
        }

        $total = @($items).Count
        $index = 0
        foreach ($item in $items) {
            $index++
            $id   = $item.Id
            $name = $item.Name
            $src  = $item.Source
            $cmd  = ""
            $userCmd = "" 
            $wingetArgs = $null
            Write-Output "PROGRESS:${index}/${total}:$name"

            # --- COMMAND GENERATION ---
            if ($tmpl) {
                $cmd = $tmpl -f $id
                $userCmd = $cmd -replace "--disable-interactivity", "" 
            }
            else {
                # --- WINGET / MSSTORE ---
                if ($src -eq "winget" -or $src -eq "msstore") {
                    $flags = "--accept-source-agreements --accept-package-agreements --disable-interactivity"
                    $userFlags = "--accept-source-agreements --accept-package-agreements"
                    if ($act -eq "Install")   { $wingetArgs = "install --id `"$id`" $flags";   $cmd = "winget $wingetArgs"; $userCmd = "winget install --id `"$id`" $userFlags" }
                    if ($act -eq "Update")    { $wingetArgs = "upgrade --id `"$id`" $flags";   $cmd = "winget $wingetArgs"; $userCmd = "winget upgrade --id `"$id`" $userFlags" }
                    if ($act -eq "Uninstall") { $wingetArgs = "uninstall --id `"$id`" $flags"; $cmd = "winget $wingetArgs"; $userCmd = "winget uninstall --id `"$id`" $userFlags" }
                }
                # --- SCOOP (Likely requires User Mode) ---
                elseif ($src -eq "scoop") {
                    if ($act -eq "Install")   { $cmd = "scoop install `"$id`"" }
                    if ($act -eq "Update")    { $cmd = "scoop update `"$id`"" }
                    if ($act -eq "Uninstall") { $cmd = "scoop uninstall `"$id`"" }
                    $userCmd = $cmd # Scoop commands are same for user
                }
                # --- RUBY GEMS ---
                elseif ($src -eq "ruby" -or $src -eq "gem") {
                    if ($act -eq "Install")   { $cmd = "gem install `"$id`"" }
                    if ($act -eq "Update")    { $cmd = "gem update `"$id`"" }
                    if ($act -eq "Uninstall") { $cmd = "gem uninstall `"$id`"" }
                    $userCmd = $cmd
                }
                # --- RUST CARGO ---
                elseif ($src -eq "cargo" -or $src -eq "rust") {
                    if ($act -eq "Install")   { $cmd = "cargo install `"$id`"" }
                    if ($act -eq "Update")    { $cmd = "cargo install --force `"$id`"" } # Cargo needs force to overwrite/update binaries
                    if ($act -eq "Uninstall") { $cmd = "cargo uninstall `"$id`"" }
                    $userCmd = $cmd
                }
                # --- PYTHON PIP ---
                elseif ($src -eq "pip" -or $src -eq "pip3") {
                    if ($act -eq "Install")   { $cmd = "pip install `"$id`"" }
                    if ($act -eq "Update")    { $cmd = "pip install --upgrade `"$id`"" }
                    if ($act -eq "Uninstall") { $cmd = "pip uninstall -y `"$id`"" }
                    $userCmd = $cmd
                }
                # --- NODE NPM ---
                elseif ($src -eq "npm") {
                    if ($act -eq "Install")   { $cmd = "npm install -g `"$id`"" }
                    if ($act -eq "Update")    { $cmd = "npm update -g `"$id`"" }
                    if ($act -eq "Uninstall") { $cmd = "npm uninstall -g `"$id`"" }
                    $userCmd = $cmd
                }
                # --- CHOCOLATEY ---
                elseif ($src -eq "chocolatey" -or $src -eq "choco") {
                    if ($act -eq "Install")   { $cmd = "choco install `"$id`" -y" }
                    if ($act -eq "Update")    { $cmd = "choco upgrade `"$id`" -y" }
                    if ($act -eq "Uninstall") { $cmd = "choco uninstall `"$id`" -y" }
                    $userCmd = $cmd
                }
            }

            if ($cmd) {
                Write-Output "LOG:[$act][$index/$total] Starting: $name ($src)..."
                Write-Output "LOG:[$act] Command: $userCmd"

                # 1. RUN COMMAND (First Attempt - Admin)
                $isPipUpdate = (($src -eq "pip" -or $src -eq "pip3") -and $act -eq "Update")
                $isChocoUpdate = (($src -eq "chocolatey" -or $src -eq "choco") -and $act -eq "Update")
                $isPythonUpdate = ($act -eq "Update" -and (([string]$id -match "(?i)\bpython([0-9\.]*)\b") -or ([string]$name -match "(?i)\bpython([0-9\.]*)\b")))
                $useVisibleWindow = ($isPipUpdate -or $isChocoUpdate -or $isPythonUpdate)

                if ($useVisibleWindow) {
                    $windowTag = "Package"
                    if ($isPipUpdate) { $windowTag = "PIP" }
                    elseif ($isChocoUpdate) { $windowTag = "Chocolatey" }
                    elseif ($isPythonUpdate) { $windowTag = "Python" }
                    Write-Output "LOG:[$act] Launching visible $windowTag window for: $name"
                    $p = Invoke-VisibleCmd $cmd "WMT $windowTag Update - $name"
                } elseif ($wingetArgs -and ($src -eq "winget" -or $src -eq "msstore")) {
                    Write-Output "LOG:[$act] Running winget with live output..."
                    $p = Invoke-WingetLive $wingetArgs
                } else {
                    Write-Output "LOG:[$act] Running... (this may take a while)"
                    $p = Invoke-WingetCmd $cmd
                }
                Write-Output "LOG:[$act][$index/$total] Process completed with exit code: $($p.ExitCode)"
                
                $hex = "0x{0:x}" -f $p.ExitCode
                
                # --- AUTO-FIX: SOURCE CORRUPTION ---
                if ($hex -eq "0x8a150003") {
                    Write-Output "LOG:[$act] WARNING: Detected Winget Source Corruption. Auto-fixing..."
                    $fixP = Invoke-WingetCmd "winget source reset --force"
                    if ($fixP.ExitCode -eq 0) {
                        Write-Output "LOG:[$act][$index/$total] Sources reset. Retrying $name..."
                        if ($wingetArgs -and ($src -eq "winget" -or $src -eq "msstore")) {
                            $p = Invoke-WingetLive $wingetArgs
                        } else {
                            $p = Invoke-WingetCmd $cmd
                        }
                        $hex = "0x{0:x}" -f $p.ExitCode 
                    }
                }

                # --- CHECK FINAL RESULT ---
                if ($p.ExitCode -eq 0) {
                    Write-Output "LOG:[$act][$index/$total] SUCCESS: $name"
                    Write-Output "RESULT:${index}:SUCCESS:$name"
                }
                elseif ($p.ExitCode -eq 3010) {
                    Write-Output "LOG:[$act][$index/$total] SUCCESS: $name (Reboot Required to complete)"
                    Write-Output "RESULT:${index}:SUCCESS:$name"
                }
                elseif ($p.ExitCode -eq 1602) {
                    Write-Output "LOG:[$act][$index/$total] CANCELLED: $name (by User)"
                    Write-Output "RESULT:${index}:CANCELLED:$name"
                }
                else {
                    # --- FAILURE HANDLING ---
                    $dec = "$($p.ExitCode)"
                    $errDesc = "Unknown Error"
                    
                    if ($ErrorCodes.ContainsKey($hex)) { $errDesc = $ErrorCodes[$hex] }
                    elseif ($ErrorCodes.ContainsKey($dec)) { $errDesc = $ErrorCodes[$dec] }

                    # Known non-retry outcomes: avoid opening fallback user-mode consoles.
                    if ($hex -eq "0x8a150006") {
                        Write-Output "LOG:[$act][$index/$total] SKIPPED [$hex] ${errDesc} - $name"
                        Write-Output "RESULT:${index}:SKIPPED:$name"
                        continue
                    }
                    if ($hex -eq "0x8a15000b" -or $dec -eq "1618") {
                        Write-Output "LOG:[$act][$index/$total] FAILED [$hex] ${errDesc} - $name (no user-mode retry)"
                        Write-Output "RESULT:${index}:FAILED:$name"
                        continue
                    }

                    if ($useVisibleWindow) {
                        Write-Output "LOG:[$act][$index/$total] FAILED [$hex] ${errDesc} - $name (see visible window output)"
                        Write-Output "RESULT:${index}:FAILED:$name"
                        continue
                    }

                    if ($dec -eq "1603") {
                        Write-Output "LOG:[$act][$index/$total] FAILED (1603): $name - Retrying in Interactive Mode (Look for popup)..."
                    } else {
                        Write-Output "LOG:[$act][$index/$total] FAILED [$hex] $errDesc - Retrying as User..."
                    }
                    
                    # --- RETRY AS USER (Fallback) ---
                    # Handles Scoop (needs user rights) and Spotify (hates Admin)
                    $rand = [Guid]::NewGuid().ToString()
                    $batPath = "$temp\WMT_Fix_${id}_${rand}.bat"
                    $batContent = @"
@echo off
title $act $name (User Mode)
echo WMT-GUI: Re-launching $name...
echo.
echo NOTE: If this is Scoop or Spotify, this usually fixes the error.
echo If an installer window appears, please click through it.
echo.
$userCmd
echo.
echo Done.
timeout /t 5
(goto) 2>nul & del "%~f0"
"@
                    Set-Content -Path $batPath -Value $batContent -Encoding Ascii
                    try {
                        Start-Process "explorer.exe" -ArgumentList "`"$batPath`""
                    } catch {
                        Write-Output "LOG:Retry failed: $($_.Exception.Message)"
                    }
                    Write-Output "RESULT:${index}:FAILED:$name"
                }
            }
            else {
                Write-Output "LOG:[$act][$index/$total] SKIPPED: No command generated for source '$src' ($name)"
                Write-Output "RESULT:${index}:SKIPPED:$name"
            }
        }
    }

    # 3. Setup Timer
    if ($script:WingetTimer) { $script:WingetTimer.Stop() }
    $script:WingetTimer = New-Object System.Windows.Threading.DispatcherTimer
    $script:WingetTimer.Interval = [TimeSpan]::FromMilliseconds(500)
    $processWingetLines = {
        param($lines)
        foreach ($line in $lines) {
            if ($line -match "^RESULT:(\d+):(SUCCESS|SKIPPED|FAILED|CANCELLED):(.*)$") {
                $doneIndex = [int]$matches[1]
                $result = $matches[2]
                if (-not $script:WingetCompletedIndexes.ContainsKey($doneIndex)) {
                    $script:WingetCompletedIndexes[$doneIndex] = $true
                    $script:WingetProgressDone++
                    if ($result -eq "SUCCESS") {
                        $script:WingetProgressSuccess++
                    }
                    elseif ($result -eq "SKIPPED") {
                        $script:WingetProgressSkipped++
                    }
                    else {
                        $script:WingetProgressFailed++
                    }
                    if ($pbWingetProgress) { $pbWingetProgress.Value = [Math]::Min($pbWingetProgress.Maximum, $script:WingetProgressDone) }
                    if ($lblWingetProgress) {
                        $lblWingetProgress.Text = "$($script:WingetProgressDone)/$($script:WingetProgressTotal) done | $($script:WingetProgressSuccess) success | $($script:WingetProgressSkipped) skipped | $($script:WingetProgressFailed) failed"
                    }
                }
                continue
            }
            if ($line -match "^PROGRESS:(\d+)/(\d+):(.+)$") {
                $i = [int]$matches[1]
                $t = [int]$matches[2]
                $n = $matches[3]
                $script:WingetCurrentIndex = $i
                $script:WingetCurrentItemName = $n
                $lblWingetStatus.Text = "$($script:WingetActiveAction) ${i}/${t}: $n"
                if ($pbWingetProgress -and $i -gt 1 -and $pbWingetProgress.Value -lt ($i - 1)) {
                    $pbWingetProgress.Value = $i - 1
                }
                continue
            }
            if ($line -match "^LOG:(.*)") {
                $logMsg = $matches[1].Trim()
                if ($logMsg -match "^\[[^\]]+\]\[(\d+)/(\d+)\]\s+(SUCCESS|SKIPPED|FAILED|CANCELLED)\b") {
                    $doneIndex = [int]$matches[1]
                    $result = $matches[3]
                    if (-not $script:WingetCompletedIndexes.ContainsKey($doneIndex)) {
                        $script:WingetCompletedIndexes[$doneIndex] = $true
                        $script:WingetProgressDone++
                        if ($result -eq "SUCCESS") {
                            $script:WingetProgressSuccess++
                        }
                        elseif ($result -eq "SKIPPED") {
                            $script:WingetProgressSkipped++
                        }
                        else {
                            $script:WingetProgressFailed++
                        }
                        if ($pbWingetProgress) { $pbWingetProgress.Value = [Math]::Min($pbWingetProgress.Maximum, $script:WingetProgressDone) }
                        if ($lblWingetProgress) {
                            $lblWingetProgress.Text = "$($script:WingetProgressDone)/$($script:WingetProgressTotal) done | $($script:WingetProgressSuccess) success | $($script:WingetProgressSkipped) skipped | $($script:WingetProgressFailed) failed"
                        }
                    }
                }
                Write-GuiLog $logMsg
            }
        }
    }
    
    $script:WingetTimer.Add_Tick({
        if (-not $script:WingetJob) { return }
        if ($script:WingetJob.HasMoreData) {
            $results = Receive-Job -Job $script:WingetJob
            if ($results) {
                & $processWingetLines $results
            }
        }
        if ($script:WingetJob.State -eq 'Running' -and $script:WingetActionStartedAt -and $script:WingetCurrentItemName) {
            $elapsed = (Get-Date) - $script:WingetActionStartedAt
            $elapsedText = [string]::Format("{0:mm\\:ss}", $elapsed)
            $curIdx = [Math]::Max(1, $script:WingetCurrentIndex)
            $lblWingetStatus.Text = "$($script:WingetActiveAction) ${curIdx}/$($script:WingetProgressTotal): $($script:WingetCurrentItemName)  (${elapsedText})"
        }
        if ($script:WingetJob.State -ne 'Running') {
            $script:WingetTimer.Stop()
            $results = Receive-Job -Job $script:WingetJob
            if ($results) {
                & $processWingetLines $results
            }
            Remove-Job -Job $script:WingetJob
            $script:WingetJob = $null
            if ($script:WingetActiveAction) {
                if ($script:WingetProgressDone -lt $script:WingetProgressTotal) {
                    $missing = $script:WingetProgressTotal - $script:WingetProgressDone
                    $script:WingetProgressDone = $script:WingetProgressTotal
                    $script:WingetProgressFailed += $missing
                    Write-GuiLog "[$($script:WingetActiveAction)] Warning: $missing item(s) ended without explicit result; marked as failed."
                }
                $lblWingetStatus.Text = "$($script:WingetActiveAction) completed. $($script:WingetProgressDone)/$($script:WingetProgressTotal) done."
                Write-GuiLog "[$($script:WingetActiveAction)] Completed."
            }
            if ($pbWingetProgress) { $pbWingetProgress.Value = $script:WingetProgressDone }
            if ($lblWingetProgress) {
                $lblWingetProgress.Text = "$($script:WingetProgressDone)/$($script:WingetProgressTotal) done | $($script:WingetProgressSuccess) success | $($script:WingetProgressSkipped) skipped | $($script:WingetProgressFailed) failed"
            }
            $btnWingetScan.IsEnabled = $true
            $btnWingetUpdateSel.IsEnabled = $true
            if ($btnWingetInstall) { $btnWingetInstall.IsEnabled = $true }
            if ($btnWingetUninstall) { $btnWingetUninstall.IsEnabled = $true }
            if ($script:WingetProgressSuccess -gt 0) {
                Write-GuiLog "Action finished. $($script:WingetProgressSuccess) successful change(s). Refreshing package list..."
                if ($btnWingetScan) {
                    $btnWingetScan.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent)))
                }
            } else {
                Write-GuiLog "Action finished. No successful changes detected; skipping auto-refresh."
            }
            $script:WingetActiveAction = $null
        }
    })
    $script:WingetTimer.Start()
}

# 3. EVENT HANDLERS
function Show-ProviderManager {
    # 1. Load Current Settings
    $settings = Get-WmtSettings
    if (-not $settings.EnabledProviders) { 
        $settings | Add-Member -MemberType NoteProperty -Name "EnabledProviders" -Value @("winget", "msstore", "pip", "npm", "chocolatey") -Force
    }
    $enabled = $settings.EnabledProviders

    # 2. Define UI
    [xml]$pXaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        Title="Package Manager Settings" Height="550" Width="500" WindowStartupLocation="CenterScreen" ResizeMode="NoResize" Background="#252526">
    <Window.Resources>
        <Style TargetType="TextBlock"><Setter Property="Foreground" Value="#DDD"/><Setter Property="VerticalAlignment" Value="Center"/></Style>
        <Style TargetType="CheckBox"><Setter Property="Foreground" Value="White"/><Setter Property="VerticalAlignment" Value="Center"/><Setter Property="Margin" Value="0,0,10,0"/></Style>
    </Window.Resources>
    <Grid Margin="20">
        <Grid.RowDefinitions><RowDefinition Height="Auto"/><RowDefinition Height="*"/><RowDefinition Height="Auto"/></Grid.RowDefinitions>
        
        <TextBlock Text="Manage Package Providers" FontSize="18" FontWeight="Bold" Margin="0,0,0,15"/>
        <TextBlock Text="Select which package managers to scan." Foreground="#AAA" Margin="0,25,0,0" Grid.Row="0"/>

        <StackPanel Grid.Row="1" Margin="0,15,0,0">
            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkWinget" IsChecked="True" IsEnabled="False" Grid.Column="0"/>
                <TextBlock Text="Winget" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Text="Windows Package Manager" Foreground="#888" FontStyle="Italic" Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkMsStore" Grid.Column="0"/>
                <TextBlock Text="MS Store" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Text="Microsoft Store Apps" Foreground="#888" FontStyle="Italic" Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkPip" Grid.Column="0"/>
                <TextBlock Text="Python (Pip)" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Name="lblPipStatus" Text="Checking..." Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkNpm" Grid.Column="0"/>
                <TextBlock Text="Node (Npm)" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Name="lblNpmStatus" Text="Checking..." Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkChoco" Grid.Column="0"/>
                <TextBlock Text="Chocolatey" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Name="lblChocoStatus" Text="Checking..." Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkScoop" Grid.Column="0"/>
                <TextBlock Text="Scoop" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Name="lblScoopStatus" Text="Checking..." Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkGem" Grid.Column="0"/>
                <TextBlock Text="Ruby (Gem)" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Name="lblGemStatus" Text="Checking..." Grid.Column="2"/>
            </Grid>

            <Grid Margin="0,0,0,10"><Grid.ColumnDefinitions><ColumnDefinition Width="30"/><ColumnDefinition Width="100"/><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/></Grid.ColumnDefinitions>
                <CheckBox Name="chkCargo" Grid.Column="0"/>
                <TextBlock Text="Rust (Cargo)" FontWeight="Bold" Grid.Column="1"/>
                <TextBlock Name="lblCargoStatus" Text="Checking..." Grid.Column="2"/>
            </Grid>
        </StackPanel>

        <StackPanel Grid.Row="2" Orientation="Horizontal" HorizontalAlignment="Right">
            <Button Name="btnSave" Content="Save &amp; Close" Background="#007ACC" Width="120" Height="30" Foreground="White"/>
        </StackPanel>
    </Grid>
</Window>
"@
    $reader = (New-Object System.Xml.XmlNodeReader $pXaml)
    $win = [Windows.Markup.XamlReader]::Load($reader)

    function Get-WinCtrl($name) { $win.FindName($name) }
    
    $chkMsStore = Get-WinCtrl "chkMsStore"
    $chkPip     = Get-WinCtrl "chkPip"
    $chkNpm     = Get-WinCtrl "chkNpm"
    $chkChoco   = Get-WinCtrl "chkChoco"
    $chkScoop   = Get-WinCtrl "chkScoop"
    $chkGem     = Get-WinCtrl "chkGem"
    $chkCargo   = Get-WinCtrl "chkCargo"
    
    # Load settings
    if ("msstore" -in $enabled)    { $chkMsStore.IsChecked = $true }
    if ("pip" -in $enabled)        { $chkPip.IsChecked = $true }
    if ("npm" -in $enabled)        { $chkNpm.IsChecked = $true }
    if ("chocolatey" -in $enabled) { $chkChoco.IsChecked = $true }
    if ("scoop" -in $enabled)      { $chkScoop.IsChecked = $true }
    if ("gem" -in $enabled)        { $chkGem.IsChecked = $true }
    if ("cargo" -in $enabled)      { $chkCargo.IsChecked = $true }

    # Helper for status check
    function Set-Status($cmd, $lbl) {
        if (Get-Command $cmd -ErrorAction SilentlyContinue) {
            (Get-WinCtrl $lbl).Text = "Installed"
            (Get-WinCtrl $lbl).Foreground = "LightGreen"
        } else {
            (Get-WinCtrl $lbl).Text = "Not Found"
            (Get-WinCtrl $lbl).Foreground = "Orange"
        }
    }

    Set-Status "pip" "lblPipStatus"
    Set-Status "npm" "lblNpmStatus"
    Set-Status "choco" "lblChocoStatus"
    Set-Status "scoop" "lblScoopStatus"
    Set-Status "gem" "lblGemStatus"
    Set-Status "cargo" "lblCargoStatus"

    # Save Event
    (Get-WinCtrl "btnSave").Add_Click({
        $newEnabled = @("winget") 
        if ($chkMsStore.IsChecked) { $newEnabled += "msstore" }
        if ($chkPip.IsChecked)     { $newEnabled += "pip" }
        if ($chkNpm.IsChecked)     { $newEnabled += "npm" }
        if ($chkChoco.IsChecked)   { $newEnabled += "chocolatey" }
        if ($chkScoop.IsChecked)   { $newEnabled += "scoop" }
        if ($chkGem.IsChecked)     { $newEnabled += "gem" }
        if ($chkCargo.IsChecked)   { $newEnabled += "cargo" }
        
        $current = Get-WmtSettings
        $current.EnabledProviders = $newEnabled
        Save-WmtSettings -Settings $current
        $win.Close()
    })

    $win.ShowDialog() | Out-Null
}

# ---------------------------------------------------------
# PARALLEL SCAN ENGINE
# ---------------------------------------------------------

# 1. SETUP MULTI-THREAD TIMER
$script:ScanTimer = New-Object System.Windows.Threading.DispatcherTimer
$script:ScanTimer.Interval = [TimeSpan]::FromMilliseconds(200)
# We now use a LIST of active scanners instead of just one
$script:ActiveScans = [System.Collections.ArrayList]::new()

$script:ScanTimer.Add_Tick({
    if ($script:ActiveScans.Count -gt 0) {
        
        # Iterate backwards so we can remove completed tasks safely
        for ($i = $script:ActiveScans.Count - 1; $i -ge 0; $i--) {
            $task = $script:ActiveScans[$i]
            
            if ($task.AsyncResult.IsCompleted) {
                try {
                    $results = $task.PowerShell.EndInvoke($task.AsyncResult)
                    $task.PowerShell.Dispose()
                    
                    # Process Results
                    foreach ($item in $results) {
                        if ($null -eq $item) { continue }

                        if ($item -is [string] -and $item.StartsWith("LOG:")) {
                            Write-GuiLog ($item.Substring(4))
                        }
                        elseif ($item.PSObject.Properties["Name"]) {
                            # --- STRICT GARBAGE FILTER ---
                            # Rejects headers, separators, and empty dashed lines
                            if ($item.Name -match "^-+$" -or $item.Id -match "^-+$") { continue }
                            if ($item.Name -eq "Name" -and $item.Id -eq "Id") { continue }
                            if ($item.Name -eq "Source" -and $item.Id -eq "Name") { continue }
                            if ($item.Version -eq "Version" -or $item.Available -eq "Available") { continue }
                            
                            # Add to UI
                            [void]$lstWinget.Items.Add($item)
                        }
                    }
                } catch {
                    Write-GuiLog "Scan Error: $($_.Exception.Message)"
                }
                
                # Remove finished task
                $script:ActiveScans.RemoveAt($i)
            }
        }
    } else {
        # All tasks finished
        if ($script:ScanTimer.IsEnabled) {
            $script:ScanTimer.Stop()
            
            $lblWingetStatus.Visibility = "Hidden"
            if ($pbWingetProgress) { $pbWingetProgress.Visibility = "Collapsed"; $pbWingetProgress.Value = 0 }
            if ($lblWingetProgress) { $lblWingetProgress.Visibility = "Collapsed"; $lblWingetProgress.Text = "" }
            $btnWingetScan.IsEnabled = $true
            if ($btnWingetUpdateAll) { $btnWingetUpdateAll.IsEnabled = $true }
            $btnWingetUpdateSel.IsEnabled = $true
            $lstWinget.Items.Refresh()
            $lstWinget.UpdateLayout()
            
            if ($lstWinget.Items.Count -eq 0) {
                 Write-GuiLog "System is up to date."
            } else {
                 Write-GuiLog "Scan Complete. Found $($lstWinget.Items.Count) updates."
            }
        }
    }
})

# 2. BUTTON CLICK (Launch Parallel Threads)
$btnWingetScan.Add_Click({
    # UI Prep
    $lblWingetTitle.Text = "Package Updates"
    $lblWingetStatus.Text = "Scanning all providers..."
    $lblWingetStatus.Visibility = "Visible"
    if ($pbWingetProgress) { $pbWingetProgress.Visibility = "Collapsed"; $pbWingetProgress.Value = 0 }
    if ($lblWingetProgress) { $lblWingetProgress.Visibility = "Collapsed"; $lblWingetProgress.Text = "" }
    $lstWinget.Items.Clear()
    $btnWingetScan.IsEnabled = $false
    if ($btnWingetUpdateAll) { $btnWingetUpdateAll.IsEnabled = $false }
    $btnWingetUpdateSel.IsEnabled = $false
    $btnWingetInstall.Visibility = "Collapsed"
    $btnWingetUpdateSel.Visibility = "Visible"
    
    Write-GuiLog " "
    Write-GuiLog "Starting Parallel Scan..."

    # Get Settings
    $settings = Get-WmtSettings
    $enabled = if ($settings.EnabledProviders) { $settings.EnabledProviders } else { @("winget") }
    $ignoreList = if ($settings.WingetIgnore) { $settings.WingetIgnore } else { @() }

    # Reset Task List
    $script:ActiveScans.Clear()

    # --- DEFINE PROVIDER WORKERS ---

    # A. WINGET & MSSTORE WORKER
    if ("winget" -in $enabled -or "msstore" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($Enabled, $IgnoreList)
            [Console]::OutputEncoding = [System.Text.UTF8Encoding]::new($false)
            function Test-Ignored($n, $i) {
                if ($IgnoreList -and ($IgnoreList -contains $n -or $IgnoreList -contains $i)) { return $true }
                return $false
            }

            # WINGET / MSSTORE
            # REMOVED: --include-unknown (causes hangs on apps with unknown versions)
            # ADDED: --disable-interactivity to prevent any prompts
            $wArgs = "list --upgrade-available --accept-source-agreements --disable-interactivity"
            if ("msstore" -notin $Enabled) { $wArgs += " --source winget" }
            
            try {
                $pInfo = New-Object System.Diagnostics.ProcessStartInfo
                $pInfo.FileName = "winget"
                $pInfo.Arguments = $wArgs
                $pInfo.RedirectStandardOutput = $true
                $pInfo.RedirectStandardError = $true
                $pInfo.UseShellExecute = $false
                $pInfo.CreateNoWindow = $true
                $pInfo.StandardOutputEncoding = [System.Text.UTF8Encoding]::new($false)
                
                $p = [System.Diagnostics.Process]::Start($pInfo)
                # ADDED: 60 second timeout to prevent hanging forever
                if (-not $p.WaitForExit(60000)) {
                    Write-Output "LOG:Winget scan timed out after 60 seconds."
                    try { $p.Kill() } catch {}
                    return
                }
                $out = $p.StandardOutput.ReadToEnd()
                
                if (-not [string]::IsNullOrWhiteSpace($out)) {
                    $lines = $out -split "`r`n"
                    $idxId = -1
                    foreach($l in $lines){ if($l -match "Name\s+Id\s+Version"){$idxId=$l.IndexOf("Id"); break} }

                    foreach($line in $lines){
                        $line=$line.Trim()
                        if(!$line -or $line -match "^-+" -or $line -match "^Name\s+Id"){continue}
                        # Filter winget status messages that look like data rows
                        if($line -match "explicit targeting" -or $line -match "following packages have an upgrade available"){continue}
                        $n=$null;$i=$null;$v=$null;$a=$null;$s="winget"
                        
                        if($idxId -gt 0 -and $line.Length -gt $idxId){
                            $n=$line.Substring(0,$idxId).Trim(); $rest=$line.Substring($idxId).Trim(); $parts=$rest -split "\s+"
                            if($parts.Count -ge 4){$i=$parts[0];$v=$parts[1];$a=$parts[2];$s=$parts[3]}
                            elseif($parts.Count -ge 3){$i=$parts[0];$v=$parts[1];$a=$parts[2]}
                        }
                        if(!$n -and $line -match '^(.+)\s+([^\s]+)\s+([^\s]+)\s+([^\s]+)\s+([^\s]+)'){
                            $n=$matches[1].Trim();$i=$matches[2].Trim();$v=$matches[3].Trim();$a=$matches[4].Trim();$s=$matches[5].Trim()
                        }
                        if($n -and $i -and $i.Length -gt 2){
                            # CHANGED: Updated function call here
                            if(Test-Ignored $n $i){continue}
                            
                            if($s -eq "msstore"){$s="msstore"}else{$s="winget"}
                            if($v -eq "Unknown"){$v="?"}
                            [PSCustomObject]@{Source=$s;Name=$n;Id=$i;Version=$v;Available=$a}
                        }
                    }
                }
            } catch { Write-Output "LOG:Winget check failed." }
        }).AddArgument($enabled).AddArgument($ignoreList)
        
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # B. PIP WORKER
    if ("pip" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($IgnoreList)
            Write-Output "LOG:Scanning Pip..."
            try {
                $pInfo=New-Object System.Diagnostics.ProcessStartInfo("cmd", "/c pip list --outdated --format=json"); $pInfo.RedirectStandardOutput=$true; $pInfo.UseShellExecute=$false; $pInfo.CreateNoWindow=$true
                $p=[System.Diagnostics.Process]::Start($pInfo)
                if (-not $p.WaitForExit(30000)) { try { $p.Kill() } catch {}; Write-Output "LOG:Pip scan timed out."; return }
                $json=$p.StandardOutput.ReadToEnd()
                if($json.Trim().StartsWith("[")){ 
                    $pkgs=$json|ConvertFrom-Json; foreach($p in $pkgs){ if($IgnoreList -contains $p.name){continue}; [PSCustomObject]@{Source="pip";Name=$p.name;Id=$p.name;Version=$p.version;Available=$p.latest_version} } 
                }
            } catch { Write-Output "LOG:Pip check failed." }
        }).AddArgument($ignoreList)
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # C. NPM WORKER
    if ("npm" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($IgnoreList)
            Write-Output "LOG:Scanning Npm..."
            try {
                $pInfo=New-Object System.Diagnostics.ProcessStartInfo("cmd", "/c npm outdated --json"); $pInfo.RedirectStandardOutput=$true; $pInfo.UseShellExecute=$false; $pInfo.CreateNoWindow=$true
                $p=[System.Diagnostics.Process]::Start($pInfo)
                if (-not $p.WaitForExit(30000)) { try { $p.Kill() } catch {}; Write-Output "LOG:Npm scan timed out."; return }
                $json=$p.StandardOutput.ReadToEnd()
                if($json.Trim().StartsWith("{")){ 
                    $pkgs=$json|ConvertFrom-Json; foreach($k in $pkgs.PSObject.Properties.Name){ if($IgnoreList -contains $k){continue}; $o=$pkgs.$k; [PSCustomObject]@{Source="npm";Name=$k;Id=$k;Version=$o.current;Available=$o.latest} } 
                }
            } catch { Write-Output "LOG:Npm check failed." }
        }).AddArgument($ignoreList)
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # D. CHOCOLATEY WORKER
    if ("chocolatey" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($IgnoreList)
            Write-Output "LOG:Scanning Chocolatey..."
            try {
                $pInfo=New-Object System.Diagnostics.ProcessStartInfo("choco", "outdated -r"); $pInfo.RedirectStandardOutput=$true; $pInfo.UseShellExecute=$false; $pInfo.CreateNoWindow=$true
                $p=[System.Diagnostics.Process]::Start($pInfo)
                if (-not $p.WaitForExit(60000)) { try { $p.Kill() } catch {}; Write-Output "LOG:Chocolatey scan timed out."; return }
                $out=$p.StandardOutput.ReadToEnd()
                $lines=$out -split "`r`n"
                foreach($l in $lines){ 
                    if(!$l){continue}; $p=$l -split "\|"; if($p.Count -ge 4){ $n=$p[0]; if($IgnoreList -contains $n){continue}; [PSCustomObject]@{Source="chocolatey";Name=$n;Id=$n;Version=$p[1];Available=$p[2]} } 
                }
            } catch { Write-Output "LOG:Choco check failed." }
        }).AddArgument($ignoreList)
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # E. SCOOP WORKER
    if ("scoop" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($IgnoreList)
            Write-Output "LOG:Scanning Scoop..."
            try {
                $pInfo = New-Object System.Diagnostics.ProcessStartInfo("cmd", "/c scoop status")
                $pInfo.RedirectStandardOutput = $true
                $pInfo.UseShellExecute = $false
                $pInfo.CreateNoWindow = $true
                
                $p = [System.Diagnostics.Process]::Start($pInfo)
                if (-not $p.WaitForExit(30000)) { try { $p.Kill() } catch {}; Write-Output "LOG:Scoop scan timed out."; return }
                $out = $p.StandardOutput.ReadToEnd()

                $lines = $out -split "`r`n"
                foreach ($line in $lines) {
                    $line = $line.Trim()
                    if ([string]::IsNullOrWhiteSpace($line) -or $line -match "^Name\s+Version" -or $line -match "^----" -or $line -match "Scoop is up to date") { continue }
                    
                    if ($line -match '^(\S+)\s+(\S+)\s+(\S+)') {
                        $n = $matches[1]; $v = $matches[2]; $l = $matches[3]
                        if ($IgnoreList -contains $n) { continue }
                        [PSCustomObject]@{ Source="scoop"; Name=$n; Id=$n; Version=$v; Available=$l }
                    }
                }
            } catch { Write-Output "LOG:Scoop check failed." }
        }).AddArgument($ignoreList)
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # F. RUBY GEMS WORKER
    if ("gem" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($IgnoreList)
            Write-Output "LOG:Scanning Ruby Gems..."
            try {
                $pInfo = New-Object System.Diagnostics.ProcessStartInfo("cmd", "/c gem outdated")
                $pInfo.RedirectStandardOutput = $true
                $pInfo.UseShellExecute = $false
                $pInfo.CreateNoWindow = $true
                
                $p = [System.Diagnostics.Process]::Start($pInfo)
                if (-not $p.WaitForExit(30000)) { try { $p.Kill() } catch {}; Write-Output "LOG:Gem scan timed out."; return }
                $out = $p.StandardOutput.ReadToEnd()

                $lines = $out -split "`r`n"
                foreach ($line in $lines) {
                    if ($line -match '^(\S+)\s+\(([\d\.]+)\s+<\s+([\d\.]+)\)') {
                        $n = $matches[1]; $v = $matches[2]; $a = $matches[3]
                        if ($IgnoreList -contains $n) { continue }
                        [PSCustomObject]@{ Source="gem"; Name=$n; Id=$n; Version=$v; Available=$a }
                    }
                }
            } catch { Write-Output "LOG:Gem check failed." }
        }).AddArgument($ignoreList)
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # G. CARGO WORKER (RUST)
    if ("cargo" -in $enabled) {
        $ps = [PowerShell]::Create()
        [void]$ps.AddScript({
            param($IgnoreList)
            Write-Output "LOG:Scanning Cargo..."
            try {
                $pInfo = New-Object System.Diagnostics.ProcessStartInfo("cmd", "/c cargo install --list")
                $pInfo.RedirectStandardOutput = $true
                $pInfo.UseShellExecute = $false
                $pInfo.CreateNoWindow = $true
                
                $p = [System.Diagnostics.Process]::Start($pInfo)
                if (-not $p.WaitForExit(30000)) { try { $p.Kill() } catch {}; Write-Output "LOG:Cargo scan timed out."; return }
                $out = $p.StandardOutput.ReadToEnd()

                $lines = $out -split "`r`n"
                foreach ($line in $lines) {
                    if ($line -match '^([a-zA-Z0-9_\-]+)\s+v([\d\.]+):') {
                        $n = $matches[1]; $v = $matches[2]
                        if ($IgnoreList -contains $n) { continue }
                        [PSCustomObject]@{ Source="cargo"; Name=$n; Id=$n; Version=$v; Available="?" }
                    }
                }
            } catch { Write-Output "LOG:Cargo check failed." }
        }).AddArgument($ignoreList)
        [void]$script:ActiveScans.Add([PSCustomObject]@{ PowerShell=$ps; AsyncResult=$ps.BeginInvoke() })
    }

    # Start Timer to monitor all threads
    $script:ScanTimer.Start()
})

# --- IGNORE SELECTED ---
$btnWingetIgnore.Add_Click({
    $selected = @($lstWinget.SelectedItems)
    if ($selected.Count -eq 0) { return }

    $msg = "Ignore $($selected.Count) package(s)?`n`nThese updates will be hidden from future scans."
    if ([System.Windows.Forms.MessageBox]::Show($msg, "Ignore Updates", "YesNo", "Question") -eq "Yes") {
        
        # 1. Get fresh settings
        $settings = Get-WmtSettings
        
        # 2. Create a fresh ArrayList to ensure it is editable
        $newList = New-Object System.Collections.ArrayList
        
        # Add existing items (checking for nulls)
        if ($settings.WingetIgnore) {
            foreach ($existing in $settings.WingetIgnore) {
                if (-not [string]::IsNullOrWhiteSpace($existing)) {
                    [void]$newList.Add($existing.ToString())
                }
            }
        }

        # 3. Add NEW items
        foreach ($item in $selected) {
            $id = $item.Id
            # Avoid duplicates
            if ($id -and ($id -notin $newList)) {
                [void]$newList.Add($id)
            }
            # Remove from GUI immediately
            $lstWinget.Items.Remove($item)
        }

        # 4. Save back as a standard array
        $settings.WingetIgnore = $newList.ToArray()
        Save-WmtSettings -Settings $settings
        
        Write-GuiLog "Ignored $($selected.Count) packages. Saved to settings.json."
    }
})

# --- MANAGE IGNORED (UNIGNORE) ---
$btnWingetUnignore.Add_Click({
    # 1. READ SETTINGS (Direct & Simple)
    $jsonPath = Join-Path (Get-DataPath) "settings.json"
    $listItems = @()

    if (Test-Path $jsonPath) {
        try {
            $json = Get-Content $jsonPath -Raw | ConvertFrom-Json
            if ($json.WingetIgnore) {
                # Force array and string conversion immediately
                $listItems = @($json.WingetIgnore) | ForEach-Object { "$_".Trim() } | Where-Object { $_ -ne "" }
            }
        } catch { Write-GuiLog "Error: $($_.Exception.Message)" }
    }

    if ($listItems.Count -eq 0) {
        [System.Windows.Forms.MessageBox]::Show("No ignored packages found.", "Manage Ignored", "OK", "Information") | Out-Null
        return
    }

    # 2. UI SETUP
    $f = New-Object System.Windows.Forms.Form
    $f.Text = "Manage Ignored Updates"
    $f.Size = "500, 400"
    $f.StartPosition = "CenterScreen"
    $f.BackColor = [System.Drawing.Color]::FromArgb(32,32,32)
    $f.ForeColor = "White"

    # 3. CONTROLS (Critical Order for Docking)
    # Add Dock=Bottom/Top controls FIRST so they claim space. Add Dock=Fill LAST.

    # -- Bottom Panel --
    $pnl = New-Object System.Windows.Forms.Panel
    $pnl.Dock = "Bottom"; $pnl.Height = 50
    $f.Controls.Add($pnl)

    # -- Top Label --
    $lbl = New-Object System.Windows.Forms.Label
    $lbl.Text = "Select packages to restore (Un-ignore):"
    $lbl.Dock = "Top"; $lbl.Height = 30; $lbl.Padding = "10,10,0,0"
    $f.Controls.Add($lbl)

    # -- ListBox (Fill) --
    $lb = New-Object System.Windows.Forms.ListBox
    $lb.Dock = "Fill"
    $lb.BackColor = [System.Drawing.Color]::FromArgb(20,20,20)
    $lb.ForeColor = "White"
    $lb.BorderStyle = "FixedSingle"
    $lb.SelectionMode = "MultiExtended"
    
    # Manual Add (Fail-safe)
    $lb.BeginUpdate()
    foreach ($item in $listItems) {
        [void]$lb.Items.Add($item)
    }
    $lb.EndUpdate()
    
    $f.Controls.Add($lb)
    
    # CRITICAL: Ensure ListBox is at the 'top' of Z-order so it fills remaining space correctly
    $lb.BringToFront()

    # -- Buttons --
    $btnRestore = New-Object System.Windows.Forms.Button
    $btnRestore.Text = "Un-Ignore Selected"
    $btnRestore.Size = "150, 30"; $btnRestore.Location = "20, 10"
    $btnRestore.BackColor = "SeaGreen"; $btnRestore.ForeColor = "White"; $btnRestore.FlatStyle = "Flat"
    $pnl.Controls.Add($btnRestore)

    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Size = "100, 30"; $btnClose.Location = "360, 10"
    $btnClose.BackColor = "DimGray"; $btnClose.ForeColor = "White"; $btnClose.FlatStyle = "Flat"
    $btnClose.Add_Click({ $f.Close() })
    $pnl.Controls.Add($btnClose)

    # 4. ACTION
    $script:RefreshNeeded = $false

    $btnRestore.Add_Click({
        $selected = @($lb.SelectedItems)
        if ($selected.Count -gt 0) {
            # Read fresh, remove items, save
            $s = Get-WmtSettings
            $current = [System.Collections.ArrayList]@($s.WingetIgnore)
            
            foreach ($item in $selected) {
                if ($current.Contains($item)) { $current.Remove($item) }
                $lb.Items.Remove($item)
            }
            
            $s.WingetIgnore = $current.ToArray()
            Save-WmtSettings -Settings $s
            $script:RefreshNeeded = $true
        }
    })

    $f.ShowDialog() | Out-Null
    
    if ($script:RefreshNeeded) {
        Write-GuiLog "List updated. Refreshing..."
        $btnWingetScan.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent)))
    }
})

# --- LISTVIEW SORTING LOGIC ---
$lstWinget = Get-Ctrl "lstWinget"
$script:WingetSortChain = New-Object System.Collections.ArrayList

function Get-CleanHeader {
    param([object]$Header)
    if ($null -eq $Header) { return "" }
    return ([string]$Header -replace '\s+[▲▼]$', '').Trim()
}

function Update-GridViewHeaders {
    param([System.Windows.Controls.ListView]$ListView, [string]$ActiveHeader, [bool]$Ascending)
    if (-not $ListView -or -not $ListView.View -or -not ($ListView.View -is [System.Windows.Controls.GridView])) { return }
    foreach ($col in $ListView.View.Columns) {
        $clean = Get-CleanHeader $col.Header
        if ($clean -eq $ActiveHeader) {
            $col.Header = if ($Ascending) { "$clean ▲" } else { "$clean ▼" }
        } else {
            $col.Header = $clean
        }
    }
}

function Set-SortChainPrimary {
    param(
        [System.Collections.ArrayList]$Chain,
        [string]$PropertyName
    )
    if (-not $Chain -or [string]::IsNullOrWhiteSpace($PropertyName)) { return $false }

    $existingIndex = -1
    for ($i = 0; $i -lt $Chain.Count; $i++) {
        if ($Chain[$i].Property -eq $PropertyName) { $existingIndex = $i; break }
    }

    if ($existingIndex -eq 0) {
        $Chain[0].Descending = -not [bool]$Chain[0].Descending
        return [bool](-not $Chain[0].Descending)
    }

    if ($existingIndex -gt 0) { $Chain.RemoveAt($existingIndex) }
    $entry = [PSCustomObject]@{ Property = $PropertyName; Descending = $false }
    $Chain.Insert(0, $entry)
    return $true
}

function Set-ListViewSort {
    param(
        [System.Windows.Controls.ListView]$ListView,
        [System.Collections.ArrayList]$Chain
    )
    if (-not $ListView -or -not $Chain -or $Chain.Count -eq 0) { return }
    $items = @($ListView.Items)
    if ($items.Count -eq 0) { return }

    $sortSpec = foreach ($rule in $Chain) {
        @{ Expression = $rule.Property; Descending = [bool]$rule.Descending }
    }
    $sorted = $items | Sort-Object -Property $sortSpec
    $ListView.Items.Clear()
    foreach ($item in $sorted) { [void]$ListView.Items.Add($item) }
}

function Resolve-WingetSortProperty {
    param([string]$Header)
    switch ($Header) {
        "Package Name" { return "Name" }
        "Installed"    { return "Version" }
        "Latest"       { return "Available" }
        default        { return $Header }
    }
}

if ($lstWinget) {
    $wingetSortHandler = [System.Windows.RoutedEventHandler]{
        param($src, $e)
        if ($e.OriginalSource -isnot [System.Windows.Controls.GridViewColumnHeader]) { return }
        $header = Get-CleanHeader $e.OriginalSource.Column.Header
        if ([string]::IsNullOrWhiteSpace($header)) { return }

        $propName = Resolve-WingetSortProperty $header
        if ([string]::IsNullOrWhiteSpace($propName)) { return }

        $isAscending = Set-SortChainPrimary -Chain $script:WingetSortChain -PropertyName $propName
        Update-GridViewHeaders -ListView $lstWinget -ActiveHeader $header -Ascending:$isAscending
        Set-ListViewSort -ListView $lstWinget -Chain $script:WingetSortChain
    }
    $lstWinget.AddHandler([System.Windows.Controls.Primitives.ButtonBase]::ClickEvent, $wingetSortHandler)
}

# ---------------------------------------------------------
# HIGH-PERFORMANCE THREADED SEARCH
# ---------------------------------------------------------

# 1. SETUP UI TIMER (Handles the Thread Callback)
$script:SearchTimer = New-Object System.Windows.Threading.DispatcherTimer
$script:SearchTimer.Interval = [TimeSpan]::FromMilliseconds(100)
$script:AsyncSearch = $null
$script:AsyncPowerShell = $null

$script:SearchTimer.Add_Tick({
    # Check if the thread has finished
    if ($script:AsyncSearch -and $script:AsyncSearch.IsCompleted) {
        $script:SearchTimer.Stop()
        
        try {
            # Get Results from the Thread
            $results = $script:AsyncPowerShell.EndInvoke($script:AsyncSearch)
            $script:AsyncPowerShell.Dispose()
            
            foreach ($item in $results) {
                # Handle Log Messages vs Result Objects
                if ($item -is [string] -and $item.StartsWith("LOG:")) {
                    Write-GuiLog ($item.Substring(4))
                }
                elseif ($item.PSObject.Properties["Name"]) {
                    [void]$lstWinget.Items.Add($item)
                }
            }
        } catch {
            Write-GuiLog "Thread Error: $($_.Exception.Message)"
        }

        # UI Cleanup
        $lblWingetStatus.Visibility = "Hidden"
        $btnWingetFind.IsEnabled = $true
        $txtWingetSearch.IsEnabled = $true
        $lblWingetStatus.Text = "Ready"
        
        if ($lstWinget.Items.Count -eq 0) { 
            [void]$lstWinget.Items.Add([PSCustomObject]@{ Source=""; Name="No results found"; Id=""; Version=""; Available="" }) 
        }
        Write-GuiLog "Search Complete. Found $($lstWinget.Items.Count) results."
        
        $script:AsyncSearch = $null
        $script:AsyncPowerShell = $null
    }
})

# 2. THE BUTTON CLICK (Starts the Thread)
$btnWingetFind.Add_Click({
    if ([string]::IsNullOrWhiteSpace($txtWingetSearch.Text) -or $txtWingetSearch.Text -in @("Search packages...", "Search new packages...")) { return }

    # UI Prep
    $query = $txtWingetSearch.Text
    $lblWingetTitle.Text = "Search Results: $query"
    $lblWingetStatus.Text = "Searching..."; $lblWingetStatus.Visibility = "Visible"
    $btnWingetUpdateSel.Visibility = "Collapsed"; $btnWingetInstall.Visibility = "Visible"
    $lstWinget.Items.Clear()
    $btnWingetFind.IsEnabled = $false 
    
    Write-GuiLog " "
    Write-GuiLog "Starting Search: '$query'"

    # Get Settings
    $settings = Get-WmtSettings
    $enabled = if ($settings.EnabledProviders) { $settings.EnabledProviders } else { @("winget", "msstore", "pip", "npm", "chocolatey") }

    # 3. DEFINE THE WORKER THREAD SCRIPT
    # This contains the EXACT logic that worked for you before.
    $scriptBlock = {
        param($Query, $Enabled)
        
        [Console]::OutputEncoding = [System.Text.UTF8Encoding]::new($false)
        function Log($msg) { Write-Output "LOG:$msg" }

        # --- A. WINGET & MSSTORE ---
        if ("winget" -in $Enabled -or "msstore" -in $Enabled) {
            Log "Searching Winget & Store..."
            
            $sourceFlag = ""
            if ("winget" -in $Enabled -and "msstore" -notin $Enabled) { $sourceFlag = "--source winget" }
            elseif ("winget" -notin $Enabled -and "msstore" -in $Enabled) { $sourceFlag = "--source msstore" }
            
            $cleanQuery = $Query.Replace('"', '')
            
            $pInfo = New-Object System.Diagnostics.ProcessStartInfo
            $pInfo.FileName = "winget"
            $pInfo.Arguments = "search --query `"$cleanQuery`" $sourceFlag --accept-source-agreements"
            
            $pInfo.RedirectStandardOutput = $true
            $pInfo.RedirectStandardError = $true
            $pInfo.UseShellExecute = $false
            $pInfo.CreateNoWindow = $true
            $pInfo.StandardOutputEncoding = [System.Text.UTF8Encoding]::new($false)

            try {
                $p = [System.Diagnostics.Process]::Start($pInfo)
                $out = $p.StandardOutput.ReadToEnd()
                $p.WaitForExit()
                
                $lines = $out -split "`r`n"
                $idxId = -1
                foreach ($line in $lines) { if ($line -match "Name\s+Id\s+Version") { $idxId = $line.IndexOf("Id"); break } }

                foreach ($line in $lines) {
                    $line = $line.Trim()
                    if ([string]::IsNullOrWhiteSpace($line)) { continue }
                    if ($line -match "^-+$" -or $line -match "^Name\s+Id" -or $line -match "^-{3,}") { continue } 
                    if ($line -match "Windows Package Manager" -or $line -match "Copyright" -or $line -match "usage:" -or $line -match "No package found") { continue }
                    
                    if ($line -match "^Name" -and $line -match "Id" -and $line -match "Version") { continue }
                    if ($line -eq "-") { continue }

                    $n=$null; $i=$null; $v=$null; $s="winget"
                    
                    # Method A: Header Offset
                    if ($idxId -gt 0 -and $line.Length -gt $idxId) {
                        $n = $line.Substring(0, $idxId).Trim()
                        $rest = $line.Substring($idxId).Trim()
                        $parts = $rest -split "\s+"
                        if ($parts.Count -ge 3) { $i = $parts[0]; $v = $parts[1]; $s = $parts[2] }
                        elseif ($parts.Count -ge 2) { $i = $parts[0]; $v = $parts[1] }
                    }
                    
                    # Method B: Regex Fallback
                    if (-not $n -and $line -match '^(.+)\s+([^\s]+)\s+([^\s]+)\s+([^\s]+)') {
                        $n = $matches[1].Trim(); $i = $matches[2].Trim(); $v = $matches[3].Trim(); $s = $matches[4].Trim()
                    }
                    elseif (-not $n -and $line -match '^(.+)\s+([^\s]+)\s+([^\s]+)') {
                        $n = $matches[1].Trim(); $i = $matches[2].Trim(); $v = $matches[3].Trim()
                    }

                    if ($n -and $i -and $i.Length -gt 2) {
                        if ($n -eq "Name" -and $i -eq "Id") { continue }
                        if ($i -eq "Version" -or $v -eq "Source") { continue }
                        if ($i -notmatch "Tag:" -and $i -notmatch "Moniker:" -and $i -notmatch "input") {
                            if ($s -eq "msstore") { $s = "msstore" } else { $s = "winget" }
                            [PSCustomObject]@{ Source=$s; Name=$n; Id=$i; Version=$v; Available="-" }
                        }
                    }
                }
            } catch { Log "Winget Error: $_" }
        }

        # --- B. NPM ---
        if ("npm" -in $Enabled) {
            Log "Searching NPM..."
            try {
                $pInfo = New-Object System.Diagnostics.ProcessStartInfo("cmd", "/c npm search `"$Query`" --json")
                $pInfo.RedirectStandardOutput = $true; $pInfo.UseShellExecute = $false; $pInfo.CreateNoWindow = $true
                $p = [System.Diagnostics.Process]::Start($pInfo); $json = $p.StandardOutput.ReadToEnd(); $p.WaitForExit()
                if ($json.Contains("[")) {
                    $json = $json.Substring($json.IndexOf("[")); $pkgs = $json | ConvertFrom-Json
                    foreach ($pkg in $pkgs) { 
                        [PSCustomObject]@{ Source="npm"; Name=$pkg.name; Id=$pkg.name; Version=$pkg.version; Available="-" } 
                    }
                }
            } catch { Log "Npm skipped." }
        }

        # --- C. CHOCO ---
        if ("chocolatey" -in $Enabled) {
            Log "Searching Chocolatey..."
            try {
                $pInfo = New-Object System.Diagnostics.ProcessStartInfo("choco", "search `"$Query`" -r")
                $pInfo.RedirectStandardOutput = $true; $pInfo.UseShellExecute = $false; $pInfo.CreateNoWindow = $true
                $p = [System.Diagnostics.Process]::Start($pInfo); $out = $p.StandardOutput.ReadToEnd(); $p.WaitForExit()
                $lines = $out -split "`r`n"
                foreach ($line in $lines) {
                    if ([string]::IsNullOrWhiteSpace($line)) { continue }; $parts = $line -split "\|"
                    if ($parts.Count -ge 2) { 
                        [PSCustomObject]@{ Source="chocolatey"; Name=$parts[0]; Id=$parts[0]; Version=$parts[1]; Available="-" } 
                    }
                }
            } catch { Log "Choco skipped." }
        }
    }

    # 4. EXECUTE THREAD (The Magic Part)
    $script:AsyncPowerShell = [PowerShell]::Create().AddScript($scriptBlock).AddArgument($query).AddArgument($enabled)
    $script:AsyncSearch = $script:AsyncPowerShell.BeginInvoke()
    $script:SearchTimer.Start()
})

function Test-WingetRestartRiskItem {
    param($Item)
    if (-not $Item) { return $false }
    $id = [string]$Item.Id
    $name = [string]$Item.Name
    $eaIdPattern = '(?i)(ElectronicArts|EA\.Desktop|EADesktop|EAapp)'
    $eaNamePattern = '(?i)\b(EA app|Electronic Arts|EA Desktop)\b'
    return ($id -match $eaIdPattern -or $name -match $eaNamePattern)
}

function Show-WingetRestartRiskWarning {
    param(
        [object[]]$Items,
        [string]$Action = "Update"
    )
    if ($Action -ne "Update") { return $true }
    $risky = @($Items | Where-Object { Test-WingetRestartRiskItem $_ })
    if ($risky.Count -eq 0) { return $true }

    $pkgNames = @($risky | ForEach-Object {
        if ([string]::IsNullOrWhiteSpace([string]$_.Name)) { [string]$_.Id } else { [string]$_.Name }
    } | Select-Object -Unique)
    $joined = ($pkgNames -join ", ")
    if ([string]::IsNullOrWhiteSpace($joined)) { $joined = "the selected EA package" }

    $msg = "Warning: Updating $joined may trigger a Windows restart (installer behavior outside WMT control).`n`nDo you want to continue?"
    $res = [System.Windows.MessageBox]::Show($msg, "Restart Warning", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    return ($res -eq [System.Windows.MessageBoxResult]::Yes)
}

# 1. Update Selected (Removed CmdTemplate to allow smart logic)
$btnWingetUpdateSel.Add_Click({ 
    $selected = @($lstWinget.SelectedItems)
    if ($selected.Count -eq 0) { return }
    if (-not (Show-WingetRestartRiskWarning -Items $selected -Action "Update")) {
        Write-GuiLog "[Update] Cancelled by user after restart warning."
        return
    }
    & $Script:StartWingetAction -ListItems $selected -ActionName "Update"
})

# 2. Install Selected (Removed CmdTemplate so it includes --accept-agreements)
$btnWingetInstall.Add_Click({ 
    & $Script:StartWingetAction -ListItems $lstWinget.SelectedItems -ActionName "Install"
})

# 3. Uninstall Selected
$btnWingetUninstall.Add_Click({ 
    $selected = @($lstWinget.SelectedItems)
    if ($selected.Count -eq 0) { return }

    $msg = "Are you sure you want to uninstall $($selected.Count) application(s)?"
    if ([System.Windows.Forms.MessageBox]::Show($msg, "Confirm", "YesNo", "Warning") -eq "Yes") {
        & $Script:StartWingetAction -ListItems $selected -ActionName "Uninstall"
    }
})

# --- System Health ---
$btnSFC.Add_Click({
    Start-Process -FilePath "powershell.exe" -ArgumentList '-NoProfile -ExecutionPolicy Bypass -Command "sfc /scannow; Write-Host; Write-Host ''Execution Complete.'' -ForegroundColor Green; Write-Host ''Press Enter to close...'' -NoNewline -ForegroundColor Gray; Read-Host"' -Verb RunAs -WindowStyle Normal
})
$btnDISMCheck.Add_Click({
    Invoke-UiCommand {
        $output = dism /online /cleanup-image /checkhealth 2>&1
        $text = ($output | Out-String).Trim()
        if ($text) { Write-Output $text }

        $message = "DISM Check completed."
        $needsRepair = $false
        if ($text -match "No component store corruption detected") {
            $message = "DISM Check: no corruption detected."
        } elseif ($text -match "component store is repairable") {
            $message = "DISM Check: corruption detected (repairable)."
            $needsRepair = $true
        } elseif ($text -match "The operation completed successfully") {
            $message = "DISM Check: completed successfully."
        }
        [System.Windows.MessageBox]::Show($message, "DISM CheckHealth", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null

        if ($needsRepair) {
            $prompt = [System.Windows.MessageBox]::Show(
                "DISM found repairable corruption.`n`nRun DISM RestoreHealth now?",
                "DISM CheckHealth",
                [System.Windows.MessageBoxButton]::YesNo,
                [System.Windows.MessageBoxImage]::Question
            )
            if ($prompt -eq "Yes") {
                Write-Output "Launching DISM RestoreHealth..."
                Start-Process -FilePath "powershell.exe" -ArgumentList '-NoProfile -ExecutionPolicy Bypass -Command "dism /online /cleanup-image /restorehealth; Write-Host; Write-Host ''Execution Complete.'' -ForegroundColor Green; Write-Host ''Press Enter to close...'' -NoNewline -ForegroundColor Gray; Read-Host"' -Verb RunAs -WindowStyle Normal
            }
        }
    } "Running DISM CheckHealth..."
})
$btnDISMRestore.Add_Click({
    Start-Process -FilePath "powershell.exe" -ArgumentList '-NoProfile -ExecutionPolicy Bypass -Command "dism /online /cleanup-image /restorehealth; Write-Host; Write-Host ''Execution Complete.'' -ForegroundColor Green; Write-Host ''Press Enter to close...'' -NoNewline -ForegroundColor Gray; Read-Host"' -Verb RunAs -WindowStyle Normal
})
$btnCHKDSK.Add_Click({ Invoke-ChkdskAll })

# --- NETWORK ---
$btnNetInfo.Add_Click({
    Invoke-UiCommand {
        $out = ipconfig /all 2>&1
        $txt = ($out | Out-String)
        Write-Output $txt
        Show-TextDialog -Title "IP Configuration" -Text $txt
    } "Showing IP configuration..."
})
$btnFlushDNS.Add_Click({
    Invoke-UiCommand {
        $out = ipconfig /flushdns 2>&1
        $txt = ($out | Out-String).Trim()
        if ($txt) { Write-Output $txt }
        [System.Windows.MessageBox]::Show("DNS cache flushed.", "Flush DNS", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
    } "Flushing DNS cache..."
})
$btnResetWifi.Add_Click({
    Invoke-UiCommand {
        $wifi = Get-NetAdapter | Where-Object { $_.Status -eq 'Up' -and $_.InterfaceDescription -match "Wi-Fi|Wireless" }
        $eth  = Get-NetAdapter | Where-Object { $_.Status -eq 'Up' -and $_.InterfaceDescription -notmatch "Wi-Fi|Wireless" -and $_.InterfaceDescription -notmatch "Bluetooth" }

        if (-not $wifi) {
            $msg = "No active Wi-Fi adapters found."
            if ($eth) {
                $ethNames = $eth | Select-Object -ExpandProperty Name
                $msg += "`nYou appear to be on Ethernet: " + ($ethNames -join ", ")
            }
            [System.Windows.MessageBox]::Show($msg, "Restart Wi-Fi", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
            Write-Output $msg
            return
        }

        $names = $wifi | Select-Object -ExpandProperty Name
        foreach ($n in $names) {
            Restart-NetAdapter -Name $n -Confirm:$false -ErrorAction SilentlyContinue
            Write-Output "Restarted Wi-Fi adapter: $n"
        }

        [System.Windows.MessageBox]::Show("Restarted Wi-Fi adapter(s): " + ($names -join ", "), "Restart Wi-Fi", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
    } "Restarting Wi-Fi adapters..."
})

$btnNetRepair.Add_Click({
    $msg = "Full Network Repair will:" +
           "`n- Release/Renew IP" +
           "`n- Flush DNS cache" +
           "`n- Reset Winsock" +
           "`n- Reset IP stack" +
           "`n`nAdapters may briefly disconnect. Continue?"

    $res = [System.Windows.MessageBox]::Show($msg, "Full Network Repair", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    
    if ($res -eq "Yes") {
        Start-NetRepair
    }
})

$btnRouteTable.Add_Click({ Invoke-UiCommand { $path = Join-Path (Get-DataPath) "RouteTable.txt"; route print | Out-File -FilePath $path -Encoding UTF8; Write-Output "Saved to $path" } "Saving routing table..." })
$btnRouteView.Add_Click({
    Invoke-UiCommand {
        $out = route print 2>&1
        $txt = ($out | Out-String)
        Write-Output $txt
        Show-TextDialog -Title "Route Table" -Text $txt
    } "Routing table"
})

$btnDnsGoogle.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Set DNS to Google (8.8.8.8 / 8.8.4.4) on all active adapters?", "DNS Preset", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Question)
    if ($res -ne "Yes") { return }
    Set-DnsAddresses -Addresses @("8.8.8.8","8.8.4.4") -Label "Google DNS"
})
$btnDnsCloudflare.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Set DNS to Cloudflare (1.1.1.1 / 1.0.0.1) on all active adapters?", "DNS Preset", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Question)
    if ($res -ne "Yes") { return }
    Set-DnsAddresses -Addresses @("1.1.1.1","1.0.0.1") -Label "Cloudflare DNS"
})
$btnDnsQuad9.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Set DNS to Quad9 (9.9.9.9 / 149.112.112.112) on all active adapters?", "DNS Preset", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Question)
    if ($res -ne "Yes") { return }
    Set-DnsAddresses -Addresses @("9.9.9.9","149.112.112.112") -Label "Quad9 DNS"
})
$btnDnsAuto.Add_Click({
    Invoke-UiCommand {
        Get-ActiveAdapters | Select-Object -ExpandProperty Name | ForEach-Object { Set-DnsClientServerAddress -InterfaceAlias $_ -ResetServerAddresses }
        Write-Output "DNS reset to automatic (DHCP)."
        [System.Windows.MessageBox]::Show("DNS reset to automatic (DHCP).", "DNS Reset", [System.Windows.MessageBoxButton]::OK, [System.Windows.MessageBoxImage]::Information) | Out-Null
    } "Resetting DNS..."
})
$btnDnsCustom.Add_Click({
    $dnsInput = [Microsoft.VisualBasic.Interaction]::InputBox("Enter DNS addresses (comma separated)", "Custom DNS", "1.1.1.1,8.8.8.8")
    if (-not $dnsInput) { return }
    $addresses = $dnsInput.Split(",", [System.StringSplitOptions]::RemoveEmptyEntries) | ForEach-Object { $_.Trim() } | Where-Object { $_ }
    $valid = @()
    foreach ($addr in $addresses) { if (Test-Connection -ComputerName $addr -Count 1 -Quiet -ErrorAction SilentlyContinue) { $valid += $addr } else { Write-GuiLog "Unreachable DNS skipped: $addr" } }
    if (-not $valid) { [System.Windows.MessageBox]::Show("No reachable DNS addresses were provided.","Custom DNS",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Warning); return }
    Set-DnsAddresses -Addresses $valid -Label "Custom DNS"
})

$btnDohAuto.Add_Click({ Enable-AllDoh })
$btnDohDisable.Add_Click({ Disable-AllDoh })

$btnHostsUpdate.Add_Click({ Invoke-HostsUpdate })
$btnHostsEdit.Add_Click({ Show-HostsEditor })
$btnHostsBackup.Add_Click({ Invoke-UiCommand { $dest = Join-Path (Get-DataPath) ("hosts_bk_{0}.bak" -f (Get-Date -Format "yyyyMMdd_HHmmss")); Copy-Item "$env:windir\System32\drivers\etc\hosts" $dest; "Backup saved to $dest" } "Backing up hosts file..." })
$btnHostsRestore.Add_Click({
    $o=New-Object System.Windows.Forms.OpenFileDialog
    $o.Filter="*.bak;*.txt|*.bak;*.txt"
    if($o.ShowDialog()-eq"OK"){
        $restoreFile = $o.FileName
        $res = [System.Windows.MessageBox]::Show("Restore hosts file from:`n$restoreFile`n`nThis will overwrite the current hosts file. Continue?","Restore Hosts",[System.Windows.MessageBoxButton]::YesNo,[System.Windows.MessageBoxImage]::Warning)
        if ($res -ne "Yes") { return }
        Invoke-UiCommand{ Copy-Item $restoreFile "$env:windir\System32\drivers\etc\hosts" -Force } "Restored hosts file from $restoreFile"
        [System.Windows.MessageBox]::Show("Hosts file restored from:`n$restoreFile","Restore Hosts",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Information) | Out-Null
    }
})

# --- FIREWALL ---
# --- FIREWALL DOUBLE-CLICK MODIFY ---
$lstFw.Add_MouseDoubleClick({
    $rule = $lstFw.SelectedItem
    if ($null -eq $rule) { return }

    # 1. Open existing dialog with the selected rule
    $result = Show-RuleDialog "Edit Firewall Rule" $rule

    # 2. If user clicked 'Save' (result is not null)
    if ($result) {
        try {
            # 3. Apply changes using the Name (ID) from the original object
            Set-NetFirewallRule -Name $rule.Name `
                -Direction $result.Direction `
                -Action $result.Action `
                -Protocol $result.Protocol `
                -LocalPort $result.Port `
                -ErrorAction Stop

            # 4. Refresh the list to show changes
            $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent)))
        }
        catch {
            [System.Windows.MessageBox]::Show("Failed to update rule:`n$($_.Exception.Message)", "Error", "OK", "Error")
        }
    }
})
# --- FIREWALL CONTEXT MENU ---
$fwCtxMenu = New-Object System.Windows.Controls.ContextMenu

# 1. Option: Copy Rule Name
$mniCopyName = New-Object System.Windows.Controls.MenuItem
$mniCopyName.Header = "Copy Rule Name"
$mniCopyName.Add_Click({
    if ($lstFw.SelectedItem) {
        try {
            [System.Windows.Forms.Clipboard]::SetText($lstFw.SelectedItem.Name)
        } catch { 
            # Fallback if clipboard is busy
        }
    }
})

# 2. Option: Copy Port/Protocol
$mniCopyPort = New-Object System.Windows.Controls.MenuItem
$mniCopyPort.Header = "Copy Port/Protocol"
$mniCopyPort.Add_Click({
    if ($lstFw.SelectedItem) {
        $info = "$($lstFw.SelectedItem.Protocol) : $($lstFw.SelectedItem.LocalPort)"
        try {
            [System.Windows.Forms.Clipboard]::SetText($info)
        } catch {}
    }
})

# 3. Option: Copy Full Details
$mniCopyAll = New-Object System.Windows.Controls.MenuItem
$mniCopyAll.Header = "Copy Full Details"
$mniCopyAll.Add_Click({
    if ($lstFw.SelectedItem) {
        # Create a nice string representation of the rule
        $rule = $lstFw.SelectedItem
        $text = "Name: $($rule.Name)`nEnabled: $($rule.Enabled)`nAction: $($rule.Action)`nDirection: $($rule.Direction)`nProtocol: $($rule.Protocol)`nPort: $($rule.LocalPort)"
        try {
            [System.Windows.Forms.Clipboard]::SetText($text)
        } catch {}
    }
})

# Add items to the menu
[void]$fwCtxMenu.Items.Add($mniCopyName)
[void]$fwCtxMenu.Items.Add($mniCopyPort)
[void]$fwCtxMenu.Items.Add((New-Object System.Windows.Controls.Separator))
[void]$fwCtxMenu.Items.Add($mniCopyAll)

# Attach to the ListView
$lstFw.ContextMenu = $fwCtxMenu
$AllFw = @()
$btnFwRefresh.Add_Click({
    $lblFwStatus.Visibility="Visible"; $lstFw.Items.Clear(); [System.Windows.Forms.Application]::DoEvents()
    $AllFw = Get-NetFirewallRule | Select-Object Name, DisplayName, @{N='Enabled';E={$_.Enabled.ToString()}}, Direction, @{N='Action';E={$_.Action.ToString()}}, @{N='Protocol';E={($_.GetNetworkProtocols().Protocol)}}, @{N='LocalPort';E={($_.GetNetworkProtocols().LocalPort)}}
    $AllFw | ForEach-Object { [void]$lstFw.Items.Add($_) }
    $lblFwStatus.Visibility="Collapsed"
})

# --- FIREWALL LISTVIEW SORTING LOGIC ---
$script:FirewallSortChain = New-Object System.Collections.ArrayList
function Resolve-FirewallSortProperty {
    param([string]$Header)
    switch ($Header) {
        "Rule Name" { return "Name" }
        "Direction" { return "Direction" }
        "Action"    { return "Action" }
        "Status"    { return "Enabled" }
        "Protocol"  { return "Protocol" }
        "Port"      { return "LocalPort" }
        default     { return $Header }
    }
}

if ($lstFw) {
    $fwSortHandler = [System.Windows.RoutedEventHandler]{
        param($src, $e)
        if ($e.OriginalSource -isnot [System.Windows.Controls.GridViewColumnHeader]) { return }
        $header = Get-CleanHeader $e.OriginalSource.Column.Header
        if ([string]::IsNullOrWhiteSpace($header)) { return }

        $propName = Resolve-FirewallSortProperty $header
        if ([string]::IsNullOrWhiteSpace($propName)) { return }

        $isAscending = Set-SortChainPrimary -Chain $script:FirewallSortChain -PropertyName $propName
        Update-GridViewHeaders -ListView $lstFw -ActiveHeader $header -Ascending:$isAscending
        Set-ListViewSort -ListView $lstFw -Chain $script:FirewallSortChain
    }
    $lstFw.AddHandler([System.Windows.Controls.Primitives.ButtonBase]::ClickEvent, $fwSortHandler)
}

$txtFwSearch.Add_TextChanged({
    $q = $txtFwSearch.Text
    $lstFw.Items.Clear()
    if ($q -and $q -notin @("Search Rules...", "Search rules...")) {
        $AllFw | Where-Object { $_.DisplayName -match $q -or $_.LocalPort -match $q } | ForEach-Object { [void]$lstFw.Items.Add($_) }
    } else {
        $AllFw | ForEach-Object { [void]$lstFw.Items.Add($_) }
    }
})
$txtFwSearch.Add_GotFocus({
    $t = $txtFwSearch
    if ($t.Text -in @("Search Rules...", "Search rules...")) { $t.Text = "" }
})
$btnFwAdd.Add_Click({ $d=Show-RuleDialog "Add Rule"; if($d){ try{New-NetFirewallRule -DisplayName $d.Name -Direction $d.Direction -Action $d.Action -Protocol $d.Protocol -LocalPort $d.Port -ErrorAction Stop; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent)))}catch{[System.Windows.MessageBox]::Show("Err: $_")} } })
$btnFwEdit.Add_Click({ if($lstFw.SelectedItem){ $d=Show-RuleDialog "Edit" $lstFw.SelectedItem; if($d){ try{Set-NetFirewallRule -Name $lstFw.SelectedItem.Name -Direction $d.Direction -Action $d.Action -Protocol $d.Protocol -LocalPort $d.Port; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent)))}catch{[System.Windows.MessageBox]::Show("Err: $_")} } } })
$btnFwEnable.Add_Click({ if($lstFw.SelectedItem){ Set-NetFirewallRule -Name $lstFw.SelectedItem.Name -Enabled True; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) } })
$btnFwDisable.Add_Click({ if($lstFw.SelectedItem){ Set-NetFirewallRule -Name $lstFw.SelectedItem.Name -Enabled False; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) } })
$btnFwDelete.Add_Click({ if($lstFw.SelectedItem){ Remove-NetFirewallRule -Name $lstFw.SelectedItem.Name; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) } })
$btnFwExport.Add_Click({ Invoke-FirewallExport })
$btnFwImport.Add_Click({ Invoke-FirewallImport; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) })
$btnFwDefaults.Add_Click({ Invoke-FirewallDefaults; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) })
$btnFwPurge.Add_Click({ Invoke-FirewallPurge; $btnFwRefresh.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) })

# --- Drivers ---
$btnDrvReport.Add_Click({ Invoke-DriverReport })
$btnDrvBackup.Add_Click({ Invoke-ExportDrivers })
$btnDrvGhost.Add_Click({ Show-GhostDevicesDialog })
$btnDrvClean.Add_Click({ Show-DriverCleanupDialog })
$btnDrvRestore.Add_Click({ Invoke-RestoreDrivers })
$btnDrvDisableWU.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Disable automatic driver updates via Windows Update?", "Driver Updates", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    if ($res -ne "Yes") { return }
    Invoke-DriverUpdates -Enable:$false
})
$btnDrvEnableWU.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Enable automatic driver updates via Windows Update?", "Driver Updates", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Question)
    if ($res -ne "Yes") { return }
    Invoke-DriverUpdates -Enable:$true
})
$btnDrvDisableMeta.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Disable device metadata downloads (icons/info) from the internet?", "Device Metadata", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning)
    if ($res -ne "Yes") { return }
    Invoke-DeviceMetadata -Enable:$false
})
$btnDrvEnableMeta.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Enable device metadata downloads (icons/info) from the internet?", "Device Metadata", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Question)
    if ($res -ne "Yes") { return }
    Invoke-DeviceMetadata -Enable:$true
})

# --- Cleanup ---
$btnCleanDisk.Add_Click({ Start-Process cleanmgr })
$btnCleanTemp.Add_Click({ Invoke-TempCleanup })
$btnCleanShortcuts.Add_Click({ Invoke-ShortcutFix })
$btnCleanReg.Add_Click({
    $form = New-Object System.Windows.Forms.Form
    $form.Text="Registry Cleanup"; $form.Size="420,290"; $form.StartPosition="CenterScreen"; $form.BackColor=[System.Drawing.Color]::FromArgb(35,35,35); $form.ForeColor="White"
    $actions = [ordered]@{
        "List Safe Keys (Obsolete)"="List"
        "Delete Safe Keys (Obsolete)"="Delete"
        "Deep Clean (Invalid Paths)"="DeepClean"
        "Backup HKLM Hive"="BackupHKLM"
        "Restore Registry Backup"="Restore"
        "Run SFC/DISM Scan"="Scan"
    }
    $y=10
    foreach ($k in $actions.Keys) {
        $btn = New-Object System.Windows.Forms.Button
        $btn.Text=$k; $btn.Tag=$actions[$k]; $btn.Left=20; $btn.Top=$y; $btn.Width=360; $btn.Height=35; $btn.BackColor="DimGray"; $btn.ForeColor="White"
        $btn.Add_Click({ param($s,$e) $form.Tag = $s.Tag; $form.Close() })
        $form.Controls.Add($btn); $y += 40
    }
    $form.ShowDialog() | Out-Null
    if ($form.Tag) { Invoke-RegistryTask -Action $form.Tag }
})
$btnCleanXbox.Add_Click({
    if ([System.Windows.MessageBox]::Show("Delete stored Xbox credentials? This signs you out of Xbox services.", "Xbox Cleanup", [System.Windows.MessageBoxButton]::YesNo, [System.Windows.MessageBoxImage]::Warning) -eq "Yes") { Start-XboxClean }
})

# --- Utilities ---
$btnUpdateServices.Add_Click({
    $confirm = [System.Windows.MessageBox]::Show("Restart Windows Update related services (wuauserv/cryptsvc/bits/appidsvc)?","Restart Update Services",[System.Windows.MessageBoxButton]::YesNo,[System.Windows.MessageBoxImage]::Warning)
    if ($confirm -ne "Yes") { return }
    $script:UpdateSvcResult = $null
    Invoke-UpdateServiceReset
    if ($script:UpdateSvcResult -and $script:UpdateSvcResult -like "OK*") {
        [System.Windows.MessageBox]::Show("Update services restarted successfully.","Restart Update Services",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Information) | Out-Null
    } else {
        $msg = if ($script:UpdateSvcResult) { $script:UpdateSvcResult } else { "Unknown error. Check log output." }
        [System.Windows.MessageBox]::Show("Failed to restart update services.`n$msg","Restart Update Services",[System.Windows.MessageBoxButton]::OK,[System.Windows.MessageBoxImage]::Error) | Out-Null
    }
})
$btnDotNetEnable.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Set .NET roll-forward? This forces apps to use the latest installed .NET version (depending on selection).","Set .NET RollForward",[System.Windows.MessageBoxButton]::YesNo,[System.Windows.MessageBoxImage]::Warning)
    if ($res -ne "Yes") { return }
    $form = New-Object System.Windows.Forms.Form
    $form.Text="Set .NET RollForward"; $form.Size="320,210"; $form.StartPosition="CenterScreen"; $form.BackColor=[System.Drawing.Color]::FromArgb(35,35,35); $form.ForeColor="White"
    $opts = @("Runtime","SDK","Both")
    $y=15; $radios=@()
    foreach ($o in $opts) { $rb=New-Object System.Windows.Forms.RadioButton; $rb.Text=$o; $rb.Tag=$o; $rb.Left=20; $rb.Top=$y; $rb.ForeColor="White"; $rb.BackColor=$form.BackColor; $form.Controls.Add($rb); $radios+=$rb; $y+=30 }
    $radios[0].Checked=$true
    $ok=New-Object System.Windows.Forms.Button; $ok.Text="Apply"; $ok.DialogResult="OK"; $ok.Left=20; $ok.Top=120; $ok.Width=260; $ok.BackColor="SeaGreen"; $ok.ForeColor="White"; $form.Controls.Add($ok); $form.AcceptButton=$ok
    
    if ($form.ShowDialog() -eq "OK") { 
        $choice = ($radios | Where-Object { $_.Checked }).Tag; 
        if ($choice) { 
            Invoke-UiCommand { param($choice) Set-DotNetRollForward -Mode $choice } "Setting .NET roll-forward ($choice)..." -ArgumentList $choice
        } 
    }
})
$btnDotNetDisable.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Remove .NET roll-forward and revert to default .NET selection?","Reset .NET RollForward",[System.Windows.MessageBoxButton]::YesNo,[System.Windows.MessageBoxImage]::Warning)
    if ($res -ne "Yes") { return }
    Invoke-UiCommand { Set-DotNetRollForward -Mode "Disable" } "Removing .NET roll-forward..."
})
$btnTaskManager.Add_Click({ Show-TaskManager })
$btnInstallGpedit.Add_Click({ Start-GpeditInstall })
$btnUtilTrim.Add_Click({
    $res = [System.Windows.MessageBox]::Show("Run SSD Trim/ReTrim now? This will optimize all detected SSD volumes.","Trim SSD",[System.Windows.MessageBoxButton]::YesNo,[System.Windows.MessageBoxImage]::Question)
    if ($res -ne "Yes") { return }
    Start-Process -FilePath "powershell.exe" -ArgumentList '-NoProfile -NoExit -ExecutionPolicy Bypass -Command "Get-PhysicalDisk | Where-Object MediaType -eq ''SSD'' | ForEach-Object { Get-Disk | Where-Object { $_.FriendlyName -eq $_.FriendlyName } | Get-Partition | Get-Volume | Where-Object DriveLetter -ne $null | ForEach-Object { Optimize-Volume -DriveLetter $_.DriveLetter -ReTrim -Verbose } }"' -Verb RunAs -WindowStyle Normal
})
$btnUtilSysInfo.Add_Click({ Invoke-SystemReports })
$btnUtilMas.Add_Click({ Invoke-MASActivation })
$btnUpdateRepair.Add_Click({ Invoke-WindowsUpdateRepairFull })
$btnCtxBuilder.Add_Click({ Show-ContextMenuBuilder })

# --- Support ---
if ($btnSupportDiscord) { $btnSupportDiscord.Add_Click({ Start-Process "https://discord.gg/bCQqKHGxja" }) }
if ($btnSupportIssue) { $btnSupportIssue.Add_Click({ Start-Process "https://github.com/ios12checker/Windows-Maintenance-Tool/issues/new/choose" }) }
if ($btnDonateIos12) { $btnDonateIos12.Add_Click({ Start-Process "https://github.com/sponsors/ios12checker" }) }
if ($btnCreditLilBattiCLI) { $btnCreditLilBattiCLI.Add_Click({ Start-Process "https://github.com/ios12checker" }) }
if ($btnCreditChaythonFeatures) { $btnCreditChaythonFeatures.Add_Click({ Start-Process "https://github.com/Chaython" }) }
if ($btnCreditChaythonCLI) { $btnCreditChaythonCLI.Add_Click({ Start-Process "https://github.com/Chaython" }) }
if ($btnCreditChaythonGUI) { $btnCreditChaythonGUI.Add_Click({ Start-Process "https://github.com/Chaython" }) }
if ($btnCreditIos12checker) { $btnCreditIos12checker.Add_Click({ Start-Process "https://github.com/ios12checker" }) }
if ($btnToggleTheme) {
    $btnToggleTheme.Add_Click({
        $nextTheme = if ($script:CurrentTheme -eq "dark") { "light" } else { "dark" }
        Set-WmtTheme -Theme $nextTheme
        $settings = Get-WmtSettings
        $settings.Theme = $nextTheme
        Save-WmtSettings -Settings $settings
    })
}
if ($btnDonate) { $btnDonate.Add_Click({ Start-Process "https://github.com/sponsors/Chaython" }) }

if ($btnNavDownloads) { $btnNavDownloads.Add_Click({ Show-DownloadStats }) }

# ==========================================
# TWEAKS & OPTIMIZATION FUNCTIONS
# ==========================================

# --- PERFORMANCE TWEAKS WITH REVERT ---
$btnPerfServicesManual.Add_Click({
    Invoke-UiCommand {
        Write-GuiLog "Optimizing services to Manual..."
        $services = @('DiagTrack', 'dmwappushservice', 'MapsBroker', 'lfsvc', 'SharedAccess', 'WbioSrvc', 'WMPNetworkSvc', 'icssvc', 'WpnService', 'PcaSvc', 'SessionEnv', 'TermService', 'UmRdpService', 'RemoteRegistry', 'RemoteAccess', 'shpamsvc', 'TapiSrv', 'TabletInputService', 'lmhosts', 'SNMPTrap', 'WebClient', 'WerSvc', 'Wecsvc', 'SDRSVC', 'fdPHost', 'FDResPub', 'HomeGroupListener', 'HomeGroupProvider', 'upnphost', 'SSDPSRV', 'swprv', 'smphost', 'SysMain', 'TrkWks', 'WMPNetworkSvc', 'WMPNetworkSvc', 'iphlpsvc', 'MSiSCSI', 'WSearch', 'WinRM', 'XblAuthManager', 'XblGameSave', 'XboxNetApiSvc')
        foreach ($svc in $services) {
            try {
                $service = Get-Service -Name $svc -ErrorAction SilentlyContinue
                if ($service -and $service.StartType -eq 'Automatic') {
                    Set-Service -Name $svc -StartupType Manual -ErrorAction SilentlyContinue
                    Write-GuiLog "Set $svc to Manual"
                }
            } catch {}
        }
        Write-GuiLog "Services optimization complete!"
    } "Optimizing services..."
})

$btnPerfServicesRevert.Add_Click({
    Invoke-UiCommand {
        Write-GuiLog "Reverting services to default..."
        $services = @('DiagTrack', 'dmwappushservice', 'MapsBroker', 'WpnService', 'PcaSvc', 'WerSvc', 'SysMain', 'WSearch', 'XblAuthManager', 'XblGameSave', 'XboxNetApiSvc', 'iphlpsvc')
        foreach ($svc in $services) {
            try {
                Set-Service -Name $svc -StartupType Automatic -ErrorAction SilentlyContinue
                Write-GuiLog "Set $svc to Automatic"
            } catch {}
        }
        Write-GuiLog "Services restored to default!"
    } "Reverting services..."
})

$btnPerfDisableHibernate.Add_Click({
    Invoke-UiCommand {
        powercfg /hibernate off
        Write-GuiLog "Hibernation disabled. Disk space freed."
    } "Disabling hibernation..."
})

$btnPerfEnableHibernate.Add_Click({
    Invoke-UiCommand {
        powercfg /hibernate on
        Write-GuiLog "Hibernation enabled."
    } "Enabling hibernation..."
})

$btnPerfDisableSuperfetch.Add_Click({
    Invoke-UiCommand {
        Stop-Service -Name SysMain -Force -ErrorAction SilentlyContinue
        Set-Service -Name SysMain -StartupType Disabled
        Write-GuiLog "Superfetch/SysMain disabled."
    } "Disabling Superfetch..."
})

$btnPerfEnableSuperfetch.Add_Click({
    Invoke-UiCommand {
        Set-Service -Name SysMain -StartupType Automatic
        Start-Service -Name SysMain -ErrorAction SilentlyContinue
        Write-GuiLog "Superfetch/SysMain enabled."
    } "Enabling Superfetch..."
})

$btnPerfDisableMemCompress.Add_Click({
    Invoke-UiCommand {
        Disable-MMAgent -MemoryCompression -ErrorAction SilentlyContinue
        Write-GuiLog "Memory compression disabled."
    } "Disabling memory compression..."
})

$btnPerfEnableMemCompress.Add_Click({
    Invoke-UiCommand {
        Enable-MMAgent -MemoryCompression -ErrorAction SilentlyContinue
        Write-GuiLog "Memory compression enabled."
    } "Enabling memory compression..."
})

$btnPerfUltimatePower.Add_Click({
    Invoke-UiCommand {
        powercfg -duplicatescheme e9a42b02-d5df-448d-aa00-03f14749eb61
        powercfg /setactive e9a42b02-d5df-448d-aa00-03f14749eb61
        Write-GuiLog "Ultimate Performance power plan enabled."
    } "Enabling Ultimate Performance..."
})

# --- APPX BLOATWARE REMOVAL ---
$script:AppxList = @(
    [PSCustomObject]@{Name="Xbox App"; Package="Microsoft.XboxApp"},
    [PSCustomObject]@{Name="Xbox Gaming Overlay"; Package="Microsoft.XboxGamingOverlay"},
    [PSCustomObject]@{Name="Xbox Game Bar"; Package="Microsoft.XboxGameOverlay"},
    [PSCustomObject]@{Name="Xbox Live"; Package="Microsoft.XboxSpeechToTextOverlay"},
    [PSCustomObject]@{Name="Xbox Identity Provider"; Package="Microsoft.XboxIdentityProvider"},
    [PSCustomObject]@{Name="Microsoft Solitaire"; Package="Microsoft.MicrosoftSolitaireCollection"},
    [PSCustomObject]@{Name="Microsoft Office Hub"; Package="Microsoft.MicrosoftOfficeHub"},
    [PSCustomObject]@{Name="OneNote"; Package="Microsoft.Office.OneNote"},
    [PSCustomObject]@{Name="Mail & Calendar"; Package="microsoft.windowscommunicationsapps"},
    [PSCustomObject]@{Name="People"; Package="Microsoft.People"},
    [PSCustomObject]@{Name="Skype"; Package="Microsoft.SkypeApp"},
    [PSCustomObject]@{Name="Maps"; Package="Microsoft.WindowsMaps"},
    [PSCustomObject]@{Name="Weather"; Package="Microsoft.BingWeather"},
    [PSCustomObject]@{Name="News"; Package="Microsoft.BingNews"},
    [PSCustomObject]@{Name="Sports"; Package="Microsoft.BingSports"},
    [PSCustomObject]@{Name="Finance"; Package="Microsoft.BingFinance"},
    [PSCustomObject]@{Name="Movies & TV"; Package="Microsoft.ZuneVideo"},
    [PSCustomObject]@{Name="Groove Music"; Package="Microsoft.ZuneMusic"},
    [PSCustomObject]@{Name="Get Help"; Package="Microsoft.GetHelp"},
    [PSCustomObject]@{Name="Get Started"; Package="Microsoft.Getstarted"},
    [PSCustomObject]@{Name="Feedback Hub"; Package="Microsoft.WindowsFeedbackHub"},
    [PSCustomObject]@{Name="Mixed Reality Portal"; Package="Microsoft.MixedReality.Portal"},
    [PSCustomObject]@{Name="3D Viewer"; Package="Microsoft.Microsoft3DViewer"},
    [PSCustomObject]@{Name="Paint 3D"; Package="Microsoft.MSPaint"},
    [PSCustomObject]@{Name="Phone Link"; Package="Microsoft.YourPhone"},
    [PSCustomObject]@{Name="Quick Assist"; Package="MicrosoftCorporationII.QuickAssist"},
    [PSCustomObject]@{Name="Family Safety"; Package="MicrosoftCorporationII.MicrosoftFamily"}
)

$btnAppxLoad.Add_Click({
    $lstAppxPackages.Items.Clear()
    foreach ($app in $script:AppxList) {
        $installed = Get-AppxPackage -Name $app.Package -ErrorAction SilentlyContinue
        if ($installed) {
            [void]$lstAppxPackages.Items.Add($app)
        }
    }
    Write-GuiLog "Loaded $($lstAppxPackages.Items.Count) removable apps."
})

$btnAppxRemoveSel.Add_Click({
    $selected = $lstAppxPackages.SelectedItems
    if ($selected.Count -eq 0) { return }
    Invoke-UiCommand {
        param($apps)
        foreach ($app in $apps) {
            try {
                Remove-AppxPackage -Package (Get-AppxPackage -Name $app.Package).PackageFullName -ErrorAction SilentlyContinue
                Write-GuiLog "Removed: $($app.Name)"
            } catch {
                Write-GuiLog "Failed to remove: $($app.Name)"
            }
        }
    } "Removing selected apps..." -ArgumentList $selected
})

$btnAppxRemoveAll.Add_Click({
    if ([System.Windows.Forms.MessageBox]::Show("Remove ALL listed apps? This cannot be undone easily.", "Confirm", "YesNo", "Warning") -eq "Yes") {
        $btnAppxRemoveSel.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent)))
    }
})

# --- WINDOWS FEATURES TOGGLE ---
$btnFeatHyperV.Add_Click({ Switch-WindowsFeature "Microsoft-Hyper-V-All" "Hyper-V" })
$btnFeatWSL.Add_Click({ Switch-WindowsFeature "Microsoft-Windows-Subsystem-Linux" "WSL" })
$btnFeatSandbox.Add_Click({ Switch-WindowsFeature "Containers-DisposableClientVM" "Windows Sandbox" })
$btnFeatDotNet35.Add_Click({ Switch-WindowsFeature "NetFx3" ".NET Framework 3.5" })
$btnFeatNFS.Add_Click({ Switch-WindowsFeature "ServicesForNFS-ClientOnly" "NFS Client" })
$btnFeatTelnet.Add_Click({ Switch-WindowsFeature "TelnetClient" "Telnet Client" })
$btnFeatIIS.Add_Click({ Switch-WindowsFeature "IIS-WebServerRole" "IIS Web Server" })
$btnFeatLegacy.Add_Click({ Switch-WindowsFeature "WindowsMediaPlayer" "Legacy Media" })

function Switch-WindowsFeature($FeatureName, $DisplayName) {
    Invoke-UiCommand {
        param($fn, $dn)
        if (-not (Get-Command Get-WindowsOptionalFeature -ErrorAction SilentlyContinue)) {
            Write-GuiLog "PowerShell feature cmdlets unavailable; trying DISM fallback..."
            $featureInfo = (dism /Online /Get-FeatureInfo /FeatureName:$fn 2>&1) -join "`n"
            if ($featureInfo -match "State\\s*:\\s*Enabled") {
                dism /Online /Disable-Feature /FeatureName:$fn /NoRestart | Out-Null
                if ($LASTEXITCODE -eq 0) { Write-GuiLog "Disabled: $dn (DISM)" }
                else { throw "DISM failed to disable $dn (code $LASTEXITCODE)." }
            } else {
                dism /Online /Enable-Feature /FeatureName:$fn /All /NoRestart | Out-Null
                if ($LASTEXITCODE -eq 0) { Write-GuiLog "Enabled: $dn (DISM)" }
                else { throw "DISM failed to enable $dn (code $LASTEXITCODE). Windows source files may be required." }
            }
            return
        }

        $feature = Get-WindowsOptionalFeature -Online -FeatureName $fn -ErrorAction SilentlyContinue

        # Some systems report NetFx3 as unknown to PowerShell even when DISM can toggle it.
        if (-not $feature) {
            if ($fn -ne "NetFx3") { throw "Feature name $fn is unknown." }

            Write-GuiLog "$dn not detected via PowerShell; trying DISM fallback..."
            $featureInfo = (dism /Online /Get-FeatureInfo /FeatureName:$fn 2>&1) -join "`n"
            if ($featureInfo -match "State\\s*:\\s*Enabled") {
                dism /Online /Disable-Feature /FeatureName:$fn /NoRestart | Out-Null
                if ($LASTEXITCODE -eq 0) { Write-GuiLog "Disabled: $dn (DISM)" }
                else { throw "DISM failed to disable $dn (code $LASTEXITCODE)." }
            } else {
                dism /Online /Enable-Feature /FeatureName:$fn /All /NoRestart | Out-Null
                if ($LASTEXITCODE -eq 0) { Write-GuiLog "Enabled: $dn (DISM)" }
                else { throw "DISM failed to enable $dn (code $LASTEXITCODE). Windows source files may be required." }
            }
            return
        }

        if ($feature.State -eq "Enabled") {
            Disable-WindowsOptionalFeature -Online -FeatureName $fn -NoRestart -ErrorAction Stop | Out-Null
            Write-GuiLog "Disabled: $dn"
        } else {
            Enable-WindowsOptionalFeature -Online -FeatureName $fn -All -NoRestart -ErrorAction Stop | Out-Null
            Write-GuiLog "Enabled: $dn"
        }
    } "Toggling $DisplayName..." -ArgumentList $FeatureName, $DisplayName
}

# --- SERVICES MANAGEMENT ---
$btnSvcOptimize.Add_Click({ $btnPerfServicesManual.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) })
$btnSvcRestore.Add_Click({ $btnPerfServicesRevert.RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) })

$btnSvcView.Add_Click({
    Get-Service | Select-Object Name, DisplayName, StartType, Status | Out-GridView -Title "Windows Services"
})

# --- SCHEDULED TASKS WITH REVERT ---
$script:TelemetryTasks = @(
    "\Microsoft\Windows\Application Experience\Microsoft Compatibility Appraiser",
    "\Microsoft\Windows\Application Experience\ProgramDataUpdater",
    "\Microsoft\Windows\Autochk\Proxy",
    "\Microsoft\Windows\Customer Experience Improvement Program\Consolidator",
    "\Microsoft\Windows\Customer Experience Improvement Program\UsbCeip",
    "\Microsoft\Windows\DiskDiagnostic\Microsoft-Windows-DiskDiagnosticDataCollector",
    "\Microsoft\Windows\Feedback\Siuf\DmClient",
    "\Microsoft\Windows\Feedback\Siuf\DmClientOnScenarioDownload",
    "\Microsoft\Windows\Windows Error Reporting\QueueReporting"
)

$btnTasksDisableTelemetry.Add_Click({
    Invoke-UiCommand {
        param($tasks)
        foreach ($task in $tasks) {
            try {
                Disable-ScheduledTask -TaskName $task -ErrorAction SilentlyContinue | Out-Null
                Write-GuiLog "Disabled: $task"
            } catch {}
        }
        Write-GuiLog "Telemetry tasks disabled!"
    } "Disabling telemetry tasks..." -ArgumentList $script:TelemetryTasks
})

$btnTasksRestore.Add_Click({
    Invoke-UiCommand {
        param($tasks)
        foreach ($task in $tasks) {
            try {
                Enable-ScheduledTask -TaskName $task -ErrorAction SilentlyContinue | Out-Null
                Write-GuiLog "Enabled: $task"
            } catch {}
        }
        Write-GuiLog "Telemetry tasks restored!"
    } "Restoring telemetry tasks..." -ArgumentList $script:TelemetryTasks
})

$btnTasksView.Add_Click({
    Get-ScheduledTask | Where-Object { $_.TaskPath -eq "\Microsoft\Windows\Application Experience\" -or $_.TaskPath -eq "\Microsoft\Windows\Customer Experience Improvement Program\" } | 
        Select-Object TaskName, TaskPath, State | Out-GridView -Title "Telemetry Tasks"
})

# --- WINDOWS UPDATE PRESETS ---
$btnWUDefault.Add_Click({
    Invoke-UiCommand {
        Remove-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate\AU" -Name "NoAutoUpdate" -ErrorAction SilentlyContinue
        Remove-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -Name "DeferFeatureUpdates" -ErrorAction SilentlyContinue
        Remove-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -Name "DeferQualityUpdates" -ErrorAction SilentlyContinue
        Set-Service -Name wuauserv -StartupType Automatic -ErrorAction SilentlyContinue
        Start-Service -Name wuauserv -ErrorAction SilentlyContinue
        Write-GuiLog "Windows Update set to Default."
    } "Applying default Windows Update settings..."
})

$btnWUSecurity.Add_Click({
    Invoke-UiCommand {
        New-Item -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -Force | Out-Null
        Set-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -Name "DeferFeatureUpdates" -Value 1
        Set-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate" -Name "DeferFeatureUpdatesPeriodInDays" -Value 365
        Set-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate\AU" -Name "NoAutoUpdate" -Value 0 -ErrorAction SilentlyContinue
        Set-Service -Name wuauserv -StartupType Automatic -ErrorAction SilentlyContinue
        Write-GuiLog "Windows Update set to Security Only (deferring features)."
    } "Applying security-only update settings..."
})

$btnWUDisable.Add_Click({
    if ([System.Windows.Forms.MessageBox]::Show("Disable ALL Windows Updates? This is not recommended for security.", "Warning", "YesNo", "Warning") -eq "Yes") {
        Invoke-UiCommand {
            Set-Service -Name wuauserv -StartupType Disabled -ErrorAction SilentlyContinue
            Stop-Service -Name wuauserv -Force -ErrorAction SilentlyContinue
            New-Item -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate\AU" -Force | Out-Null
            Set-ItemProperty -Path "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate\AU" -Name "NoAutoUpdate" -Value 1
            Write-GuiLog "Windows Update DISABLED."
        } "Disabling Windows Update..."
    }
})

# --- SOFTWARE CATALOG ---
$script:SoftwareCatalog = @(
    [PSCustomObject]@{Category="Browsers"; Name="Google Chrome"; Description="Fast, secure web browser"; Source="winget"; Id="Google.Chrome"},
    [PSCustomObject]@{Category="Browsers"; Name="Mozilla Firefox"; Description="Privacy-focused browser"; Source="winget"; Id="Mozilla.Firefox"},
    [PSCustomObject]@{Category="Browsers"; Name="Microsoft Edge"; Description="Chromium-based browser"; Source="winget"; Id="Microsoft.Edge"},
    [PSCustomObject]@{Category="Browsers"; Name="Brave"; Description="Privacy-focused Chromium browser"; Source="winget"; Id="Brave.Brave"},
    [PSCustomObject]@{Category="Development"; Name="Visual Studio Code"; Description="Popular code editor"; Source="winget"; Id="Microsoft.VisualStudioCode"},
    [PSCustomObject]@{Category="Development"; Name="Git"; Description="Version control system"; Source="winget"; Id="Git.Git"},
    [PSCustomObject]@{Category="Development"; Name="Node.js"; Description="JavaScript runtime"; Source="winget"; Id="OpenJS.NodeJS"},
    [PSCustomObject]@{Category="Development"; Name="Python"; Description="Programming language"; Source="winget"; Id="Python.Python.3"},
    [PSCustomObject]@{Category="Development"; Name="Notepad++"; Description="Advanced text editor"; Source="winget"; Id="Notepad++.Notepad++"},
    [PSCustomObject]@{Category="Utilities"; Name="7-Zip"; Description="File archiver"; Source="winget"; Id="7zip.7zip"},
    [PSCustomObject]@{Category="Utilities"; Name="WinRAR"; Description="Archive manager"; Source="winget"; Id="RARLab.WinRAR"},
    [PSCustomObject]@{Category="Utilities"; Name="Everything"; Description="Fast file search"; Source="winget"; Id="voidtools.Everything"},
    [PSCustomObject]@{Category="Utilities"; Name="PowerToys"; Description="Microsoft productivity tools"; Source="winget"; Id="Microsoft.PowerToys"},
    [PSCustomObject]@{Category="Utilities"; Name="HWiNFO"; Description="Hardware monitoring"; Source="winget"; Id="REALiX.HWiNFO"},
    [PSCustomObject]@{Category="Multimedia"; Name="VLC Media Player"; Description="Universal media player"; Source="winget"; Id="VideoLAN.VLC"},
    [PSCustomObject]@{Category="Multimedia"; Name="Spotify"; Description="Music streaming"; Source="winget"; Id="Spotify.Spotify"},
    [PSCustomObject]@{Category="Multimedia"; Name="OBS Studio"; Description="Streaming/recording software"; Source="winget"; Id="OBSProject.OBSStudio"},
    [PSCustomObject]@{Category="Multimedia"; Name="GIMP"; Description="Image editor"; Source="winget"; Id="GIMP.GIMP"},
    [PSCustomObject]@{Category="Gaming"; Name="Steam"; Description="Game platform"; Source="winget"; Id="Valve.Steam"},
    [PSCustomObject]@{Category="Gaming"; Name="Discord"; Description="Chat for gamers"; Source="winget"; Id="Discord.Discord"},
    [PSCustomObject]@{Category="Gaming"; Name="Epic Games Launcher"; Description="Epic game store"; Source="winget"; Id="EpicGames.EpicGamesLauncher"},
    [PSCustomObject]@{Category="Security"; Name="Malwarebytes"; Description="Anti-malware"; Source="winget"; Id="Malwarebytes.Malwarebytes"},
    [PSCustomObject]@{Category="Security"; Name="Bitwarden"; Description="Password manager"; Source="winget"; Id="Bitwarden.Bitwarden"}
)

if ($btnShowCatalog -and $btnBackToUpdates -and $btnCatalogSearch -and $btnCatalogInstall -and $btnCatalogSelectAll -and $btnCatalogClear -and $btnCatAll -and $btnCatBrowsers -and $btnCatDev -and $btnCatUtils -and $btnCatMedia -and $btnCatGames -and $btnCatSecurity -and $pnlCatalog -and $lstCatalog -and $txtCatalogSearch) {
    $btnShowCatalog.Add_Click({
        $pnlUpdates.Visibility = "Collapsed"
        $pnlCatalog.Visibility = "Visible"
        $lstCatalog.Items.Clear()
        foreach ($item in $script:SoftwareCatalog) {
            [void]$lstCatalog.Items.Add($item)
        }
    })

    $btnBackToUpdates.Add_Click({
        $pnlCatalog.Visibility = "Collapsed"
        $pnlUpdates.Visibility = "Visible"
    })

    $btnCatalogSearch.Add_Click({
        $query = $txtCatalogSearch.Text.ToLower()
        $lstCatalog.Items.Clear()
        $filtered = $script:SoftwareCatalog | Where-Object { $_.Name -like "*$query*" -or $_.Description -like "*$query*" }
        foreach ($item in $filtered) { [void]$lstCatalog.Items.Add($item) }
    })

    $btnCatAll.Add_Click({
        $lstCatalog.Items.Clear()
        foreach ($item in $script:SoftwareCatalog) { [void]$lstCatalog.Items.Add($item) }
    })

    $btnCatBrowsers.Add_Click({ Get-CatalogByCategory "Browsers" })
    $btnCatDev.Add_Click({ Get-CatalogByCategory "Development" })
    $btnCatUtils.Add_Click({ Get-CatalogByCategory "Utilities" })
    $btnCatMedia.Add_Click({ Get-CatalogByCategory "Multimedia" })
    $btnCatGames.Add_Click({ Get-CatalogByCategory "Gaming" })
    $btnCatSecurity.Add_Click({ Get-CatalogByCategory "Security" })
}

function Get-CatalogByCategory($Category) {
    $lstCatalog.Items.Clear()
    $filtered = $script:SoftwareCatalog | Where-Object { $_.Category -eq $Category }
    foreach ($item in $filtered) { [void]$lstCatalog.Items.Add($item) }
}

if ($btnCatalogInstall -and $btnCatalogSelectAll -and $btnCatalogClear -and $lstCatalog) {
    $btnCatalogInstall.Add_Click({
        $selected = $lstCatalog.SelectedItems
        if ($selected.Count -eq 0) { return }
        Invoke-UiCommand {
            param($items)
            foreach ($item in $items) {
                Write-GuiLog "Installing: $($item.Name)..."
                $proc = Start-Process -FilePath "winget" -ArgumentList "install --id `"$($item.Id)`" --accept-source-agreements --accept-package-agreements --silent" -Wait -PassThru
                if ($proc.ExitCode -eq 0) {
                    Write-GuiLog "Installed: $($item.Name)"
                } else {
                    Write-GuiLog "Failed: $($item.Name)"
                }
            }
        } "Installing software..." -ArgumentList $selected
    })

    $btnCatalogSelectAll.Add_Click({ $lstCatalog.SelectAll() })
    $btnCatalogClear.Add_Click({ $lstCatalog.SelectedItems.Clear() })
}

# --- LAUNCH ---
$window.Add_Loaded({ 
    $settings = Get-WmtSettings

    # Restore persisted window geometry/state when valid
    if ($settings.WindowBounds) {
        $wb = $settings.WindowBounds
        if ($wb.Width -ge $window.MinWidth -and $wb.Height -ge $window.MinHeight) {
            $window.Width = [double]$wb.Width
            $window.Height = [double]$wb.Height
        }
        if (($wb.Left -ne 0 -or $wb.Top -ne 0) -and $settings.WindowState -ne "Maximized") {
            $window.Left = [double]$wb.Left
            $window.Top = [double]$wb.Top
        }
    }

    Set-WmtTheme -Theme $settings.Theme
    if ($settings.WindowState -eq "Maximized") {
        $window.WindowState = [System.Windows.WindowState]::Maximized
    }

    # 1. Click the Updates tab by default
    (Get-Ctrl "btnTabUpdates").RaiseEvent((New-Object System.Windows.RoutedEventArgs([System.Windows.Controls.Button]::ClickEvent))) 
    
    # 2. Trigger the background update check
    Start-UpdateCheckBackground
})

$window.Add_Closing({
    try {
        $settings = Get-WmtSettings
        $settings.Theme = $script:CurrentTheme
        $settings.WindowState = [string]$window.WindowState

        $rb = if ($window.WindowState -eq [System.Windows.WindowState]::Normal) { $null } else { $window.RestoreBounds }
        if ($rb) {
            $settings.WindowBounds = @{
                Top    = [double]$rb.Top
                Left   = [double]$rb.Left
                Width  = [double]$rb.Width
                Height = [double]$rb.Height
            }
        } else {
            $settings.WindowBounds = @{
                Top    = [double]$window.Top
                Left   = [double]$window.Left
                Width  = [double]$window.Width
                Height = [double]$window.Height
            }
        }
        Save-WmtSettings -Settings $settings
    } catch {}
})

# 3. Show the Window
$window.ShowDialog() | Out-Null

